import * as ec2 from '@aws-cdk/aws-ec2';
import * as lambda from '@aws-cdk/aws-lambda';
import { Duration, Resource } from '@aws-cdk/core';
import { Construct } from 'constructs';
import { ISecret } from './secret';
import { CfnRotationSchedule } from './secretsmanager.generated';
/**
 * Options to add a rotation schedule to a secret.
 *
 * @stability stable
 */
export interface RotationScheduleOptions {
    /**
     * A Lambda function that can rotate the secret.
     *
     * @default - either `rotationLambda` or `hostedRotation` must be specified
     * @stability stable
     */
    readonly rotationLambda?: lambda.IFunction;
    /**
     * Hosted rotation.
     *
     * @default - either `rotationLambda` or `hostedRotation` must be specified
     * @stability stable
     */
    readonly hostedRotation?: HostedRotation;
    /**
     * Specifies the number of days after the previous rotation before Secrets Manager triggers the next automatic rotation.
     *
     * @default Duration.days(30)
     * @stability stable
     */
    readonly automaticallyAfter?: Duration;
}
/**
 * Construction properties for a RotationSchedule.
 *
 * @stability stable
 */
export interface RotationScheduleProps extends RotationScheduleOptions {
    /**
     * The secret to rotate.
     *
     * If hosted rotation is used, this must be a JSON string with the following format:
     *
     * ```
     * {
     *    "engine": <required: database engine>,
     *    "host": <required: instance host name>,
     *    "username": <required: username>,
     *    "password": <required: password>,
     *    "dbname": <optional: database name>,
     *    "port": <optional: if not specified, default port will be used>,
     *    "masterarn": <required for multi user rotation: the arn of the master secret which will be used to create users/change passwords>
     * }
     * ```
     *
     * This is typically the case for a secret referenced from an `AWS::SecretsManager::SecretTargetAttachment`
     * or an `ISecret` returned by the `attach()` method of `Secret`.
     *
     * @stability stable
     */
    readonly secret: ISecret;
}
/**
 * A rotation schedule.
 *
 * @stability stable
 */
export declare class RotationSchedule extends Resource {
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props: RotationScheduleProps);
}
/**
 * Single user hosted rotation options.
 *
 * @stability stable
 */
export interface SingleUserHostedRotationOptions {
    /**
     * A name for the Lambda created to rotate the secret.
     *
     * @default - a CloudFormation generated name
     * @stability stable
     */
    readonly functionName?: string;
    /**
     * A list of security groups for the Lambda created to rotate the secret.
     *
     * @default - a new security group is created
     * @stability stable
     */
    readonly securityGroups?: ec2.ISecurityGroup[];
    /**
     * The VPC where the Lambda rotation function will run.
     *
     * @default - the Lambda is not deployed in a VPC
     * @stability stable
     */
    readonly vpc?: ec2.IVpc;
    /**
     * The type of subnets in the VPC where the Lambda rotation function will run.
     *
     * @default - the Vpc default strategy if not specified.
     * @stability stable
     */
    readonly vpcSubnets?: ec2.SubnetSelection;
}
/**
 * Multi user hosted rotation options.
 *
 * @stability stable
 */
export interface MultiUserHostedRotationOptions extends SingleUserHostedRotationOptions {
    /**
     * The master secret for a multi user rotation scheme.
     *
     * @stability stable
     */
    readonly masterSecret: ISecret;
}
/**
 * A hosted rotation.
 *
 * @stability stable
 */
export declare class HostedRotation implements ec2.IConnectable {
    private readonly type;
    private readonly props;
    private readonly masterSecret?;
    /**
     * MySQL Single User.
     *
     * @stability stable
     */
    static mysqlSingleUser(options?: SingleUserHostedRotationOptions): HostedRotation;
    /**
     * MySQL Multi User.
     *
     * @stability stable
     */
    static mysqlMultiUser(options: MultiUserHostedRotationOptions): HostedRotation;
    /**
     * PostgreSQL Single User.
     *
     * @stability stable
     */
    static postgreSqlSingleUser(options?: SingleUserHostedRotationOptions): HostedRotation;
    /**
     * PostgreSQL Multi User.
     *
     * @stability stable
     */
    static postgreSqlMultiUser(options: MultiUserHostedRotationOptions): HostedRotation;
    /**
     * Oracle Single User.
     *
     * @stability stable
     */
    static oracleSingleUser(options?: SingleUserHostedRotationOptions): HostedRotation;
    /**
     * Oracle Multi User.
     *
     * @stability stable
     */
    static oracleMultiUser(options: MultiUserHostedRotationOptions): HostedRotation;
    /**
     * MariaDB Single User.
     *
     * @stability stable
     */
    static mariaDbSingleUser(options?: SingleUserHostedRotationOptions): HostedRotation;
    /**
     * MariaDB Multi User.
     *
     * @stability stable
     */
    static mariaDbMultiUser(options: MultiUserHostedRotationOptions): HostedRotation;
    /**
     * SQL Server Single User.
     *
     * @stability stable
     */
    static sqlServerSingleUser(options?: SingleUserHostedRotationOptions): HostedRotation;
    /**
     * SQL Server Multi User.
     *
     * @stability stable
     */
    static sqlServerMultiUser(options: MultiUserHostedRotationOptions): HostedRotation;
    /**
     * Redshift Single User.
     *
     * @stability stable
     */
    static redshiftSingleUser(options?: SingleUserHostedRotationOptions): HostedRotation;
    /**
     * Redshift Multi User.
     *
     * @stability stable
     */
    static redshiftMultiUser(options: MultiUserHostedRotationOptions): HostedRotation;
    /**
     * MongoDB Single User.
     *
     * @stability stable
     */
    static mongoDbSingleUser(options?: SingleUserHostedRotationOptions): HostedRotation;
    /**
     * MongoDB Multi User.
     *
     * @stability stable
     */
    static mongoDbMultiUser(options: MultiUserHostedRotationOptions): HostedRotation;
    private _connections?;
    private constructor();
    /**
     * Binds this hosted rotation to a secret.
     *
     * @stability stable
     */
    bind(secret: ISecret, scope: Construct): CfnRotationSchedule.HostedRotationLambdaProperty;
    /**
     * Security group connections for this hosted rotation.
     *
     * @stability stable
     */
    get connections(): ec2.Connections;
}
/**
 * Hosted rotation type.
 *
 * @stability stable
 */
export declare class HostedRotationType {
    readonly name: string;
    readonly isMultiUser?: boolean | undefined;
    /**
     * MySQL Single User.
     *
     * @stability stable
     */
    static readonly MYSQL_SINGLE_USER: HostedRotationType;
    /**
     * MySQL Multi User.
     *
     * @stability stable
     */
    static readonly MYSQL_MULTI_USER: HostedRotationType;
    /**
     * PostgreSQL Single User.
     *
     * @stability stable
     */
    static readonly POSTGRESQL_SINGLE_USER: HostedRotationType;
    /**
     * PostgreSQL Multi User.
     *
     * @stability stable
     */
    static readonly POSTGRESQL_MULTI_USER: HostedRotationType;
    /**
     * Oracle Single User.
     *
     * @stability stable
     */
    static readonly ORACLE_SINGLE_USER: HostedRotationType;
    /**
     * Oracle Multi User.
     *
     * @stability stable
     */
    static readonly ORACLE_MULTI_USER: HostedRotationType;
    /**
     * MariaDB Single User.
     *
     * @stability stable
     */
    static readonly MARIADB_SINGLE_USER: HostedRotationType;
    /**
     * MariaDB Multi User.
     *
     * @stability stable
     */
    static readonly MARIADB_MULTI_USER: HostedRotationType;
    /**
     * SQL Server Single User.
     *
     * @stability stable
     */
    static readonly SQLSERVER_SINGLE_USER: HostedRotationType;
    /**
     * SQL Server Multi User.
     *
     * @stability stable
     */
    static readonly SQLSERVER_MULTI_USER: HostedRotationType;
    /**
     * Redshift Single User.
     *
     * @stability stable
     */
    static readonly REDSHIFT_SINGLE_USER: HostedRotationType;
    /**
     * Redshift Multi User.
     *
     * @stability stable
     */
    static readonly REDSHIFT_MULTI_USER: HostedRotationType;
    /**
     * MongoDB Single User.
     *
     * @stability stable
     */
    static readonly MONGODB_SINGLE_USER: HostedRotationType;
    /**
     * MongoDB Multi User.
     *
     * @stability stable
     */
    static readonly MONGODB_MULTI_USER: HostedRotationType;
    /**
     * @param name The type of rotation
     * @param isMultiUser Whether the rotation uses the mutli user scheme
     */
    private constructor();
}
