# PyPop7 (a Pure-PYthon library of POPulation-based black-box OPtimization)

[![GNU General Public License v3.0](https://img.shields.io/badge/license-GNU%20GPL--v3.0-green.svg)](https://github.com/Evolutionary-Intelligence/pypop/blob/main/LICENSE) [![PyPI for pypop7](https://img.shields.io/badge/PyPI-pypop7-yellowgreen.svg)](https://pypi.org/project/pypop7/) [![Documentation Status](https://readthedocs.org/projects/pypop/badge/?version=latest)](https://pypop.readthedocs.io/en/latest/?badge=latest) [![Downloads](https://static.pepy.tech/badge/pypop7)](https://pepy.tech/project/pypop7) [![Python](https://img.shields.io/badge/Python-3-yellow.svg)](https://www.python.org/) [![arxiv](https://img.shields.io/badge/arxiv-2212.05652-red)](https://arxiv.org/abs/2212.05652)

```PyPop7``` is a *Pure-PYthon* library of **POPulation-based OPtimization** for single-objective, real-parameter, black-box problems (currently actively maintained). Its goal is to provide a *unified* interface and *elegant* implementations for **Black-Box Optimization (BBO)**, *particularly population-based optimizers*, in order to facilitate research repeatability, benchmarking of BBO, and real-world applications.

<p align="center">
<img src="https://github.com/Evolutionary-Intelligence/pypop/blob/main/docs/logo/PyPop-Logo-Small-0.png" alt="drawing" width="321"/>
</p>

More specifically, for alleviating their **curse of dimensionality**, the primary focus of ```PyPop7``` is to cover their **State Of The Art for Large-Scale Optimization (LSO)**, though many of their small/medium-scaled versions and variants are also included here (mainly for theoretical or benchmarking purposes).

## How to Quickly Use PyPop7

The following three steps are enough to utilize the optimization power of this library [PyPop7](https://pypi.org/project/pypop7/):

1. Use [pip](https://pypi.org/project/pip/) to install ```pypop7``` on the Python3-based virtual environment via [venv](https://docs.python.org/3/library/venv.html) or [conda](https://docs.conda.io/projects/conda/en/latest/user-guide/tasks/manage-environments.html) (*a strong suggestion*):

```bash
$ pip install pypop7
```

For simplicity, all required library dependencies (except special cases) are *automatically* installed according to [setup.cfg](https://github.com/Evolutionary-Intelligence/pypop/blob/main/setup.cfg).

2. Define the objective/cost function (called *fitness function* in this library) for the optimization problem at hand,

3. Run one or more black-box optimizers on this optimization problem:

```Python
import numpy as np  # for numerical computation, which is also the computing engine of pypop7

# 2. Define your own objective/cost function for the optimization problem at hand:
#   the below example is Rosenbrock, the notorious test function from the optimization community
def rosenbrock(x):
    return 100.0*np.sum(np.power(x[1:] - np.power(x[:-1], 2), 2)) + np.sum(np.power(x[:-1] - 1, 2))

# define the fitness (cost) function and also its settings
ndim_problem = 1000
problem = {'fitness_function': rosenbrock,  # cost function
           'ndim_problem': ndim_problem,  # dimension
           'lower_boundary': -5.0*np.ones((ndim_problem,)),  # search boundary
           'upper_boundary': 5.0*np.ones((ndim_problem,))}

# 3. Run one or more black-box optimizers on the given optimization problem:
#   here we choose LM-MA-ES owing to its low complexity and metric-learning ability for LSO
#   https://pypop.readthedocs.io/en/latest/es/lmmaes.html
from pypop7.optimizers.es.lmmaes import LMMAES
# define all the necessary algorithm options (which differ among different optimizers)
options = {'fitness_threshold': 1e-10,  # terminate when the best-so-far fitness is lower than this threshold
           'max_runtime': 3600,  # 1 hours (terminate when the actual runtime exceeds it)
           'seed_rng': 0,  # seed of random number generation (which must be explicitly set for repeatability)
           'x': 4.0*np.ones((ndim_problem,)),  # initial mean of search (mutation/sampling) distribution
           'sigma': 3.0,  # initial global step-size of search distribution (not necessarily optimal)
           'verbose': 500}
lmmaes = LMMAES(problem, options)  # initialize the optimizer
results = lmmaes.optimize()  # run its (time-consuming) search process
print(results)
```

Note that for ```PyPop7```, the number ```7``` is added just because ```pypop``` has been registered by [other](http://pypop.org/) in [PyPI](https://pypi.org/). The icon *butterfly* for `PyPop7` is used to respect to the book (a complete variorum edition) of **Fisher**, ["the greatest of Darwin's successors"](https://link.springer.com/article/10.1007/s00265-010-1122-x): [The Genetical Theory of Natural Selection](https://global.oup.com/academic/product/the-genetical-theory-of-natural-selection-9780198504405?cc=gb&lang=en&) (where four butterflies were drawn in its cover), which inspired the proposal of [Genetic Algorithms (GA)](https://dl.acm.org/doi/10.1145/321127.321128).

For a list of **public use cases** of `PyPop7`, see [this online document](https://pypop.readthedocs.io/en/latest/applications.html) for more details. For new/missed black-box optimizers, we provide a *unified* API interface to freely add them if they satisfy the following design philosophy (see [development-guide](https://pypop.readthedocs.io/en/latest/development-guide.html) for details).

## A Large Number of Black-Box Optimizers (BBO)

<p align="center">
<img src="https://github.com/Evolutionary-Intelligence/pypop/blob/main/docs/logo/Summary-BBO.png" alt="drawing"/>
</p>

Note that Ant Colony Optimization ([ACO](https://www.sciencedirect.com/science/article/pii/B9781558603776500396)) and Tabu Search ([TS](https://www.science.org/doi/10.1126/science.267.5198.664)) are not covered in this open-source library, since they work mainly in *[discrete/combinatorial](https://github.com/airbus/discrete-optimization)* search spaces. Furthermore, brute-force search (exhaustive/grid search) is also excluded here, since it works only for *very low* (typically < 10) dimensions. In the near future version, we will consider adding [Simultaneous Perturbation Stochastic Approximation (SPSA)](https://www.jhuapl.edu/SPSA/) into this open-source library.

******* *** *******
* ![large--scale--optimization](https://img.shields.io/badge/***-large--scale--optimization-orange.svg): indicates the specific BBO version for LSO (dimension >= 1000).
* ![competitor](https://img.shields.io/badge/**-competitor-blue.svg): indicates the competitive (or *de facto*) BBO version for *small/medium-dimensional* problems (though it may work well under certain LSO circumstances).
* ![baseline](https://img.shields.io/badge/*-baseline-lightgrey.svg): indicates the baseline BBO version mainly for *theoretical* interest, owing to its simplicity (relatively ease to mathematical analysis).

Note that this classification based on only the dimension of objective function is just a rough estimation for algorithm selection. **In practice**, perhaps the simplest way to algorithm selection
is trial-and-error or to try more advanced [Automated Algorithm Selection](https://doi.org/10.1162/evco_a_00242) techniques.

******* *** *******

* **Evolution Strategies (ES)** [e.g., [Ollivier et al., 2017, JMLR](https://www.jmlr.org/papers/v18/14-467.html); [Hansen et al., 2015](https://link.springer.com/chapter/10.1007/978-3-662-43505-2_44); [Bäck et al., 2013](https://link.springer.com/book/10.1007/978-3-642-40137-4); [Rudolph, 2012](https://link.springer.com/referenceworkentry/10.1007/978-3-540-92910-9_22); [Beyer&Schwefel, 2002](https://link.springer.com/article/10.1023/A:1015059928466); [Rechenberg, 1989](https://link.springer.com/chapter/10.1007/978-3-642-83814-9_6); [Schwefel, 1984](https://link.springer.com/article/10.1007/BF01876146)]
  * ![large--scale--optimization](https://img.shields.io/badge/***-large--scale--optimization-orange.svg) Mixture Model-based Evolution Strategy (**[MMES](https://github.com/Evolutionary-Intelligence/pypop/blob/main/pypop7/optimizers/es/mmes.py)**) [[He et al., 2021, TEVC](https://ieeexplore.ieee.org/abstract/document/9244595)]
  * ![large--scale--optimization](https://img.shields.io/badge/***-large--scale--optimization-orange.svg) Fast Covariance Matrix Adaptation Evolution Strategy (**[FCMAES](https://github.com/Evolutionary-Intelligence/pypop/blob/main/pypop7/optimizers/es/fcmaes.py)**) [[Li et al., 2020, TCYB](https://ieeexplore.ieee.org/abstract/document/8533604); [Li&Zhang, 2016, PPSN](https://link.springer.com/chapter/10.1007/978-3-319-45823-6_70)]
  * ![large--scale--optimization](https://img.shields.io/badge/***-large--scale--optimization-orange.svg) Limited-Memory Matrix Adaptation Evolution Strategy (**[LMMAES](https://github.com/Evolutionary-Intelligence/pypop/blob/main/pypop7/optimizers/es/lmmaes.py)**) [[Loshchilov et al., 2019, TEVC](https://ieeexplore.ieee.org/abstract/document/8410043)]
  * ![large--scale--optimization](https://img.shields.io/badge/***-large--scale--optimization-orange.svg) Limited Memory Covariance Matrix Adaptation (**[LMCMA](https://github.com/Evolutionary-Intelligence/pypop/blob/main/pypop7/optimizers/es/lmcma.py)**) [[Loshchilov, 2017, ECJ](https://direct.mit.edu/evco/article-abstract/25/1/143/1041/LM-CMA-An-Alternative-to-L-BFGS-for-Large-Scale)]
    * ![large--scale--optimization](https://img.shields.io/badge/***-large--scale--optimization-orange.svg) Limited Memory Covariance Matrix Adaptation Evolution Strategy (**[LMCMAES](https://github.com/Evolutionary-Intelligence/pypop/blob/main/pypop7/optimizers/es/lmcmaes.py)**) [[Loshchilov, 2014, GECCO](https://dl.acm.org/doi/abs/10.1145/2576768.2598294)]
  * ![large--scale--optimization](https://img.shields.io/badge/***-large--scale--optimization-orange.svg) Rank-M Evolution Strategy (**[RMES](https://github.com/Evolutionary-Intelligence/pypop/blob/main/pypop7/optimizers/es/rmes.py)**) [[Li&Zhang, 2018, TEVC](https://ieeexplore.ieee.org/document/8080257)]
    * ![large--scale--optimization](https://img.shields.io/badge/***-large--scale--optimization-orange.svg) Rank-One Evolution Strategy (**[R1ES](https://github.com/Evolutionary-Intelligence/pypop/blob/main/pypop7/optimizers/es/r1es.py)**) [[Li&Zhang, 2018, TEVC](https://ieeexplore.ieee.org/document/8080257)]
  * ![large--scale--optimization](https://img.shields.io/badge/***-large--scale--optimization-orange.svg) Projection-based Covariance Matrix Adaptation (**[VKDCMA](https://github.com/Evolutionary-Intelligence/pypop/blob/main/pypop7/optimizers/es/vkdcma.py)**) [[Akimoto&Hansen, 2016, PPSN](https://link.springer.com/chapter/10.1007/978-3-319-45823-6_1); [Akimoto&Hansen, 2016, GECCO](https://dl.acm.org/doi/abs/10.1145/2908812.2908863)]
    * ![large--scale--optimization](https://img.shields.io/badge/***-large--scale--optimization-orange.svg) Linear Covariance Matrix Adaptation (**[VDCMA](https://github.com/Evolutionary-Intelligence/pypop/blob/main/pypop7/optimizers/es/vdcma.py)**) [[Akimoto et al., 2014, GECCO](https://dl.acm.org/doi/abs/10.1145/2576768.2598258)]
  * ![large--scale--optimization](https://img.shields.io/badge/***-large--scale--optimization-orange.svg) Cholesky-CMA-ES-2016 (**[CCMAES2016](https://github.com/Evolutionary-Intelligence/pypop/blob/main/pypop7/optimizers/es/ccmaes2016.py)**) [[Krause et al., 2016, NeurIPS](https://proceedings.neurips.cc/paper/2016/hash/289dff07669d7a23de0ef88d2f7129e7-Abstract.html)]
    * ![large--scale--optimization](https://img.shields.io/badge/***-large--scale--optimization-orange.svg) (1+1)-Active-Cholesky-CMA-ES-2015 (**[OPOA2015](https://github.com/Evolutionary-Intelligence/pypop/blob/main/pypop7/optimizers/es/opoa2015.py)**) [[Krause&Igel, 2015, FOGA](https://dl.acm.org/doi/abs/10.1145/2725494.2725496)]
    * ![large--scale--optimization](https://img.shields.io/badge/***-large--scale--optimization-orange.svg) (1+1)-Active-Cholesky-CMA-ES (**[OPOA2010](https://github.com/Evolutionary-Intelligence/pypop/blob/main/pypop7/optimizers/es/opoa2010.py)**) [[Arnold&Hansen, 2010, GECCO](https://dl.acm.org/doi/abs/10.1145/1830483.1830556)]
  * ![large--scale--optimization](https://img.shields.io/badge/***-large--scale--optimization-orange.svg) Cholesky-CMA-ES (**[CCMAES2009](https://github.com/Evolutionary-Intelligence/pypop/blob/main/pypop7/optimizers/es/ccmaes2009.py)**) [[Suttorp et al., 2009, MLJ](https://link.springer.com/article/10.1007/s10994-009-5102-1)]
    * ![large--scale--optimization](https://img.shields.io/badge/***-large--scale--optimization-orange.svg) (1+1)-Cholesky-CMA-ES-2009 (**[OPOC2009](https://github.com/Evolutionary-Intelligence/pypop/blob/main/pypop7/optimizers/es/opoc2009.py)**) [[Suttorp et al., 2009, MLJ](https://link.springer.com/article/10.1007/s10994-009-5102-1)]
    * ![large--scale--optimization](https://img.shields.io/badge/***-large--scale--optimization-orange.svg) (1+1)-Cholesky-CMA-ES (**[OPOC2006](https://github.com/Evolutionary-Intelligence/pypop/blob/main/pypop7/optimizers/es/opoc2006.py)**) [[Igel et al., 2006, GECCO](https://dl.acm.org/doi/abs/10.1145/1143997.1144082)]
  * ![large--scale--optimization](https://img.shields.io/badge/***-large--scale--optimization-orange.svg) Separable Covariance Matrix Adaptation Evolution Strategy (**[SEPCMAES](https://github.com/Evolutionary-Intelligence/pypop/blob/main/pypop7/optimizers/es/sepcmaes.py)**) [[Bäck et al., 2013](https://link.springer.com/book/10.1007/978-3-642-40137-4); [Ros&Hansen, 2008, PPSN](https://link.springer.com/chapter/10.1007/978-3-540-87700-4_30)]
  * ![competitor](https://img.shields.io/badge/**-competitor-blue.svg) Diagonal Decoding Covariance Matrix Adaptation (**[DDCMA](https://github.com/Evolutionary-Intelligence/pypop/blob/main/pypop7/optimizers/es/ddcma.py)**) [[Akimoto&Hansen, 2020, ECJ](https://direct.mit.edu/evco/article/28/3/405/94999/Diagonal-Acceleration-for-Covariance-Matrix)]
  * ![competitor](https://img.shields.io/badge/**-competitor-blue.svg) Matrix Adaptation Evolution Strategy (**[MAES](https://github.com/Evolutionary-Intelligence/pypop/blob/main/pypop7/optimizers/es/maes.py)**) [[Beyer&Sendhoff, 2017, TEVC](https://ieeexplore.ieee.org/abstract/document/7875115/)]
    * ![competitor](https://img.shields.io/badge/**-competitor-blue.svg) Fast Matrix Adaptation Evolution Strategy (**[FMAES](https://github.com/Evolutionary-Intelligence/pypop/blob/main/pypop7/optimizers/es/fmaes.py)**) [[Beyer, 2020, GECCO](https://dl.acm.org/doi/abs/10.1145/3377929.3389870); [Loshchilov et al., 2019, TEVC](https://ieeexplore.ieee.org/abstract/document/8410043)]
  * ![competitor](https://img.shields.io/badge/**-competitor-blue.svg) Covariance Matrix Adaptation Evolution Strategy (**[CMAES](https://github.com/Evolutionary-Intelligence/pypop/blob/main/pypop7/optimizers/es/cmaes.py)**) [e.g. [Hansen, 2016](https://arxiv.org/abs/1604.00772); [Hansen et al., 2003, ECJ](https://direct.mit.edu/evco/article-abstract/11/1/1/1139/Reducing-the-Time-Complexity-of-the-Derandomized); [Hansen et al., 2001, ECJ](https://direct.mit.edu/evco/article-abstract/9/2/159/892/Completely-Derandomized-Self-Adaptation-in); [Hansen&Ostermeier, 1996, CEC](https://ieeexplore.ieee.org/abstract/document/542381)]
  * ![baseline](https://img.shields.io/badge/*-baseline-lightgrey.svg) Self-Adaptation Matrix Adaptation Evolution Strategy (**[SAMAES](https://github.com/Evolutionary-Intelligence/pypop/blob/main/pypop7/optimizers/es/samaes.py)**) [[Beyer, 2020, GECCO](https://dl.acm.org/doi/abs/10.1145/3377929.3389870)]
    * ![baseline](https://img.shields.io/badge/*-baseline-lightgrey.svg) Self-Adaptation Evolution Strategy (**[SAES](https://github.com/Evolutionary-Intelligence/pypop/blob/main/pypop7/optimizers/es/saes.py)**) [e.g. [Beyer, 2020, GECCO](https://dl.acm.org/doi/abs/10.1145/3377929.3389870); [Beyer, 2007, Scholarpedia](http://www.scholarpedia.org/article/Evolution_strategies)]
    * ![baseline](https://img.shields.io/badge/*-baseline-lightgrey.svg) Cumulative Step-size Adaptation Evolution Strategy (**[CSAES](https://github.com/Evolutionary-Intelligence/pypop/blob/main/pypop7/optimizers/es/csaes.py)**)  [e.g. [Hansen et al., 2015](https://link.springer.com/chapter/10.1007/978-3-662-43505-2_44); [Ostermeier et al., 1994, PPSN](https://link.springer.com/chapter/10.1007/3-540-58484-6_263)]
    * ![baseline](https://img.shields.io/badge/*-baseline-lightgrey.svg) Derandomized Self-Adaptation Evolution Strategy (**[DSAES](https://github.com/Evolutionary-Intelligence/pypop/blob/main/pypop7/optimizers/es/dsaes.py)**) [e.g. [Hansen et al., 2015](https://link.springer.com/chapter/10.1007/978-3-662-43505-2_44); [Ostermeier et al., 1994, ECJ](https://direct.mit.edu/evco/article-abstract/2/4/369/1407/A-Derandomized-Approach-to-Self-Adaptation-of)]
    * ![baseline](https://img.shields.io/badge/*-baseline-lightgrey.svg) Schwefel's Self-Adaptation Evolution Strategy (**[SSAES](https://github.com/Evolutionary-Intelligence/pypop/blob/main/pypop7/optimizers/es/ssaes.py)**) [e.g. [Hansen et al., 2015](https://link.springer.com/chapter/10.1007/978-3-662-43505-2_44); [Beyer&Schwefel, 2002](https://link.springer.com/article/10.1023/A:1015059928466); [Schwefel, 1988](https://link.springer.com/chapter/10.1007/978-3-642-73953-8_8); [Schwefel, 1984, AOR](https://link.springer.com/article/10.1007/BF01876146)]
    * ![baseline](https://img.shields.io/badge/*-baseline-lightgrey.svg) Rechenberg's (1+1)-Evolution Strategy with 1/5th success rule (**[RES](https://github.com/Evolutionary-Intelligence/pypop/blob/main/pypop7/optimizers/es/res.py)**) [e.g. [Hansen et al., 2015](https://link.springer.com/chapter/10.1007/978-3-662-43505-2_44); [Kern et al., 2004](https://link.springer.com/article/10.1023/B:NACO.0000023416.59689.4e); [Rechenberg, 1989](https://link.springer.com/chapter/10.1007/978-3-642-83814-9_6); [Rechenberg, 1984](https://link.springer.com/chapter/10.1007/978-3-642-69540-7_13); [Schumer&Steiglitz, 1968, TAC](https://ieeexplore.ieee.org/abstract/document/1098903)]
* **Natural Evolution Strategies ([NES](https://github.com/Evolutionary-Intelligence/pypop/blob/main/pypop7/optimizers/nes/nes.py))** [e.g. [Wierstra et al., 2014, JMLR](https://jmlr.org/papers/v15/wierstra14a.html); [Yi et al., 2009, ICML](https://dl.acm.org/doi/abs/10.1145/1553374.1553522); [Wierstra et al., 2008, CEC](https://ieeexplore.ieee.org/document/4631255)]
  * ![large--scale--optimization](https://img.shields.io/badge/***-large--scale--optimization-orange.svg) Rank-One Natural Evolution Strategy (**[R1NES](https://github.com/Evolutionary-Intelligence/pypop/blob/main/pypop7/optimizers/nes/r1nes.py)**) [[Sun et al., 2013, GECCO](https://dl.acm.org/doi/abs/10.1145/2464576.2464608)]
  * ![large--scale--optimization](https://img.shields.io/badge/***-large--scale--optimization-orange.svg) Separable Natural Evolution Strategy (**[SNES](https://github.com/Evolutionary-Intelligence/pypop/blob/main/pypop7/optimizers/nes/snes.py)**) [[Schaul et al., 2011, GECCO](https://dl.acm.org/doi/abs/10.1145/2001576.2001692)]
  * ![competitor](https://img.shields.io/badge/**-competitor-blue.svg) Exponential Natural Evolution Strategies (**[XNES](https://github.com/Evolutionary-Intelligence/pypop/blob/main/pypop7/optimizers/nes/xnes.py)**) [e.g. [Glasmachers et al., 2010, GECCO](https://dl.acm.org/doi/abs/10.1145/1830483.1830557)]
  * ![competitor](https://img.shields.io/badge/**-competitor-blue.svg) Exact Natural Evolution Strategy (**[ENES](https://github.com/Evolutionary-Intelligence/pypop/blob/main/pypop7/optimizers/nes/enes.py)**) [e.g. [Sun et al., 2009, ICML](https://dl.acm.org/doi/abs/10.1145/1553374.1553522)]
  * ![baseline](https://img.shields.io/badge/*-baseline-lightgrey.svg) Original Natural Evolution Strategy (**[ONES](https://github.com/Evolutionary-Intelligence/pypop/blob/main/pypop7/optimizers/nes/ones.py)**) [e.g. [Wierstra et al., 2008, CEC](https://ieeexplore.ieee.org/abstract/document/4631255)]
  * ![baseline](https://img.shields.io/badge/*-baseline-lightgrey.svg) Search Gradient-based Evolution Strategy (**[SGES](https://github.com/Evolutionary-Intelligence/pypop/blob/main/pypop7/optimizers/nes/sges.py)**) [e.g. [Wierstra et al., 2008, CEC](https://ieeexplore.ieee.org/abstract/document/4631255)]
* **Estimation of Distribution Algorithms ([EDA](https://github.com/Evolutionary-Intelligence/pypop/blob/main/pypop7/optimizers/eda/eda.py))** [e.g. [Brookes et al., 2020, GECCO](https://dl.acm.org/doi/abs/10.1145/3377929.3389938); [Larrañaga&Lozano, 2002](https://link.springer.com/book/10.1007/978-1-4615-1539-5); [Pelikan et al., 2002](https://link.springer.com/article/10.1023/A:1013500812258); [Mühlenbein&Paaß, 1996, PPSN](https://link.springer.com/chapter/10.1007/3-540-61723-X_982); [Baluja&Caruana, 1995, ICML](https://www.sciencedirect.com/science/article/pii/B9781558603776500141)]
  * ![large--scale--optimization](https://img.shields.io/badge/***-large--scale--optimization-orange.svg) Random-Projection Estimation of Distribution Algorithm (**[RPEDA](https://github.com/Evolutionary-Intelligence/pypop/blob/main/pypop7/optimizers/eda/rpeda.py)**) [[Kabán et al., 2016, ECJ](https://direct.mit.edu/evco/article-abstract/24/2/255/1016/Toward-Large-Scale-Continuous-EDA-A-Random-Matrix)]
  * ![large--scale--optimization](https://img.shields.io/badge/***-large--scale--optimization-orange.svg) Univariate Marginal Distribution Algorithm (**[UMDA](https://github.com/Evolutionary-Intelligence/pypop/blob/main/pypop7/optimizers/eda/umda.py)**) [e.g. [Larrañaga&Lozano, 2002](https://link.springer.com/book/10.1007/978-1-4615-1539-5); [Mühlenbein, 1997, ECJ](https://tinyurl.com/yt78c786)]
  * ![competitor](https://img.shields.io/badge/**-competitor-blue.svg) Adaptive Estimation of Multivariate Normal Algorithm (**[AEMNA](https://github.com/Evolutionary-Intelligence/pypop/blob/main/pypop7/optimizers/eda/aemna.py)**)[[Larrañaga&Lozano, 2002](https://link.springer.com/book/10.1007/978-1-4615-1539-5)]
  * ![baseline](https://img.shields.io/badge/*-baseline-lightgrey.svg) Estimation of Multivariate Normal Algorithm (**[EMNA](https://github.com/Evolutionary-Intelligence/pypop/blob/main/pypop7/optimizers/eda/emna.py)**) [e.g. [Larrañaga&Lozano, 2002](https://link.springer.com/book/10.1007/978-1-4615-1539-5)]
* **Cross-Entropy Method ([CEM](https://github.com/Evolutionary-Intelligence/pypop/blob/main/pypop7/optimizers/cem/cem.py))** [e.g. [Rubinstein&Kroese, 2016](https://onlinelibrary.wiley.com/doi/book/10.1002/9781118631980); [Hu et al., 2007, OR](https://pubsonline.informs.org/doi/abs/10.1287/opre.1060.0367); [Kroese et al., 2006, MCAP](https://link.springer.com/article/10.1007/s11009-006-9753-0); [De Boer et al., 2005, AOR](https://link.springer.com/article/10.1007/s10479-005-5724-z); [Rubinstein&Kroese, 2004](https://link.springer.com/book/10.1007/978-1-4757-4321-0)]
  * ![large--scale--optimization](https://img.shields.io/badge/***-large--scale--optimization-orange.svg) Differentiable Cross-Entropy Method (**[DCEM](https://github.com/Evolutionary-Intelligence/pypop/blob/main/pypop7/optimizers/cem/dcem.py)**) [[Amos&Yarats, 2020, ICML](https://proceedings.mlr.press/v119/amos20a.html)]
  * ![competitor](https://img.shields.io/badge/**-competitor-blue.svg) Dynamic-Smoothing Cross-Entropy Method (**[DSCEM](https://github.com/Evolutionary-Intelligence/pypop/blob/main/pypop7/optimizers/cem/dscem.py)**) [e.g. [Kroese et al., 2006, MCAP](https://link.springer.com/article/10.1007/s11009-006-9753-0)]
  * ![baseline](https://img.shields.io/badge/*-baseline-lightgrey.svg) Model Reference Adaptive Search (**[MRAS](https://github.com/Evolutionary-Intelligence/pypop/blob/main/pypop7/optimizers/cem/mras.py)**) [e.g. [Hu et al., 2007, OR](https://pubsonline.informs.org/doi/abs/10.1287/opre.1060.0367)]
  * ![baseline](https://img.shields.io/badge/*-baseline-lightgrey.svg) Standard Cross-Entropy Method (**[SCEM](https://github.com/Evolutionary-Intelligence/pypop/blob/main/pypop7/optimizers/cem/scem.py)**) [e.g. [Kroese et al., 2006, MCAP](https://link.springer.com/article/10.1007/s11009-006-9753-0)]
* **Differential Evolution ([DE](https://github.com/Evolutionary-Intelligence/pypop/blob/main/pypop7/optimizers/de/de.py))** [e.g. [Price, 2013](https://link.springer.com/chapter/10.1007/978-3-642-30504-7_8); [Price et al., 2005](https://link.springer.com/book/10.1007/3-540-31306-0); [Storn&Price, 1997, JGO](https://link.springer.com/article/10.1023/A:1008202821328)]
  * ![large--scale--optimization](https://img.shields.io/badge/***-large--scale--optimization-orange.svg) Success-History based Adaptive Differential Evolution (**[SHADE](https://github.com/Evolutionary-Intelligence/pypop/blob/main/pypop7/optimizers/de/shade.py)**) [[Tanabe&Fukunaga, 2013, CEC](https://ieeexplore.ieee.org/document/6557555)]
  * ![large--scale--optimization](https://img.shields.io/badge/***-large--scale--optimization-orange.svg) Adaptive Differential Evolution (**[JADE](https://github.com/Evolutionary-Intelligence/pypop/blob/main/pypop7/optimizers/de/jade.py)**) [[Zhang&Sanderson, 2009, TEVC](https://doi.org/10.1109/TEVC.2009.2014613)]
  * ![competitor](https://img.shields.io/badge/**-competitor-blue.svg) Composite Differential Evolution (**[CODE](https://github.com/Evolutionary-Intelligence/pypop/blob/main/pypop7/optimizers/de/code.py)**) [[Wang et al., 2011, TEVC](https://doi.org/10.1109/TEVC.2010.2087271)]
  * ![baseline](https://img.shields.io/badge/*-baseline-lightgrey.svg) Trigonometric-mutation Differential Evolution (**[TDE](https://github.com/Evolutionary-Intelligence/pypop/blob/main/pypop7/optimizers/de/tde.py)**) [[Fan&Lampinen, 2003, JGO](https://link.springer.com/article/10.1023/A:1024653025686)]
  * ![baseline](https://img.shields.io/badge/*-baseline-lightgrey.svg) Classic Differential Evolution (**[CDE](https://github.com/Evolutionary-Intelligence/pypop/blob/main/pypop7/optimizers/de/cde.py)**) [e.g. [Storn&Price, 1997, JGO](https://link.springer.com/article/10.1023/A:1008202821328)]
* **Particle Swarm Optimizer ([PSO](https://github.com/Evolutionary-Intelligence/pypop/blob/main/pypop7/optimizers/pso/pso.py))** [e.g. [Fornasier et al., 2021, JMLR](https://jmlr.csail.mit.edu/papers/v22/21-0259.html); [Bonyadi&Michalewicz, 2017, ECJ](https://direct.mit.edu/evco/article-abstract/25/1/1/1040/Particle-Swarm-Optimization-for-Single-Objective); [Rahmat-Samii et al., 2012, PIEEE](https://ieeexplore.ieee.org/document/6204306); [Escalante et al., 2009, JMLR](https://www.jmlr.org/papers/v10/escalante09a.html); [Dorigo et al., 2008](http://www.scholarpedia.org/article/Particle_swarm_optimization); [Poli et al., 2007, SI](https://link.springer.com/article/10.1007/s11721-007-0002-0); [Shi&Eberhart, 1998, CEC](https://ieeexplore.ieee.org/abstract/document/699146); [Kennedy&Eberhart, 1995, ICNN](https://ieeexplore.ieee.org/document/488968)]
  * ![large--scale--optimization](https://img.shields.io/badge/***-large--scale--optimization-orange.svg) Cooperative Coevolving Particle Swarm Optimizer (**[CCPSO2](https://github.com/Evolutionary-Intelligence/pypop/blob/main/pypop7/optimizers/pso/ccpso2.py)**) [[Li&Yao, 2012, TEVC](https://ieeexplore.ieee.org/document/5910380/)]
  * ![large--scale--optimization](https://img.shields.io/badge/***-large--scale--optimization-orange.svg) Incremental Particle Swarm Optimizer (**[IPSO](https://github.com/Evolutionary-Intelligence/pypop/blob/main/pypop7/optimizers/pso/ipso.py)**) [[de Oca et al., 2011, TSMCB](https://ieeexplore.ieee.org/document/5582312)]
  * ![competitor](https://img.shields.io/badge/**-competitor-blue.svg) Comprehensive Learning Particle Swarm Optimizer (**[CLPSO](https://github.com/Evolutionary-Intelligence/pypop/blob/main/pypop7/optimizers/pso/clpso.py)**) [[Liang et al., 2006, TEVC](https://ieeexplore.ieee.org/abstract/document/1637688)]
  * ![baseline](https://img.shields.io/badge/*-baseline-lightgrey.svg) Cooperative Particle Swarm Optimizer (**[CPSO](https://github.com/Evolutionary-Intelligence/pypop/blob/main/pypop7/optimizers/pso/cpso.py)**) [[Van den Bergh&Engelbrecht, 2004, TEVC](https://ieeexplore.ieee.org/document/1304845)]
  * ![baseline](https://img.shields.io/badge/*-baseline-lightgrey.svg) Standard Particle Swarm Optimizer with a Local (ring) topology (**[SPSOL](https://github.com/Evolutionary-Intelligence/pypop/blob/main/pypop7/optimizers/pso/spsol.py)**) [e.g. [Shi&Eberhart, 1998, CEC](https://ieeexplore.ieee.org/abstract/document/699146); [Kennedy&Eberhart, 1995, ICNN](https://ieeexplore.ieee.org/document/488968)]
  * ![baseline](https://img.shields.io/badge/*-baseline-lightgrey.svg) Standard Particle Swarm Optimizer with a global topology (**[SPSO](https://github.com/Evolutionary-Intelligence/pypop/blob/main/pypop7/optimizers/pso/spso.py)**) [e.g. [Shi&Eberhart, 1998, CEC](https://ieeexplore.ieee.org/abstract/document/699146); [Kennedy&Eberhart, 1995, ICNN](https://ieeexplore.ieee.org/document/488968)]
* **Cooperative Co-evolution ([CC](https://github.com/Evolutionary-Intelligence/pypop/blob/main/pypop7/optimizers/cc/cc.py))** [e.g. [Gomez et al., 2008, JMLR](https://jmlr.org/papers/v9/gomez08a.html); [Panait et al., 2008, JMLR](https://www.jmlr.org/papers/v9/panait08a.html); [Moriarty&Miikkulainen, 1995, ICML](https://www.sciencedirect.com/science/article/pii/B9781558603776500566); [Potter&De Jong, 1994, PPSN](https://link.springer.com/chapter/10.1007/3-540-58484-6_269)]
  * ![large--scale--optimization](https://img.shields.io/badge/***-large--scale--optimization-orange.svg) Hierarchical Cooperative Co-evolution (**[HCC](https://github.com/Evolutionary-Intelligence/pypop/blob/main/pypop7/optimizers/cc/hcc.py)**) [[Mei et al., 2016, ACM-TOMS](https://dl.acm.org/doi/10.1145/2791291); [Gomez&Schmidhuber, 2005, ACM-GECCO](https://dl.acm.org/doi/10.1145/1068009.1068092)]
  * ![large--scale--optimization](https://img.shields.io/badge/***-large--scale--optimization-orange.svg) CoOperative CO-evolutionary Covariance Matrix Adaptation (**[COCMA](https://github.com/Evolutionary-Intelligence/pypop/blob/main/pypop7/optimizers/cc/cocma.py)**) [[Mei et al., 2016, ACM-TOMS](https://dl.acm.org/doi/10.1145/2791291); [Potter&De Jong, 1994, PPSN](https://link.springer.com/chapter/10.1007/3-540-58484-6_269)]
  * ![competitor](https://img.shields.io/badge/**-competitor-blue.svg) CoOperative co-Evolutionary Algorithm (**[COEA](https://github.com/Evolutionary-Intelligence/pypop/blob/main/pypop7/optimizers/cc/coea.py)**) [e.g. [Panait et al., 2008, JMLR](https://www.jmlr.org/papers/v9/panait08a.html); [Potter&De Jong, 1994, PPSN](https://link.springer.com/chapter/10.1007/3-540-58484-6_269)]
  * ![baseline](https://img.shields.io/badge/*-baseline-lightgrey.svg) CoOperative SYnapse NeuroEvolution (**[COSYNE](https://github.com/Evolutionary-Intelligence/pypop/blob/main/pypop7/optimizers/cc/cosyne.py)**) [[Gomez et al., 2008, JMLR](https://jmlr.org/papers/v9/gomez08a.html); [Moriarty&Miikkulainen, 1995, ICML](https://www.sciencedirect.com/science/article/pii/B9781558603776500566)]
* **Simulated Annealing ([SA](https://github.com/Evolutionary-Intelligence/pypop/blob/main/pypop7/optimizers/sa/sa.py))** [see e.g. [Bertsimas&Tsitsiklis, 1993, Statistical Science](https://tinyurl.com/yknunnpt); [Kirkpatrick et al., 1983, Science](https://www.science.org/doi/10.1126/science.220.4598.671); [Hastings, 1970, Biometrika](https://academic.oup.com/biomet/article/57/1/97/284580); [Metropolis et al., 1953, JCP](https://aip.scitation.org/doi/abs/10.1063/1.1699114)]
  * ![large--scale--optimization](https://img.shields.io/badge/***-large--scale--optimization-orange.svg) Enhanced Simulated Annealing (**[ESA](https://github.com/Evolutionary-Intelligence/pypop/blob/main/pypop7/optimizers/sa/esa.py)**) [[Siarry et al., 1997, ACM-TOMS](https://dl.acm.org/doi/abs/10.1145/264029.264043)]
  * ![competitor](https://img.shields.io/badge/**-competitor-blue.svg) Corana et al.' Simulated Annealing (**[CSA](https://github.com/Evolutionary-Intelligence/pypop/blob/main/pypop7/optimizers/sa/csa.py)**) [[Corana et al., 1987, ACM-TOMS](https://dl.acm.org/doi/abs/10.1145/29380.29864)]
  * ![baseline](https://img.shields.io/badge/*-baseline-lightgrey.svg) Noisy Simulated Annealing (**[NSA](https://github.com/Evolutionary-Intelligence/pypop/blob/main/pypop7/optimizers/sa/nsa.py)**) [[Bouttier&Gavra, 2019, JMLR](https://www.jmlr.org/papers/v20/16-588.html)]
* **Genetic Algorithms ([GA](https://github.com/Evolutionary-Intelligence/pypop/blob/main/pypop7/optimizers/ga/ga.py))** [e.g. [Forrest, 1993, Science](https://www.science.org/doi/abs/10.1126/science.8346439); [Holland, 1973, SICOMP](https://epubs.siam.org/doi/10.1137/0202009); [Holland, 1962, JACM](https://dl.acm.org/doi/10.1145/321127.321128)]
  * ![large--scale--optimization](https://img.shields.io/badge/***-large--scale--optimization-orange.svg) Active Subspace Genetic Algorithm (**[ASGA](https://github.com/Evolutionary-Intelligence/pypop/blob/main/pypop7/optimizers/ga/asga.py)**) [[Demo et al., 2021, SISC](https://epubs.siam.org/doi/10.1137/20M1345219)]
  * ![large--scale--optimization](https://img.shields.io/badge/***-large--scale--optimization-orange.svg) Global and Local genetic algorithm (**[GL25](https://github.com/Evolutionary-Intelligence/pypop/blob/main/pypop7/optimizers/ga/gl25.py)**) [[García-Martínez et al., 2008, EJOR](https://www.sciencedirect.com/science/article/abs/pii/S0377221706006308)]
  * ![competitor](https://img.shields.io/badge/**-competitor-blue.svg) Generalized Generation Gap with Parent-Centric Recombination (**[G3PCX](https://github.com/Evolutionary-Intelligence/pypop/blob/main/pypop7/optimizers/ga/g3pcx.py)**) [[Deb et al., 2002, ECJ](https://direct.mit.edu/evco/article-abstract/10/4/371/1136/A-Computationally-Efficient-Evolutionary-Algorithm)]
  * ![baseline](https://img.shields.io/badge/*-baseline-lightgrey.svg) GENetic ImplemenTOR (**[GENITOR](https://github.com/Evolutionary-Intelligence/pypop/blob/main/pypop7/optimizers/ga/genitor.py)**) [e.g. [Whitley et al., 1993, MLJ](https://link.springer.com/article/10.1023/A:1022674030396)]
* **Evolutionary Programming ([EP](https://github.com/Evolutionary-Intelligence/pypop/blob/main/pypop7/optimizers/ep/ep.py))** [e.g. [Yao et al., 1999, TEVC](https://ieeexplore.ieee.org/abstract/document/771163); [Fogel, 1994, Statistics and Computing](https://link.springer.com/article/10.1007/BF00175356)]
  * ![large--scale--optimization](https://img.shields.io/badge/***-large--scale--optimization-orange.svg) Lévy distribution based Evolutionary Programming (**[LEP](https://github.com/Evolutionary-Intelligence/pypop/blob/main/pypop7/optimizers/ep/lep.py)**) [[Lee&Yao, 2004, TEVC](https://ieeexplore.ieee.org/document/1266370)]
  * ![competitor](https://img.shields.io/badge/**-competitor-blue.svg) Fast Evolutionary Programming (**[FEP](https://github.com/Evolutionary-Intelligence/pypop/blob/main/pypop7/optimizers/ep/fep.py)**) [[Yao et al., 1999, TEVC](https://ieeexplore.ieee.org/abstract/document/771163)]
  * ![baseline](https://img.shields.io/badge/*-baseline-lightgrey.svg) Classical Evolutionary Programming (**[CEP](https://github.com/Evolutionary-Intelligence/pypop/blob/main/pypop7/optimizers/ep/cep.py)**) [e.g. [Yao et al., 1999, TEVC](https://ieeexplore.ieee.org/abstract/document/771163); [Bäck&Schwefel, 1993, ECJ](https://direct.mit.edu/evco/article-abstract/1/1/1/1092/An-Overview-of-Evolutionary-Algorithms-for)]
* **Direct Search ([DS](https://github.com/Evolutionary-Intelligence/pypop/blob/main/pypop7/optimizers/ds/ds.py))** [e.g. [Powell, 1998, Acta-Numerica](https://www.cambridge.org/core/journals/acta-numerica/article/abs/direct-search-algorithms-for-optimization-calculations/23FA5B19EAF122E02D3724DB1841238C); [Wright, 1996](https://nyuscholars.nyu.edu/en/publications/direct-search-methods-once-scorned-now-respectable); [Hooke&Jeeves, 1961, JACM](https://dl.acm.org/doi/10.1145/321062.321069)]
  * ![competitor](https://img.shields.io/badge/**-competitor-blue.svg) Powell's search method (**[POWELL](https://github.com/Evolutionary-Intelligence/pypop/blob/main/pypop7/optimizers/ds/powell.py)**) [[SciPy]( https://docs.scipy.org/doc/scipy/reference/optimize.minimize-powell.html); [Powell, 1964, Computer](https://github.com/Evolutionary-Intelligence/pypop/blob/main/pypop7/optimizers/ds/powell.py)]
  * ![baseline](https://img.shields.io/badge/*-baseline-lightgrey.svg) Generalized Pattern Search (**[GPS](https://github.com/Evolutionary-Intelligence/pypop/blob/main/pypop7/optimizers/ds/gps.py)**) [[Kochenderfer&Wheeler, 2019](https://algorithmsbook.com/optimization/files/chapter-7.pdf); [Torczon, 1997, SIAM-JO](https://epubs.siam.org/doi/abs/10.1137/S1052623493250780)]
  * ![baseline](https://img.shields.io/badge/*-baseline-lightgrey.svg) Nelder-Mead simplex method (**[NM](https://github.com/Evolutionary-Intelligence/pypop/blob/main/pypop7/optimizers/ds/nm.py)**) [[Dean et al., 1975, Science](https://www.science.org/doi/10.1126/science.189.4205.805); [Nelder&Mead, 1965, Computer](https://academic.oup.com/comjnl/article-abstract/7/4/308/354237)]
  * ![baseline](https://img.shields.io/badge/*-baseline-lightgrey.svg) Hooke-Jeeves direct search method (**[HJ](https://github.com/Evolutionary-Intelligence/pypop/blob/main/pypop7/optimizers/ds/hj.py)**) [[Kochenderfer&Wheeler, 2019](https://algorithmsbook.com/optimization/files/chapter-7.pdf); [Kaupe, 1963, CACM](https://dl.acm.org/doi/pdf/10.1145/366604.366632); [Hooke&Jeeves, 1961, JACM](https://dl.acm.org/doi/10.1145/321062.321069)]
  * ![baseline](https://img.shields.io/badge/*-baseline-lightgrey.svg) Coordinate Search (**[CS](https://github.com/Evolutionary-Intelligence/pypop/blob/main/pypop7/optimizers/ds/cs.py)**) [[Torczon, 1997, SIAM-JO](https://epubs.siam.org/doi/abs/10.1137/S1052623493250780); [Fermi&Metropolis, 1952](https://www.osti.gov/servlets/purl/4377177)]
* **Random (stochastic) Search ([RS](https://github.com/Evolutionary-Intelligence/pypop/blob/main/pypop7/optimizers/rs/rs.py))** [e.g. [Murphy, 2023](https://probml.github.io/pml-book/book2.html); [Gao&Sener, 2022, ICML](https://proceedings.mlr.press/v162/gao22f.html); [Russell&Norvig, 2021](http://aima.cs.berkeley.edu/); [Nesterov&Spokoiny, 2017, FoCM](https://link.springer.com/article/10.1007/s10208-015-9296-2); [Bergstra&Bengio, 2012, JMLR](https://www.jmlr.org/papers/v13/bergstra12a.html); [Schmidhuber et al., 2001](https://ml.jku.at/publications/older/ch9.pdf); [Cvijović&Klinowski, 1995, Science](https://www.science.org/doi/abs/10.1126/science.267.5198.664); [Rastrigin, 1986](https://link.springer.com/content/pdf/10.1007/BFb0007129.pdf); [Solis&Wets, 1981, MOOR](https://pubsonline.informs.org/doi/abs/10.1287/moor.6.1.19); [Brooks, 1958, OR](https://pubsonline.informs.org/doi/abs/10.1287/opre.6.2.244)]
  * ![large--scale--optimization](https://img.shields.io/badge/***-large--scale--optimization-orange.svg) BErnoulli Smoothing (**[BES](https://github.com/Evolutionary-Intelligence/pypop/blob/main/pypop7/optimizers/rs/bes.py)**) [[Gao&Sener, 2022, ICML](https://proceedings.mlr.press/v162/gao22f.html)]
  * ![large--scale--optimization](https://img.shields.io/badge/***-large--scale--optimization-orange.svg) Gaussian Smoothing (**[GS](https://github.com/Evolutionary-Intelligence/pypop/blob/main/pypop7/optimizers/rs/gs.py)**) [[Nesterov&Spokoiny, 2017, FoCM](https://link.springer.com/article/10.1007/s10208-015-9296-2)]
  * ![competitor](https://img.shields.io/badge/**-competitor-blue.svg) Simple Random Search (**[SRS](https://github.com/Evolutionary-Intelligence/pypop/blob/main/pypop7/optimizers/rs/srs.py)**) [[Rosenstein&Barto, 2001, IJCAI](https://dl.acm.org/doi/abs/10.5555/1642194.1642206)]
  * ![baseline](https://img.shields.io/badge/*-baseline-lightgrey.svg) Annealed Random Hill Climber (**[ARHC](https://github.com/Evolutionary-Intelligence/pypop/blob/main/pypop7/optimizers/rs/arhc.py)**) [e.g. [Russell&Norvig, 2021](http://aima.cs.berkeley.edu/); [Schaul et al., 2010, JMLR](https://jmlr.org/papers/v11/schaul10a.html)]
  * ![baseline](https://img.shields.io/badge/*-baseline-lightgrey.svg) Random Hill Climber (**[RHC](https://github.com/Evolutionary-Intelligence/pypop/blob/main/pypop7/optimizers/rs/rhc.py)**) [e.g. [Russell&Norvig, 2021](http://aima.cs.berkeley.edu/); [Schaul et al., 2010, JMLR](https://jmlr.org/papers/v11/schaul10a.html)]
  * ![baseline](https://img.shields.io/badge/*-baseline-lightgrey.svg) Pure Random Search (**[PRS](https://github.com/Evolutionary-Intelligence/pypop/blob/main/pypop7/optimizers/rs/prs.py)**) [e.g. [Bergstra&Bengio, 2012, JMLR](https://www.jmlr.org/papers/v13/bergstra12a.html); [Schmidhuber et al., 2001](https://ml.jku.at/publications/older/ch9.pdf); [Brooks, 1958, OR](https://pubsonline.informs.org/doi/abs/10.1287/opre.6.2.244)]

## Design Philosophy

Given a large number of (black-box) optimizers which keep increasing almost every day, we need some (possibly) widely acceptable criteria to select from them, as presented below in details:

* Respect for **Beauty (Elegance)**

  From the *problem-solving* perspective, we empirically prefer to choose the best optimizer for the black-box optimization problem at hand. For the new problem, however, the best optimizer is often *unknown* in advance (when without *a prior* knowledge). As a rule of thumb, we need to compare a (often small) set of available/well-known optimizers and finally choose the best one according to some predefined performance criteria. From the *academic research* perspective, however, we prefer so-called **beautiful** optimizers, though always keeping the [No Free Lunch Theorems](https://ieeexplore.ieee.org/document/585893) in mind. Typically, the beauty of one optimizer comes from the following attractive features: **model novelty**, **competitive performance on at least one class of real-world problems**, **theoretical insights (e.g., convergence)**, **clarity/simplicity for understanding and implementation**, and **repeatability**.

  If you find any to meet the above standard, welcome to launch [issues](https://github.com/Evolutionary-Intelligence/pypop/issues) or [pulls](https://github.com/Evolutionary-Intelligence/pypop/pulls) or [discussions](https://github.com/Evolutionary-Intelligence/pypop/discussions). We will consider it to be included in the *pypop7* library as soon as possible, if possible. Note that any [superficial](https://onlinelibrary.wiley.com/doi/full/10.1111/itor.13176) [imitation](https://dl.acm.org/doi/10.1145/3402220.3402221) to well-established optimizers (i.e. [Old Wine in a New Bottle](https://link.springer.com/article/10.1007/s11721-021-00202-9)) will be **NOT** considered here. Sometimes, several **very complex** optimizers could obtain the top rank on some competitions consisting of only *artificially-constructed* benchmark functions. However, these optimizers may become **over-skilled** on these artifacts. In our opinions, a good optimizer should be elegant and [generalizable](http://incompleteideas.net/IncIdeas/BitterLesson.html). If there is no persuasive real-world application reported for it, we will not consider any **very complex** optimizer in this library, in order to avoid the possible [repeatability](https://dl.acm.org/doi/full/10.1145/3466624) and [overfitting](http://incompleteideas.net/IncIdeas/BitterLesson.html) issues.

* Respect for **Diversity**

  Given the universality of **black-box optimization (BBO)** in science and engineering, different research communities have designed different optimizers/methods. The type and number of optimizers are continuing to increase as the more powerful optimizers are always desirable for new and more challenging applications. On the one hand, some of these methods may share *more or less* similarities. On the other hand, they may also show *significant* differences (w.r.t. motivations / objectives / implementations / communities / practitioners). Therefore, we hope to cover such a diversity from different research communities such as artificial intelligence (particularly machine learning [evolutionary computation](https://github.com/Evolutionary-Intelligence/DistributedEvolutionaryComputation) and
  zeroth-order optimization), mathematical optimization/programming (particularly global optimization), operations research / management science, automatic control, electronic engineering, open-source software, physics, chemistry, and others.

* Respect for **Originality**

  For each optimizer included in *PyPop7*, we expect to give its original/representative reference (sometimes also including its good implementations/improvements). If you find some important references missed, please do NOT hesitate to contact us (and we will be happy to add it if necessary).

* Respect for **Repeatability**

  For randomized search, properly controlling randomness is very crucial to repeat numerical experiments. Here we follow the [Random Sampling](https://numpy.org/doc/stable/reference/random/generator.html) suggestions from [NumPy](https://numpy.org/doc/stable/reference/random/). In other worlds, you must **explicitly** set the random seed for each optimizer. For more discussions about **repeatability** from machine learning, evolutionary computation, and metaheuristics communities, refer to the following papers, to name a few:

  * Swan, J., Adriaensen, S., Brownlee, A.E., Hammond, K., Johnson, C.G., Kheiri, A., Krawiec, F., Merelo, J.J.,
    Minku, L.L., Özcan, E., Pappa, G.L., et al., 2022. [Metaheuristics “in the large”](https://www.sciencedirect.com/science/article/pii/S0377221721004707). European Journal of Operational Research, 297(2), pp.393-406.
  * López-Ibáñez, M., Branke, J. and Paquete, L., 2021. [Reproducibility in evolutionary computation](https://dl.acm.org/doi/abs/10.1145/3466624). ACM Transactions on Evolutionary Learning and Optimization, 1(4), pp.1-21.
  * Sonnenburg, S., Braun, M.L., Ong, C.S., Bengio, S., Bottou, L., Holmes, G., LeCunn, Y., Muller, K.R., Pereira, F., Rasmussen, C.E., Ratsch, G., et al., 2007. [The need for open source software in machine learning](https://jmlr.csail.mit.edu/papers/volume8/sonnenburg07a/sonnenburg07a.pdf). Journal of Machine Learning Research, 8, pp.2443-2466.

## Computational Efficiency

For LSO, computational efficiency is an indispensable performance criterion of DFO [in the post-Moore era](https://www.science.org/doi/10.1126/science.aam9744). To obtain high-performance computation as much as possible, [NumPy](https://www.nature.com/articles/s41586-020-2649-2) is heavily used in this library as the base of numerical computation along with [SciPy](https://www.nature.com/articles/s41592-019-0686-2). Sometimes, [Numba](https://numba.pydata.org/) is also utilized, in order to further accelerate the wall-clock time.

## References

For each algorithm family, we provide several *representative* applications published on some [top-tier](https://github.com/Evolutionary-Intelligence/DistributedEvolutionaryComputation) journals and conferences (such as, Nature, Science, PNAS, PRL, JACS, PIEEE, etc.).

* Derivative-Free Optimization (DFO)
  * Berahas, A.S., Cao, L., Choromanski, K. and Scheinberg, K., 2022. [A theoretical and empirical comparison of gradient approximations in derivative-free optimization](https://link.springer.com/article/10.1007/s10208-021-09513-z). Foundations of Computational Mathematics, 22(2), pp.507-560.
  * Porcelli, M. and Toint, P.L., 2022. [Exploiting problem structure in derivative free optimization](https://dl.acm.org/doi/abs/10.1145/3474054). ACM Transactions on Mathematical Software, 48(1), pp.1-25.
  * Gao, K. and Sener, O., 2022, June. [Generalizing Gaussian smoothing for random search](https://proceedings.mlr.press/v162/gao22f.html). In International Conference on Machine Learning (pp. 7077-7101). PMLR.
  * Kochenderfer, M.J. and Wheeler, T.A., 2019. [Algorithms for optimization](https://algorithmsbook.com/optimization/). MIT Press.
  * Larson, J., Menickelly, M. and Wild, S.M., 2019. [Derivative-free optimization methods](https://www.cambridge.org/core/journals/acta-numerica/article/abs/derivativefree-optimization-methods/84479E2B03A9BFFE0F9CD46CF9FCD289). Acta Numerica, 28, pp.287-404.
  * Nesterov, Y., 2018. [Lectures on convex optimization](https://link.springer.com/book/10.1007/978-3-319-91578-4). Berlin: Springer International Publishing.
  * Nesterov, Y. and Spokoiny, V., 2017. [Random gradient-free minimization of convex functions](https://link.springer.com/article/10.1007/s10208-015-9296-2). Foundations of Computational Mathematics, 17(2), pp.527-566.
  * Conn, A.R., Scheinberg, K. and Vicente, L.N., 2009. [Introduction to derivative-free optimization](https://epubs.siam.org/doi/book/10.1137/1.9780898718768). Society for Industrial and Applied Mathematics.
* Evolutionary Computation (EC) [ [[Box, 1957]](https://rss.onlinelibrary.wiley.com/doi/abs/10.2307/2985505) ]
  * Eiben, A.E. and Smith, J., 2015. [From evolutionary computation to the evolution of things](https://www.nature.com/articles/nature14544.). Nature, 521(7553), pp.476-482. [ [http://www.evolutionarycomputation.org/](http://www.evolutionarycomputation.org/) ]
  * Miikkulainen, R. and Forrest, S., 2021. [A biological perspective on evolutionary computation](https://www.nature.com/articles/s42256-020-00278-8). Nature Machine Intelligence, 3(1), pp.9-15.
  * Hansen, N. and Auger, A., 2014. [Principled design of continuous stochastic search: From theory to practice](https://link.springer.com/chapter/10.1007/978-3-642-33206-7_8). Theory and Principled Methods for the Design of Metaheuristics, pp.145-180.
  * De Jong, K.A., 2006. [Evolutionary computation: A unified approach](https://mitpress.mit.edu/9780262041942/evolutionary-computation/). MIT Press.
  * Beyer, H.G. and Deb, K., 2001. [On self-adaptive features in real-parameter evolutionary algorithms](https://ieeexplore.ieee.org/abstract/document/930314). IEEE Transactions on Evolutionary Computation, 5(3), pp.250-270.
  * Salomon, R., 1998. [Evolutionary algorithms and gradient search: Similarities and differences](https://ieeexplore.ieee.org/abstract/document/728207). IEEE Transactions on Evolutionary Computation, 2(2), pp.45-55.
  * Fogel, D.B., 1998. [Evolutionary computation: The fossil record](https://ieeexplore.ieee.org/book/5263042). IEEE Press.
  * Wolpert, D.H. and Macready, W.G., 1997. [No free lunch theorems for optimization](https://ieeexplore.ieee.org/document/585893). IEEE Transactions on Evolutionary Computation, 1(1), pp.67-82.
  * Bäck, T. and Schwefel, H.P., 1993. [An overview of evolutionary algorithms for parameter optimization](https://direct.mit.edu/evco/article-abstract/1/1/1/1092/An-Overview-of-Evolutionary-Algorithms-for). Evolutionary Computation, 1(1), pp.1-23.
  * Forrest, S., 1993. [Genetic algorithms: Principles of natural selection applied to computation](https://www.science.org/doi/10.1126/science.8346439). Science, 261(5123), pp.872-878.
  * [Taxonomy](https://link.springer.com/article/10.1007/s11047-020-09820-4)
* Benchmarking [ [benchmarking-network](https://sites.google.com/view/benchmarking-network) + [iohprofiler](https://iohprofiler.github.io/) ]
  * Kudela, J., 2022. [A critical problem in benchmarking and analysis of evolutionary computation methods](https://www.nature.com/articles/s42256-022-00579-0). Nature Machine Intelligence, 4(12), pp.1238-1245.
  * Meunier, L., Rakotoarison, H., Wong, P.K., Roziere, B., Rapin, J., Teytaud, O., Moreau, A. and Doerr, C., 2022. [Black-box optimization revisited: Improving algorithm selection wizards through massive benchmarking](https://ieeexplore.ieee.org/abstract/document/9524335). IEEE Transactions on Evolutionary Computation, 26(3), pp.490-500.
  * Hansen, N., Auger, A., Ros, R., Mersmann, O., Tušar, T. and Brockhoff, D., 2021. [COCO: A platform for comparing continuous optimizers in a black-box setting](https://www.tandfonline.com/doi/full/10.1080/10556788.2020.1808977). Optimization Methods and Software, 36(1), pp.114-144.
  * Auger, A. and Hansen, N., 2021, July. [Benchmarking: State-of-the-art and beyond](https://dl.acm.org/doi/abs/10.1145/3449726.3461424). In Proceedings of Genetic and Evolutionary Computation Conference Companion (pp. 339-340). ACM.
  * Varelas, K., El Hara, O.A., Brockhoff, D., Hansen, N., Nguyen, D.M., Tušar, T. and Auger, A., 2020. [Benchmarking large-scale continuous optimizers: The bbob-largescale testbed, a COCO software guide and beyond](https://www.sciencedirect.com/science/article/abs/pii/S156849462030675X). Applied Soft Computing, 97, p.106737.
  * Hansen, N., Ros, R., Mauny, N., Schoenauer, M. and Auger, A., 2011. [Impacts of invariance in search: When CMA-ES and PSO face ill-conditioned and non-separable problems](https://www.sciencedirect.com/science/article/pii/S1568494611000974). Applied Soft Computing, 11(8), pp.5755-5769.
  * Moré, J.J. and Wild, S.M., 2009. [Benchmarking derivative-free optimization algorithms](https://epubs.siam.org/doi/abs/10.1137/080724083). SIAM Journal on Optimization, 20(1), pp.172-191.
  * Whitley, D., Rana, S., Dzubera, J. and Mathias, K.E., 1996. [Evaluating evolutionary algorithms](https://www.sciencedirect.com/science/article/pii/0004370295001247). Artificial Intelligence, 85(1-2), pp.245-276.
  * Salomon, R., 1996. [Re-evaluating genetic algorithm performance under coordinate rotation of benchmark functions. A survey of some theoretical and practical aspects of genetic algorithms](https://www.sciencedirect.com/science/article/abs/pii/0303264796016218). BioSystems, 39(3), pp.263-278.
  * Fogel, D.B. and Beyer, H.G., 1995. [A note on the empirical evaluation of intermediate recombination](https://direct.mit.edu/evco/article-abstract/3/4/491/749/A-Note-on-the-Empirical-Evaluation-of-Intermediate). Evolutionary Computation, 3(4), pp.491-495.
  * Moré, J.J., Garbow, B.S. and Hillstrom, K.E., 1981. [Testing unconstrained optimization software](https://dl.acm.org/doi/10.1145/355934.355936). ACM Transactions on Mathematical Software, 7(1), pp.17-41.
* Evolution Strategy (ES) [ [A visual guide to ES](https://blog.otoro.net/2017/10/29/visual-evolution-strategies/) + [[Li et al., 2020]](https://www.sciencedirect.com/science/article/abs/pii/S221065021930584X) + [[Akimoto&Hansen, 2022, GECCO-Companion]](http://www.cmap.polytechnique.fr/~nikolaus.hansen/gecco-2022-cma-tutorial.pdf) ]
  * Akimoto, Y., Auger, A., Glasmachers, T. and Morinaga, D., 2022. [Global linear convergence of evolution strategies on more than smooth strongly convex functions](https://epubs.siam.org/doi/abs/10.1137/20M1373815). SIAM Journal on Optimization, 32(2), pp.1402-1429.
  * Glasmachers, T. and Krause, O., 2022. [Convergence analysis of the Hessian estimation evolution strategy](https://direct.mit.edu/evco/article-abstract/30/1/27/102711/Convergence-Analysis-of-the-Hessian-Estimation). ECJ, 30(1), pp.27-50.
  * He, X., Zheng, Z. and Zhou, Y., 2021. [MMES: Mixture model-based evolution strategy for large-scale optimization](https://ieeexplore.ieee.org/abstract/document/9244595). IEEE-TEVC, 25(2), pp.320-333.
  * Akimoto, Y. and Hansen, N., 2020. [Diagonal acceleration for covariance matrix adaptation evolution strategies](https://direct.mit.edu/evco/article/28/3/405/94999/Diagonal-Acceleration-for-Covariance-Matrix). ECJ, 28(3), pp.405-435.
  * Beyer, H.G., 2020, July. [Design principles for matrix adaptation evolution strategies](https://dl.acm.org/doi/abs/10.1145/3377929.3389870). In ACM-GECCO Companion (pp. 682-700). ACM.
  * Choromanski, K., Pacchiano, A., Parker-Holder, J. and Tang, Y., 2019. [From complexity to simplicity: Adaptive es-active subspaces for blackbox optimization](https://papers.nips.cc/paper/2019/hash/88bade49e98db8790df275fcebb37a13-Abstract.html). In NeurIPS.
  * Varelas, K., Auger, A., Brockhoff, D., Hansen, N., ElHara, O.A., Semet, Y., Kassab, R. and Barbaresco, F., 2018, September. [A comparative study of large-scale variants of CMA-ES](https://link.springer.com/chapter/10.1007/978-3-319-99253-2_1). In PPSN (pp. 3-15). Springer, Cham.
  * Li, Z. and Zhang, Q., 2018. [A simple yet efficient evolution strategy for large-scale black-box optimization](https://ieeexplore.ieee.org/abstract/document/8080257). IEEE-TEVC, 22(5), pp.637-646.
  * Lehman, J., Chen, J., Clune, J. and Stanley, K.O., 2018, July. [ES is more than just a traditional finite-difference approximator](https://dl.acm.org/doi/abs/10.1145/3205455.3205474). In ACM-GECCO (pp. 450-457). ACM.
  * Ollivier, Y., Arnold, L., Auger, A. and Hansen, N., 2017. [Information-geometric optimization algorithms: A unifying picture via invariance principles](https://www.jmlr.org/papers/v18/14-467.html). Journal of Machine Learning Research, 18(18), pp.1-65.
  * Loshchilov, I., 2017. [LM-CMA: An alternative to L-BFGS for large-scale black box optimization](https://direct.mit.edu/evco/article-abstract/25/1/143/1041/LM-CMA-An-Alternative-to-L-BFGS-for-Large-Scale). ECJ, 25(1), pp.143-171. [ Loshchilov, I., 2014, July. [A computationally efficient limited memory CMA-ES for large scale optimization](https://dl.acm.org/doi/abs/10.1145/2576768.2598294). In ACM-GECCO (pp. 397-404). ACM. ] + [ Loshchilov, I., Glasmachers, T. and Beyer, H.G., 2019. [Large scale black-box optimization by limited-memory matrix adaptation](https://ieeexplore.ieee.org/abstract/document/8410043). IEEE-TEVC, 23(2), pp.353-358. ]
  * Beyer, H.G. and Sendhoff, B., 2017. [Simplify your covariance matrix adaptation evolution strategy](https://ieeexplore.ieee.org/document/7875115). IEEE Transactions on Evolutionary Computation, 21(5), pp.746-759.
  * Krause, O., Arbonès, D.R. and Igel, C., 2016. [CMA-ES with optimal covariance update and storage complexity](https://proceedings.neurips.cc/paper/2016/hash/289dff07669d7a23de0ef88d2f7129e7-Abstract.html). In Advances in Neural Information Processing Systems, 29, pp.370-378.
  * Akimoto, Y. and Hansen, N., 2016, July. [Projection-based restricted covariance matrix adaptation for high dimension](https://dl.acm.org/doi/abs/10.1145/2908812.2908863). In Proceedings of Annual Conference on Genetic and Evolutionary Computation (pp. 197-204). ACM.
  * Auger, A. and Hansen, N., 2016. [Linear convergence of comparison-based step-size adaptive randomized search via stability of Markov chains](https://epubs.siam.org/doi/10.1137/140984038). SIAM Journal on Optimization, 26(3), pp.1589-1624.
  * Krause, O. and Igel, C., 2015, January. [A more efficient rank-one covariance matrix update for evolution strategies](https://dl.acm.org/doi/abs/10.1145/2725494.2725496). In Proceedings of ACM Conference on Foundations of Genetic Algorithms (pp. 129-136). ACM.
  * Hansen, N., Arnold, D.V. and Auger, A., 2015. [Evolution strategies](https://link.springer.com/chapter/10.1007/978-3-662-43505-2_44). In Springer Handbook of Computational Intelligence (pp. 871-898). Springer, Berlin, Heidelberg.
  * Diouane, Y., Gratton, S. and Vicente, L.N., 2015. [Globally convergent evolution strategies](https://link.springer.com/article/10.1007/s10107-014-0793-x). Mathematical Programming, 152(1), pp.467-490.
  * Hansen, N., Atamna, A. and Auger, A., 2014, September. [How to assess step-size adaptation mechanisms in randomised search](https://link.springer.com/chapter/10.1007/978-3-319-10762-2_6). In International Conference on Parallel Problem Solving from Nature (pp. 60-69). Springer, Cham.
  * Akimoto, Y., Auger, A. and Hansen, N., 2014, July. [Comparison-based natural gradient optimization in high dimension](https://dl.acm.org/doi/abs/10.1145/2576768.2598258). In Proceedings of Annual Conference on Genetic and Evolutionary Computation (pp. 373-380). ACM.
  * Hansen, N. and Auger, A., 2014. [Principled design of continuous stochastic search: From theory to practice](https://link.springer.com/chapter/10.1007/978-3-642-33206-7_8). In Theory and Principled Methods for the Design of Metaheuristics (pp. 145-180). Springer, Berlin, Heidelberg.
  * Bäck, T., Foussette, C. and Krause, P., 2013. [Contemporary evolution strategies](https://link.springer.com/book/10.1007/978-3-642-40137-4). Berlin: Springer.
  * Rudolph, G., 2012. [Evolutionary strategies](https://link.springer.com/referenceworkentry/10.1007/978-3-540-92910-9_22). In Handbook of Natural Computing (pp. 673-698). Springer Berlin, Heidelberg.
  * Akimoto, Y., Nagata, Y., Ono, I. and Kobayashi, S., 2012. [Theoretical foundation for CMA-ES from information geometry perspective](https://link.springer.com/article/10.1007/s00453-011-9564-8). Algorithmica, 64(4), pp.698-716.
  * Akimoto, Y., 2011. [Design of evolutionary computation for continuous optimization](https://drive.google.com/file/d/18PW9syYDy-ndJA7wBmE2hRlxXJRBTTir/view). Doctoral Dissertation, Tokyo Institute of Technology.
  * Akimoto, Y., Nagata, Y., Ono, I. and Kobayashi, S., 2010, September. [Bidirectional relation between CMA evolution strategies and natural evolution strategies](https://link.springer.com/chapter/10.1007/978-3-642-15844-5_16). In International Conference on Parallel Problem Solving from Nature (pp. 154-163). Springer, Berlin, Heidelberg.
  * Arnold, D.V. and Hansen, N., 2010, July. [Active covariance matrix adaptation for the (1+1)-CMA-ES](https://dl.acm.org/doi/abs/10.1145/1830483.1830556). In Proceedings of Annual Conference on Genetic and Evolutionary Computation (pp. 385-392). ACM.
  * Suttorp, T., Hansen, N. and Igel, C., 2009. [Efficient covariance matrix update for variable metric evolution strategies](https://link.springer.com/article/10.1007/s10994-009-5102-1). Machine Learning, 75(2), pp.167-197.
  * Arnold, D.V. and MacLeod, A., 2006, July. [Hierarchically organised evolution strategies on the parabolic ridge](https://dl.acm.org/doi/abs/10.1145/1143997.1144080). In Proceedings of Annual Conference on Genetic and Evolutionary Computation (pp. 437-444). ACM.
  * Igel, C., Suttorp, T. and Hansen, N., 2006, July. [A computational efficient covariance matrix update and a (1+1)-CMA for evolution strategies](https://dl.acm.org/doi/abs/10.1145/1143997.1144082). In Proceedings of Annual Conference on Genetic and Evolutionary Computation (pp. 453-460). ACM.
  * Auger, A. and Hansen, N., 2005, September. [A restart CMA evolution strategy with increasing population size](https://ieeexplore.ieee.org/abstract/document/1554902). In IEEE Congress on Evolutionary Computation (pp. 1769-1776). IEEE.
  * Beyer, H.G. and Schwefel, H.P., 2002. [Evolution strategies–A comprehensive introduction](https://link.springer.com/article/10.1023/A:1015059928466). Natural Computing, 1(1), pp.3-52.
  * Hansen, N. and Ostermeier, A., 1996, May. [Adapting arbitrary normal mutation distributions in evolution strategies: The covariance matrix adaptation](https://ieeexplore.ieee.org/abstract/document/542381). In Proceedings of IEEE International Conference on Evolutionary Computation (pp. 312-317). IEEE. [ Hansen, N. and Ostermeier, A., 2001. [Completely derandomized self-adaptation in evolution strategies](https://direct.mit.edu/evco/article-abstract/9/2/159/892/Completely-Derandomized-Self-Adaptation-in). Evolutionary Computation, 9(2), pp.159-195. ] + [ Hansen, N., Müller, S.D. and Koumoutsakos, P., 2003. [Reducing the time complexity of the derandomized evolution strategy with covariance matrix adaptation (CMA-ES)](https://direct.mit.edu/evco/article-abstract/11/1/1/1139/Reducing-the-Time-Complexity-of-the-Derandomized). Evolutionary Computation, 11(1), pp.1-18. ]
  * Rudolph, G., 1992. [On correlated mutations in evolution strategies](https://ls11-www.cs.tu-dortmund.de/people/rudolph/publications/papers/PPSN92.pdf). In International Conference on Parallel Problem Solving from Nature (pp. 105-114).
  * Schwefel, H.P., 1984. [Evolution strategies: A family of non-linear optimization techniques based on imitating some principles of organic evolution](https://link.springer.com/article/10.1007/BF01876146). Annals of Operations Research, 1(2), pp.165-167. [ Schwefel, H.P., 1988. [Collective intelligence in evolving systems](https://link.springer.com/chapter/10.1007/978-3-642-73953-8_8). In Ecodynamics (pp. 95-100). Springer, Berlin, Heidelberg. ]
  * Rechenberg, I., 1984. [The evolution strategy. A mathematical model of darwinian evolution](https://link.springer.com/chapter/10.1007/978-3-642-69540-7_13). In Synergetics—from Microscopic to Macroscopic Order (pp. 122-132). Springer, Berlin, Heidelberg. [ Rechenberg, I., 1989. [Evolution strategy: Nature’s way of optimization](https://link.springer.com/chapter/10.1007/978-3-642-83814-9_6). In Optimization: Methods and Applications, Possibilities and Limitations (pp. 106-126). Springer, Berlin, Heidelberg. ]
  * Applications: e.g., [Lee et al., 2023, Science Robotics](https://www.science.org/doi/10.1126/scirobotics.adg3705); [Koginov et al., 2023, IEEE-TMRB](https://ieeexplore.ieee.org/document/10302449); [Sun et al., 2023, ACL](https://aclanthology.org/2023.acl-long.625.pdf); [Deng et al., 2023](https://www.chimechallenge.org/current/workshop/papers/CHiME_2023_DASR_deng.pdf); [Zhang et al., 2023, NeurIPS](https://arxiv.org/pdf/2310.18622.pdf); [Tjanaka et al., 2023, IEEE-LRA](https://scalingcmamae.github.io/); [Yu et al., 2023, IJCAI](https://www.ijcai.org/proceedings/2023/0187.pdf); [Zhu et al., 2023, IEEE/ASME-TMECH](https://ieeexplore.ieee.org/abstract/document/10250896); [Fadini et al., 2023](https://laas.hal.science/hal-04162737/file/versatileQuadrupedCodesign_preprint.pdf); [Ma et al., 2023](https://cs.brown.edu/~gdk/pubs/skillgen_verbs.pdf); [Kim et al., 2023, Science Robotics](https://www.science.org/doi/10.1126/scirobotics.add1053); [Slade et al., 2022, Nature](https://www.nature.com/articles/s41586-022-05191-1); [Sun et al., 2022, ICML](https://proceedings.mlr.press/v162/sun22e.html); [Tjanaka et al., 2022, GECCO](https://dl.acm.org/doi/10.1145/3512290.3528705); [Wang&Ponce, 2022, GECCO](https://dl.acm.org/doi/10.1145/3512290.3528725), [Tian&Ha, 2022, EvoStar](https://link.springer.com/chapter/10.1007/978-3-031-03789-4_18); [Hansel et al., 2021](https://link.springer.com/chapter/10.1007/978-3-030-41188-6_7); [Anand et al., 2021, MLST](https://iopscience.iop.org/article/10.1088/2632-2153/abf3ac); [Nomura et al., 2021, AAAI](https://ojs.aaai.org/index.php/AAAI/article/view/17109); [Zheng et al., 2021, IEEE-ASRU](https://ieeexplore.ieee.org/abstract/document/9688232); [Liu et al., 2019, AAAI](https://ojs.aaai.org/index.php/AAAI/article/view/4345); [Dong et al., 2019, CVPR](https://ieeexplore.ieee.org/document/8953400); [Ha&Schmidhuber, 2018, NeurIPS](https://papers.nips.cc/paper/2018/hash/2de5d16682c3c35007e4e92982f1a2ba-Abstract.html); [Müller&Glasmachers, 2018, PPSN](https://link.springer.com/chapter/10.1007/978-3-319-99259-4_33); [Chrabąszcz et al., 2018, IJCAI](https://www.ijcai.org/proceedings/2018/197); [OpenAI, 2017](https://openai.com/blog/evolution-strategies/); [Zhang et al., 2017, Science](https://www.science.org/doi/10.1126/science.aal5054); [Agrawal et al., 2014, TVCG](https://ieeexplore.ieee.org/abstract/document/6781622); [Heidrich-Meisner&Igel, 2009, ICML](https://dl.acm.org/doi/abs/10.1145/1553374.1553426); [Lipson&Pollack, 2000, Nature](https://www.nature.com/articles/35023115); [Wiesmann et al., 1998, IEEE-TEVC](https://ieeexplore.ieee.org/abstract/document/738986).
* Natural Evolution Strategies (NES)
  * Wierstra, D., Schaul, T., Glasmachers, T., Sun, Y., Peters, J. and Schmidhuber, J., 2014. [Natural evolution strategies](https://jmlr.org/papers/v15/wierstra14a.html). Journal of Machine Learning Research, 15(1), pp.949-980.
  * Schaul, T., 2011. [Studies in continuous black-box optimization](https://people.idsia.ch/~schaul/publications/thesis.pdf). Doctoral Dissertation, Technische Universität München.
  * Yi, S., Wierstra, D., Schaul, T. and Schmidhuber, J., 2009, June. [Stochastic search using the natural gradient](https://dl.acm.org/doi/10.1145/1553374.1553522). In Proceedings of International Conference on Machine Learning (pp. 1161-1168).
  * Wierstra, D., Schaul, T., Peters, J. and Schmidhuber, J., 2008, June. [Natural evolution strategies](https://ieeexplore.ieee.org/abstract/document/4631255). In IEEE Congress on Evolutionary Computation (pp. 3381-3387). IEEE.
  * Applications: e.g., [Yu et al., USENIX Security](https://www.usenix.org/conference/usenixsecurity23/presentation/yuzhiyuan); [Flageat et al., 2023](https://arxiv.org/abs/2303.06137); [Yan et al., 2023](https://arxiv.org/abs/2302.04477); [Feng et al., 2023](https://arxiv.org/abs/2303.06280); [Wei et al., 2022, IJCV](https://link.springer.com/article/10.1007/s11263-022-01604-w); [Agarwal et al., 2022, ICRA](https://ieeexplore.ieee.org/abstract/document/9811565); [Farid et al., 2022, CoRL](https://proceedings.mlr.press/v164/farid22a.html); [Feng et al., 2022, CVPR](https://openaccess.thecvf.com/content/CVPR2022/html/Feng_Boosting_Black-Box_Attack_With_Partially_Transferred_Conditional_Adversarial_Distribution_CVPR_2022_paper.html); [Berliner et al., 2022, ICLR](https://openreview.net/forum?id=JJCjv4dAbyL); [Kirsch et al., 2022, AAAI](https://ojs.aaai.org/index.php/AAAI/article/view/20681); [Jain et al., 2022, USENIX Security](https://www.usenix.org/conference/usenixsecurity22/presentation/jain); [Ilyas et al., 2018, ICML](https://proceedings.mlr.press/v80/ilyas18a.html).
* Estimation of Distribution Algorithm (EDA) [ [MIMIC [NeurIPS-1996]](https://proceedings.neurips.cc/paper/1996/hash/4c22bd444899d3b6047a10b20a2f26db-Abstract.html) + [BOA [GECCO-1999]](https://dl.acm.org/doi/abs/10.5555/2933923.2933973) + [[ECJ-2005]](https://direct.mit.edu/evco/article-abstract/13/1/99/1198/Drift-and-Scaling-in-Estimation-of-Distribution) ]
  * Brookes, D., Busia, A., Fannjiang, C., Murphy, K. and Listgarten, J., 2020, July. [A view of estimation of distribution algorithms through the lens of expectation-maximization](https://dl.acm.org/doi/abs/10.1145/3377929.3389938). In Proceedings of Genetic and Evolutionary Computation Conference Companion (pp. 189-190). ACM.
  * Kabán, A., Bootkrajang, J. and Durrant, R.J., 2016. [Toward large-scale continuous EDA: A random matrix theory perspective](https://direct.mit.edu/evco/article-abstract/24/2/255/1016/Toward-Large-Scale-Continuous-EDA-A-Random-Matrix). Evolutionary Computation, 24(2), pp.255-291.
  * Pelikan, M., Hauschild, M.W. and Lobo, F.G., 2015. [Estimation of distribution algorithms](https://link.springer.com/chapter/10.1007/978-3-662-43505-2_45). In Springer Handbook of Computational Intelligence (pp. 899-928). Springer, Berlin, Heidelberg.
  * Dong, W., Chen, T., Tiňo, P. and Yao, X., 2013. [Scaling up estimation of distribution algorithms for continuous optimization](https://ieeexplore.ieee.org/document/6461934). IEEE Transactions on Evolutionary Computation, 17(6), pp.797-822.
  * Hauschild, M. and Pelikan, M., 2011. [An introduction and survey of estimation of distribution algorithms](https://www.sciencedirect.com/science/article/abs/pii/S2210650211000435). Swarm and Evolutionary Computation, 1(3), pp.111-128.
  * Teytaud, F. and Teytaud, O., 2009, July. [Why one must use reweighting in estimation of distribution algorithms](https://dl.acm.org/doi/10.1145/1569901.1569964). In Proceedings of ACM Annual Conference on Genetic and Evolutionary Computation (pp. 453-460).
  * Larrañaga, P. and Lozano, J.A. eds., 2001. [Estimation of distribution algorithms: A new tool for evolutionary computation](https://link.springer.com/book/10.1007/978-1-4615-1539-5). Springer Science & Business Media.
  * Mühlenbein, H., 1997. [The equation for response to selection and its use for prediction](https://tinyurl.com/yt78c786).  Evolutionary Computation, 5(3), pp.303-346.
  * Baluja, S. and Caruana, R., 1995. [Removing the genetics from the standard genetic algorithm](https://www.sciencedirect.com/science/article/pii/B9781558603776500141). In International Conference on Machine Learning (pp. 38-46). Morgan Kaufmann.
* Cross-Entropy Method (CEM)
  * Pinneri, C., Sawant, S., Blaes, S., Achterhold, J., Stueckler, J., Rolinek, M. and Martius, G., 2021, October. [Sample-efficient cross-entropy method for real-time planning](https://proceedings.mlr.press/v155/pinneri21a.html). In Conference on Robot Learning (pp. 1049-1065). PMLR.
  * Amos, B. and Yarats, D., 2020, November. [The differentiable cross-entropy method](https://proceedings.mlr.press/v119/amos20a.html). In International Conference on Machine Learning (pp. 291-302). PMLR.
  * Rubinstein, R.Y. and Kroese, D.P., 2016. [Simulation and the Monte Carlo method (Third Edition)](https://onlinelibrary.wiley.com/doi/book/10.1002/9781118631980). John Wiley & Sons.
  * Hu, J., Fu, M.C. and Marcus, S.I., 2007. [A model reference adaptive search method for global optimization](https://pubsonline.informs.org/doi/abs/10.1287/opre.1060.0367). Operations Research, 55(3), pp.549-568.
  * De Boer, P.T., Kroese, D.P., Mannor, S. and Rubinstein, R.Y., 2005. [A tutorial on the cross-entropy method](https://link.springer.com/article/10.1007/s10479-005-5724-z). Annals of Operations Research, 134(1), pp.19-67.
  * Rubinstein, R.Y. and Kroese, D.P., 2004. [The cross-entropy method: A unified approach to combinatorial optimization, Monte-Carlo simulation, and machine learning](https://link.springer.com/book/10.1007/978-1-4757-4321-0). New York: Springer.
  * Mannor, S., Rubinstein, R.Y. and Gat, Y., 2003. [The cross entropy method for fast policy search](https://dl.acm.org/doi/abs/10.5555/3041838.3041903). In Proceedings of International Conference on Machine Learning (pp. 512-519).
  * Applications: e.g., [Wang&Ba,2020, ICLR](https://openreview.net/forum?id=H1exf64KwH); [Hafner et al., 2019, ICML](https://proceedings.mlr.press/v97/hafner19a.html); [Pourchot&Sigaud, 2019, ICLR](https://openreview.net/forum?id=BkeU5j0ctQ); [Simmons-Edler et al., 2019, ICML-RL4RealLife](https://openreview.net/forum?id=SyeHbtgSiN); [Chua et al., 2018, NeurIPS](https://proceedings.neurips.cc/paper/2018/file/3de568f8597b94bda53149c7d7f5958c-Paper.pdf); [Duan et al., 2016, ICML](https://proceedings.mlr.press/v48/duan16.html); [Kobilarov, 2012, IJRR](https://journals.sagepub.com/doi/10.1177/0278364912444543).
* Differential Evolution (DE)
  * Price, K.V., 2013. [Differential evolution](https://link.springer.com/chapter/10.1007/978-3-642-30504-7_8). In Handbook of Optimization (pp. 187-214). Springer, Berlin, Heidelberg.
  * Tanabe, R. and Fukunaga, A., 2013, June. [Success-history based parameter adaptation for differential evolution](https://ieeexplore.ieee.org/document/6557555). In IEEE Congress on Evolutionary Computation (pp. 71-78). IEEE.
  * Wang, Y., Cai, Z., and Zhang, Q. 2011. [Differential evolution with composite trial vector generation strategies and control parameters](https://doi.org/10.1109/TEVC.2010.2087271). IEEE Transactions on Evolutionary Computation, 15(1), pp.55–66.
  * Zhang, J., and Sanderson, A. C. 2009. [JADE: Adaptive differential evolution with optional external archive](https://ieeexplore.ieee.org/document/5208221/). IEEE Transactions on Evolutionary Computation, 13(5), pp.945–958.
  * Price, K.V., Storn, R.M. and Lampinen, J.A., 2005. [Differential evolution: A practical approach to global optimization](https://link.springer.com/book/10.1007/3-540-31306-0). Springer Science & Business Media.
  * Fan, H.Y. and Lampinen, J., 2003. [A trigonometric mutation operation to differential evolution](https://link.springer.com/article/10.1023/A:1024653025686). Journal of Global Optimization, 27(1), pp.105-129.
  * Storn, R.M. and Price, K.V. 1997. [Differential evolution – a simple and efficient heuristic for global optimization over continuous spaces](https://doi.org/10.1023/A:1008202821328). Journal of Global Optimization, 11(4), pp.341–359.
  * Applications: e.g., [Higgins et al., 2023, Science](https://www.science.org/doi/10.1126/science.add5190); [McNulty et al., 2023, PRL](https://journals.aps.org/prl/abstract/10.1103/PhysRevLett.130.100801); [Koob et al., 2023, Psychological Review](https://psycnet.apa.org/record/2021-99615-001); [Colombo et al., 2023, Sci. Adv.](https://www.science.org/doi/full/10.1126/sciadv.ade5839); [Lichtinger&Biggin, 2023, JCTC](https://pubs.acs.org/doi/full/10.1021/acs.jctc.3c00140); [Liang et al., 2023, NSDI](https://www.usenix.org/system/files/nsdi23-liang-chieh-jan.pdf); [Shinn et al., 2023, Nature Neuroscience](https://www.nature.com/articles/s41593-023-01299-3); [Schad et al., 2023, ApJ](https://iopscience.iop.org/article/10.3847/1538-4357/acabbd); [Hoyer et al., 2023, MNRAS](https://academic.oup.com/mnras/article/520/3/4664/6994541); [Hoyer et al., 2023, ApJL](https://iopscience.iop.org/article/10.3847/2041-8213/aca53e); [Abdelnabi&Fritz, 2023,USENIX Security](https://www.usenix.org/conference/usenixsecurity23/presentation/abdelnabi); [Kotov et al., 2023, Cell Reports](https://www.cell.com/cell-reports/pdf/S2211-1247(23)00842-2.pdf); [Sidhartha et al., 2023, CVPR](https://openaccess.thecvf.com/content/CVPR2023/papers/Sidhartha_Adaptive_Annealing_for_Robust_Geometric_Estimation_CVPR_2023_paper.pdf); [Hardy et al., 2023, MNRAS](https://academic.oup.com/mnras/article-abstract/520/4/6111/6998583); [Boucher et al., 2023](https://arxiv.org/pdf/2306.07033.pdf); [Michel et al., 2023, PRA](https://journals.aps.org/pra/abstract/10.1103/PhysRevA.107.042602); [Woo et al., 2023, iScience](https://www.cell.com/iscience/pdf/S2589-0042(22)02008-9.pdf); [Bozkurt et al., 2023](https://arxiv.org/pdf/2307.04830.pdf); [Ma et al., 2023, KDD](https://dl.acm.org/doi/pdf/10.1145/3580305.3599524); [Zhou et al., 2023](https://arxiv.org/pdf/2301.12738.pdf); [Czarnik et al., 2023](https://arxiv.org/pdf/2307.05302.pdf); [Katic et al., 2023, iScience](https://pdf.sciencedirectassets.com/318494/1-s2.0-S2589004222X00138/1-s2.0-S2589004222021472/main.pdf); [Khajehnejad et al., 2023, RSIF](https://royalsocietypublishing.org/doi/full/10.1098/rsif.2022.0808); [Digman&Cornish, 2023, PRD](https://journals.aps.org/prd/abstract/10.1103/PhysRevD.108.023022); [Rommel et al., 2023](https://arxiv.org/pdf/2308.08062.pdf); [Li et al., 2022, Science](https://www.science.org/doi/full/10.1126/science.abj2096); [Schlegelmilch et al., 2022, Psychological Review](https://psycnet.apa.org/record/2021-81557-001); [Mackin et al., 2022, Nature Communications](https://www.nature.com/articles/s41467-022-31405-1); [Liu&Wang, 2022, JSAC](https://ieeexplore.ieee.org/abstract/document/9681872); [Zhou et al., 2022, Nature Computational Science](https://www.nature.com/articles/s43588-022-00232-1); [Fischer et al., 2022, TOCHI](https://dl.acm.org/doi/full/10.1145/3524122); [Ido et al., 2022, npj Quantum Materials](https://www.nature.com/articles/s41535-022-00452-8); [Clark et al., 2022, NECO](https://direct.mit.edu/neco/article/34/7/1545/111332/Reduced-Dimension-Biophysical-Neuron-Models); [Powell et al., 2022, ApJ](https://iopscience.iop.org/article/10.3847/1538-4357/ac8934/meta); [Vo et al., 2022, ICLR](https://openreview.net/forum?id=73MEhZ0anV); [Andersson et al., 2022, ApJ](https://iopscience.iop.org/article/10.3847/1538-4357/ac64a4/meta); [Naudin et al., 2022, NECO](https://direct.mit.edu/neco/article-abstract/34/10/2075/112571/A-Simple-Model-of-Nonspiking-Neurons); [Perini et al., 2022, AAAI](https://ojs.aaai.org/index.php/AAAI/article/view/20331); [Sterbentz et al., 2022, Physics of Fluids](https://pubs.aip.org/aip/pof/article/34/8/082109/2846942); [Mishra et al., 2021, Science](https://www.science.org/doi/full/10.1126/science.aav0780); [Tiwari et al., 2021, PRB](https://journals.aps.org/prb/abstract/10.1103/PhysRevB.103.014432); [Mok et al., 2021, Communications Physics](https://www.nature.com/articles/s42005-021-00572-w); [Vinker et al., 2021, CVPR](https://openaccess.thecvf.com/content/ICCV2021/papers/Vinker_Unpaired_Learning_for_High_Dynamic_Range_Image_Tone_Mapping_ICCV_2021_paper.pdf); [Mehta et al., 2021, JCAP](https://iopscience.iop.org/article/10.1088/1475-7516/2021/07/033); [Trueblood et al., 2021, Psychological Review](https://psycnet.apa.org/record/2020-63299-001); [Verdonck et al., 2021, Psychological Review](https://psycnet.apa.org/record/2020-66308-001); [Robert et al., 2021, npj Quantum Information](https://www.nature.com/articles/s41534-021-00368-4); [Canton et al., 2021, ApJ](https://iopscience.iop.org/article/10.3847/1538-4357/ac2f9a/meta); [Leslie et al., 2021, PRD](https://journals.aps.org/prd/abstract/10.1103/PhysRevD.104.123030); [Fengler et al., 2021, eLife](https://elifesciences.org/articles/65074); [Li et al., 2021, TQE](https://ieeexplore.ieee.org/abstract/document/9495278); [Chen et al., 2021, ACS Photonics](https://pubs.acs.org/doi/full/10.1021/acsphotonics.1c00915); [Menczel et al., 2021, J. Phys. A: Math. Theor.](https://iopscience.iop.org/article/10.1088/1751-8121/ac0c8f/meta); [Feng et al., 2021, JSAC](https://ieeexplore.ieee.org/abstract/document/9461747); [DES Collaboration, 2021, A&A](https://www.aanda.org/articles/aa/full_html/2021/12/aa41744-21/aa41744-21.html); [An et al., 2020, PNAS](https://www.pnas.org/doi/suppl/10.1073/pnas.1920338117); [Su et al., 2019, TEVC](https://ieeexplore.ieee.org/abstract/document/8601309); [Laganowsky et al., 2014, Nature](https://www.nature.com/articles/nature13419).
* Particle Swarm Optimizer (PSO) [ [swarm intelligence](https://www.cs.cmu.edu/~arielpro/15381f16/c_slides/781f16-26.pdf) | [scholarpedia](http://www.scholarpedia.org/article/Particle_swarm_optimization) ]
  * Sünnen, P., 2023. [Analysis of a consensus-based optimization method on hypersurfaces and applications](https://mediatum.ub.tum.de/doc/1647263/document.pdf). Doctoral dissertation, Technische Universität München.
  * Fornasier, M., Huang, H., Pareschi, L. and Sünnen, P., 2021. [Consensus-based optimization on the sphere: Convergence to global minimizers and machine learning](https://jmlr.csail.mit.edu/papers/v22/21-0259.html). Journal of Machine Learning Research, 22(1), pp.10722-10776.
  * Carrillo, J.A., Choi, Y.P., Totzeck, C. and Tse, O., 2018. [An analytical framework for consensus-based global optimization method](https://www.worldscientific.com/doi/abs/10.1142/S0218202518500276). Mathematical Models and Methods in Applied Sciences, 28(06), pp.1037-1066.
  * Blackwell, T. and Kennedy, J., 2018. [Impact of communication topology in particle swarm optimization](https://ieeexplore.ieee.org/abstract/document/8531770). IEEE Transactions on Evolutionary Computation, 23(4), pp.689-702.
  * Pinnau, R., Totzeck, C., Tse, O. and Martin, S., 2017. [A consensus-based model for global optimization and its mean-field limit](https://www.worldscientific.com/doi/abs/10.1142/S0218202517400061). Mathematical Models and Methods in Applied Sciences, 27(01), pp.183-204.
  * Bonyadi, M.R. and Michalewicz, Z., 2017. [Particle swarm optimization for single objective continuous space problems: A review](https://direct.mit.edu/evco/article-abstract/25/1/1/1040/Particle-Swarm-Optimization-for-Single-Objective). Evolutionary Computation, 25(1), pp.1-54.
  * Escalante, H.J., Montes, M. and Sucar, L.E., 2009. [Particle swarm model selection](https://www.jmlr.org/papers/v10/escalante09a.html). Journal of Machine Learning Research, 10(15), pp.405−440.
  * Floreano, D. and Mattiussi, C., 2008. [Bio-inspired artificial intelligence: Theories, methods, and technologies](https://mitpress.mit.edu/9780262062718/bio-inspired-artificial-intelligence/). MIT Press.
  * Poli, R., Kennedy, J. and Blackwell, T., 2007. [Particle swarm optimization](https://link.springer.com/article/10.1007/s11721-007-0002-0). Swarm Intelligence, 1(1), pp.33-57.
  * Venter, G. and Sobieszczanski-Sobieski, J., 2003. [Particle swarm optimization](https://arc.aiaa.org/doi/abs/10.2514/2.2111). AIAA Journal, 41(8), pp.1583-1589.
  * Parsopoulos, K.E. and Vrahatis, M.N., 2002. [Recent approaches to global optimization problems through particle swarm optimization](https://link.springer.com/article/10.1023/A:1016568309421). Natural Computing, 1(2), pp.235-306.
  * Clerc, M. and Kennedy, J., 2002. [The particle swarm-explosion, stability, and convergence in a multidimensional complex space](https://ieeexplore.ieee.org/abstract/document/985692). IEEE Transactions on Evolutionary Computation, 6(1), pp.58-73.
  * Eberhart, R.C., Shi, Y. and Kennedy, J., 2001. [Swarm intelligence](https://www.elsevier.com/books/swarm-intelligence/eberhart/978-1-55860-595-4). Elsevier.
  * Shi, Y. and Eberhart, R., 1998, May. [A modified particle swarm optimizer](https://ieeexplore.ieee.org/abstract/document/699146). In IEEE World Congress on Computational Intelligence (pp. 69-73). IEEE.
  * Kennedy, J. and Eberhart, R., 1995, November. [Particle swarm optimization](https://ieeexplore.ieee.org/document/488968). In Proceedings of International Conference on Neural Networks (pp. 1942-1948). IEEE.
  * Eberhart, R. and Kennedy, J., 1995, October. [A new optimizer using particle swarm theory](https://ieeexplore.ieee.org/abstract/document/494215). In Proceedings of International Symposium on Micro Machine and Human Science (pp. 39-43). IEEE.
  * Applications: e.g., [Grabner et al., 2023, Nature Communications](https://www.nature.com/articles/s41467-023-38943-2); [Morselli et al., 2023, IEEE-TWC](https://ieeexplore.ieee.org/abstract/document/10127621); [Reddy et al., 2023, IEEE-TC](https://ieeexplore.ieee.org/document/10005787); [Zhang et al., 2022, CVPR](https://openaccess.thecvf.com/content/CVPR2022/html/Zhang_On_Adversarial_Robustness_of_Trajectory_Prediction_for_Autonomous_Vehicles_CVPR_2022_paper.html); [Yang et al., PRL, 2022](https://journals.aps.org/prl/abstract/10.1103/PhysRevLett.128.065701); [Guan et al., 2022, PRL](https://journals.aps.org/prl/abstract/10.1103/PhysRevLett.128.186001); [Zhong et al., 2022, CVPR](https://openaccess.thecvf.com/content/CVPR2022/html/Zhong_Shadows_Can_Be_Dangerous_Stealthy_and_Effective_Physical-World_Adversarial_Attack_CVPR_2022_paper.html); [Singh&Hecke, 2021, PRL](https://journals.aps.org/prl/abstract/10.1103/PhysRevLett.126.248002); [Weiel, et al., 2021, Nature Mach. Intell](https://www.nature.com/articles/s42256-021-00366-3); [Wintermantel et al., 2020, PRL](https://journals.aps.org/prl/abstract/10.1103/PhysRevLett.124.070503); [Tang et al., 2019, TPAMI](https://ieeexplore.ieee.org/abstract/document/8386667); [Sheng et al., 2019, TPAMI](https://ieeexplore.ieee.org/abstract/document/8502935); [CMS Collaboration, 2019, JHEP](https://www.research-collection.ethz.ch/handle/20.500.11850/331761); [Wang et al., 2019, TVCG](https://ieeexplore.ieee.org/abstract/document/8826012); [Zhang et al., 2018, PRL](https://journals.aps.org/prl/abstract/10.1103/PhysRevLett.121.255703); [Leditzky et al., 2018, PRL](https://journals.aps.org/prl/abstract/10.1103/PhysRevLett.121.160501); [Pham et al., 2018, TPAMI](https://ieeexplore.ieee.org/abstract/document/8085141); [Villeneuve et al., 2017, Science](https://www.science.org/doi/10.1126/science.aam8393); [Choi et al., 2017, PRL](https://journals.aps.org/prl/abstract/10.1103/PhysRevLett.118.223605); [González-Echevarría, et al., 2017, TCAD](https://ieeexplore.ieee.org/document/7465733); [Zhu et al., 2017, PRL](https://journals.aps.org/prl/abstract/10.1103/PhysRevLett.119.106101); [Choi et al., 2017, ICCV](https://openaccess.thecvf.com/content_iccv_2017/html/Choi_Robust_Hand_Pose_ICCV_2017_paper.html); [Pickup et al., 2016, IJCV](https://link.springer.com/article/10.1007/s11263-016-0903-8); [Li et al., 2015, PRL](https://journals.aps.org/prl/abstract/10.1103/PhysRevLett.115.105502); [Sharp et al., 2015, CHI](https://dl.acm.org/doi/abs/10.1145/2702123.2702179); [Taneja et al., 2015, TPAMI](https://ieeexplore.ieee.org/abstract/document/7045528); [Zhang et al., 2015, IJCV](https://link.springer.com/article/10.1007/s11263-015-0819-8); [Meyer et al., 2015, ICCV](https://research.nvidia.com/publication/2015-12_robust-model-based-3d-head-pose-estimation); [Tompson et al., 2014, TOG](https://dl.acm.org/doi/abs/10.1145/2629500); [Baca et al., 2013, Cell](https://www.cell.com/cell/fulltext/S0092-8674(13)00343-7); [Li et al., PRL, 2013](https://journals.aps.org/prl/abstract/10.1103/PhysRevLett.110.136403); [Kawakami et al., 2013, IJCV](https://link.springer.com/article/10.1007/s11263-013-0632-1); [Kim et al., 2012, Nature](https://www.nature.com/articles/nature11546); [Rahmat-Samii et al., 2012, PIEEE](https://ieeexplore.ieee.org/document/6204306); [Oikonomidis et al., 2012, CVPR](https://ieeexplore.ieee.org/document/6247885); [Li et al., 2011, TPAMI](https://ieeexplore.ieee.org/abstract/document/5567109); [Zhao et al., 2011, PRL](https://journals.aps.org/prl/abstract/10.1103/PhysRevLett.107.215502); [Zhu et al., 2011, PRL](https://journals.aps.org/prl/abstract/10.1103/PhysRevLett.106.145501); [Lv et al., 2011, PRL](https://journals.aps.org/prl/abstract/10.1103/PhysRevLett.106.015503); [Hentschel&Sanders, 2010, PRL](https://journals.aps.org/prl/abstract/10.1103/PhysRevLett.104.063603); [Pontani&Conway, 2010, JGCD](https://arc.aiaa.org/doi/abs/10.2514/1.48475); [Zhang et al., 2008, CVPR](https://ieeexplore.ieee.org/document/4587512); [Liebelt&Schertler, 2007, CVPR](https://ieeexplore.ieee.org/abstract/document/4270192); [Hassan et al., 2005, AIAA](https://arc.aiaa.org/doi/abs/10.2514/6.2005-1897)].
* Cooperative Coevolution (CC)
  * Gomez, F., Schmidhuber, J. and Miikkulainen, R., 2008. [Accelerated neural evolution through cooperatively coevolved synapses](https://www.jmlr.org/papers/v9/gomez08a.html). Journal of Machine Learning Research, 9(31), pp.937-965.
  * Panait, L., Tuyls, K. and Luke, S., 2008. [Theoretical advantages of lenient learners: An evolutionary game theoretic perspective](https://jmlr.org/papers/volume9/panait08a/panait08a.pdf). Journal of Machine Learning Research, 9, pp.423-457.
  * Schmidhuber, J., Wierstra, D., Gagliolo, M. and Gomez, F., 2007. [Training recurrent networks by evolino](https://direct.mit.edu/neco/article-abstract/19/3/757/7156/Training-Recurrent-Networks-by-Evolino). Neural Computation, 19(3), pp.757-779.
  * Gomez, F.J. and Schmidhuber, J., 2005, June. [Co-evolving recurrent neurons learn deep memory POMDPs](https://dl.acm.org/doi/10.1145/1068009.1068092). In Proceedings of Annual Conference on Genetic and Evolutionary Computation (pp. 491-498).
  * Fan, J., Lau, R. and Miikkulainen, R., 2003. [Utilizing domain knowledge in neuroevolution](https://www.aaai.org/Library/ICML/2003/icml03-025.php). In International Conference on Machine Learning (pp. 170-177).
  * Potter, M.A. and De Jong, K.A., 2000. [Cooperative coevolution: An architecture for evolving coadapted subcomponents](https://direct.mit.edu/evco/article-abstract/8/1/1/859/Cooperative-Coevolution-An-Architecture-for). Evolutionary Computation, 8(1), pp.1-29.
  * Gomez, F.J. and Miikkulainen, R., 1999, July. [Solving non-Markovian control tasks with neuroevolution](https://www.ijcai.org/Proceedings/99-2/Papers/097.pdf). In Proceedings of International Joint Conference on Artificial Intelligence (pp. 1356-1361).
  * Moriarty, D.E. and Mikkulainen, R., 1996. [Efficient reinforcement learning through symbiotic evolution](https://link.springer.com/article/10.1023/A:1018004120707). Machine Learning, 22(1), pp.11-32.
  * Moriarty, D.E. and Miikkulainen, R., 1995. [Efficient learning from delayed rewards through symbiotic evolution](https://www.sciencedirect.com/science/article/pii/B9781558603776500566). In International Conference on Machine Learning (pp. 396-404). Morgan Kaufmann.
  * Potter, M.A. and De Jong, K.A., 1994, October. [A cooperative coevolutionary approach to function optimization](https://link.springer.com/chapter/10.1007/3-540-58484-6_269). In International Conference on Parallel Problem Solving from Nature (pp. 249-257). Springer, Berlin, Heidelberg.
* Simultaneous Perturbation Stochastic Approximation (SPSA) [ [https://www.jhuapl.edu/SPSA/](https://www.jhuapl.edu/SPSA/) ]
* Simulated Annealing (SA)
  * Bouttier, C. and Gavra, I., 2019. [Convergence rate of a simulated annealing algorithm with noisy observations](https://www.jmlr.org/papers/v20/16-588.html). Journal of Machine Learning Research, 20(1), pp.127-171.
  * Gerber, M. and Bornn, L., 2017. [Improving simulated annealing through derandomization](https://link.springer.com/article/10.1007/s10898-016-0461-1). Journal of Global Optimization, 68(1), pp.189-217.
  * Siarry, P., Berthiau, G., Durdin, F. and Haussy, J., 1997. [Enhanced simulated annealing for globally minimizing functions of many-continuous variables](https://dl.acm.org/doi/abs/10.1145/264029.264043). ACM Transactions on Mathematical Software, 23(2), pp.209-228.
  * Bertsimas, D. and Tsitsiklis, J., 1993. [Simulated annealing](https://tinyurl.com/yknunnpt). Statistical Science, 8(1), pp.10-15.
  * Corana, A., Marchesi, M., Martini, C. and Ridella, S., 1987. [Minimizing multimodal functions of continuous variables with the “simulated annealing” algorithm](https://dl.acm.org/doi/abs/10.1145/29380.29864). ACM Transactions on Mathematical Software, 13(3), pp.262-280. [ [Corrigenda](https://dl.acm.org/doi/10.1145/66888.356281) ]
  * Kirkpatrick, S., Gelatt, C.D. and Vecchi, M.P., 1983. [Optimization by simulated annealing](https://science.sciencemag.org/content/220/4598/671). Science, 220(4598), pp.671-680.
  * Hastings, W.K., 1970. [Monte Carlo sampling methods using Markov chains and their applications](https://academic.oup.com/biomet/article/57/1/97/284580). Biometrika, 57(1), pp.97-109.
  * Metropolis, N., Rosenbluth, A.W., Rosenbluth, M.N., Teller, A.H. and Teller, E., 1953. [Equation of state calculations by fast computing machines](https://aip.scitation.org/doi/abs/10.1063/1.1699114). Journal of Chemical Physics, 21(6), pp.1087-1092.
  * Applications: e.g., [Young et al., 2023, Nature](https://www.nature.com/articles/s41586-023-05823-0); [Kim et al., 2023, Nature](https://www.nature.com/articles/s41586-023-06123-3); [Passalacqua et al., 2023, Nature](https://www.nature.com/articles/s41586-023-06229-8); [Pronker et al., 2023, Nature](https://www.nature.com/articles/s41586-023-06599-z); [Sullivan&Seljak, 2023](https://arxiv.org/pdf/2310.00745.pdf); [Holm et al., 2023, Nature](https://www.nature.com/articles/s41586-023-05908-w); [Snyder et al., 2023, Nature](https://www.nature.com/articles/s41586-022-05409-2); [Samyak&Palacios, 2023, Biometrika](https://academic.oup.com/biomet/advance-article/doi/10.1093/biomet/asad025/7143381); [Bouchet et al., 2023, PNAS](https://www.pnas.org/doi/abs/10.1073/pnas.2221407120); [Li&Yu, 2023, ACM-TOG](https://dl.acm.org/doi/10.1145/3592096); [Zhao et al., 2023, VLDBJ](https://link.springer.com/article/10.1007/s00778-023-00802-3); [Zhong et al., 2023, IEEE/ACM-TASLP](https://ieeexplore.ieee.org/abstract/document/10214657); [Wang et al., 2023, IEEE-TMC](https://ieeexplore.ieee.org/abstract/document/10011565); [Filippo et al., 2023, IJCAI](https://www.ijcai.org/proceedings/2023/0644.pdf); [Barnes et al., 2023, ApJ](https://iopscience.iop.org/article/10.3847/1538-4357/acba8e); [Melo et al., 2023](https://www.biorxiv.org/content/10.1101/2023.05.31.542906v1.full.pdf); [Bruna et al., 2023](https://www.biorxiv.org/content/biorxiv/early/2023/01/15/2023.01.13.524024.full.pdf); [Holm et al., 2023](https://arxiv.org/pdf/2309.04468.pdf); [Jenson et al., 2023, Nature](https://www.nature.com/articles/s41586-023-05862-7); [Kolesov et al., 2016, IEEE-TPAMI](https://ieeexplore.ieee.org/document/7130637)
* Genetic Algorithm (GA)
  * Whitley, D., 2019. [Next generation genetic algorithms: A user’s guide and tutorial](https://link.springer.com/chapter/10.1007/978-3-319-91086-4_8). In Handbook of Metaheuristics (pp. 245-274). Springer, Cham.
  * Levine, D., 1997. [Commentary—Genetic algorithms: A practitioner's view](https://pubsonline.informs.org/doi/abs/10.1287/ijoc.9.3.256). INFORMS Journal on Computing, 9(3), pp.256-259.
  * Goldberg, D.E., 1994. [Genetic and evolutionary algorithms come of age](https://dl.acm.org/doi/10.1145/175247.175259). Communications of the ACM, 37(3), pp.113-120.
  * Forrest, S., 1993. [Genetic algorithms: Principles of natural selection applied to computation](https://www.science.org/doi/10.1126/science.8346439). Science, 261(5123), pp.872-878.
  * De Jong, K.A., 1993. [Are genetic algorithms function optimizer?](https://www.sciencedirect.com/science/article/pii/B9780080948324500064). Foundations of Genetic Algorithms, pp.5-17.
  * Mitchell, M., Holland, J. and Forrest, S., 1993. [When will a genetic algorithm outperform hill climbing](https://proceedings.neurips.cc/paper/1993/hash/ab88b15733f543179858600245108dd8-Abstract.html). Advances in Neural Information Processing Systems (pp. 51-58).
  * Holland, J.H., 1992. [Adaptation in natural and artificial systems: An introductory analysis with applications to biology, control, and artificial intelligence](https://direct.mit.edu/books/book/2574/Adaptation-in-Natural-and-Artificial-SystemsAn). MIT Press.
  * Holland, J.H., 1992. [Genetic algorithms](https://www.scientificamerican.com/article/genetic-algorithms/). Scientific American, 267(1), pp.66-73.
  * Whitley, D., 1989, December. [The GENITOR algorithm and selection pressure: Why rank-based allocation of reproductive trials is best](https://dl.acm.org/doi/10.5555/93126.93169). In Proceedings of International Conference on Genetic Algorithms (pp. 116-121).
  * Goldberg, D.E. and Holland, J.H., 1988. [Genetic algorithms and machine learning](https://link.springer.com/article/10.1023/A:1022602019183). Machine Learning, 3(2), pp.95-99.
  * Holland, J.H., 1973. [Genetic algorithms and the optimal allocation of trials](https://epubs.siam.org/doi/10.1137/0202009). SIAM Journal on Computing, 2(2), pp.88-105.
  * Holland, J.H., 1962. [Outline for a logical theory of adaptive systems](https://dl.acm.org/doi/10.1145/321127.321128). Journal of the ACM, 9(3), pp.297-314.
  * Applications: e.g., [Wang, 2023, Harvard Ph.D. Dissertation](https://dash.harvard.edu/bitstream/handle/1/37374599/dissertation.pdf); [Lee et al., 2022, Science Robotics](https://www.science.org/doi/10.1126/scirobotics.abq7278); [Whitelam&Tamblyn, 2021, PRL](https://journals.aps.org/prl/abstract/10.1103/PhysRevLett.127.018003); [Walker et al., 2021, Nature Chemistry](https://www.nature.com/articles/s41557-020-00626-6); [Chen et al., 2020, Nature](https://www.nature.com/articles/s41586-019-1901-0); [Whitley et al., 1993, MLJ](https://link.springer.com/article/10.1007/BF00993045).
* Evolutionary Programming (EP)
  * Yao, X., Liu, Y. and Lin, G., 1999. [Evolutionary programming made faster](https://ieeexplore.ieee.org/abstract/document/771163). IEEE Transactions on Evolutionary Computation, 3(2), pp.82-102.
  * Fogel, D.B., 1999. [An overview of evolutionary programming](https://link.springer.com/chapter/10.1007/978-1-4612-1542-4_5). In Evolutionary Algorithms (pp. 89-109). Springer, New York, NY.
  * Fogel, D.B. and Fogel, L.J., 1995, September. [An introduction to evolutionary programming](https://link.springer.com/chapter/10.1007/3-540-61108-8_28). In European Conference on Artificial Evolution (pp. 21-33). Springer, Berlin, Heidelberg.
  * Fogel, D.B., 1994. [Evolutionary programming: An introduction and some current directions](https://link.springer.com/article/10.1007/BF00175356). Statistics and Computing, 4(2), pp.113-129.
  * Bäck, T. and Schwefel, H.P., 1993. [An overview of evolutionary algorithms for parameter optimization](https://direct.mit.edu/evco/article-abstract/1/1/1/1092/An-Overview-of-Evolutionary-Algorithms-for). Evolutionary Computation, 1(1), pp.1-23.
  * Applications: e.g., [Hoorfar, 2007, IEEE-TAP](https://ieeexplore.ieee.org/document/4120264); [Cui et al., 2006, MS](https://pubsonline.informs.org/doi/abs/10.1287/mnsc.1060.0514); [Damavandi&Safavi-Naeini, 2005, IEEE-TCSI](https://ieeexplore.ieee.org/document/1427899).
* Pattern Search
  * Audet, C., Le Digabel, S., Rochon Montplaisir, V. and Tribes, C., 2022. [Algorithm XXXX: NOMAD version 4: Nonlinear optimization with the MADS algorithm](https://dl.acm.org/doi/abs/10.1145/3544489). ACM Transactions on Mathematical Software.
  * Brent, R.P., 2013. [Algorithms for minimization without derivatives](https://maths-people.anu.edu.au/~brent/pub/pub011.html). Courier Corporation.
  * Singer, S. and Nelder, J., 2009. [Nelder-mead algorithm](http://var.scholarpedia.org/article/Nelder-Mead_algorithm). Scholarpedia, 4(7), p.2928.
  * Kolda, T.G., Lewis, R.M. and Torczon, V., 2003. [Optimization by direct search: New perspectives on some classical and modern methods](https://epubs.siam.org/doi/abs/10.1137/S003614450242889). SIAM Review, 45(3), pp.385-482.
  * Lagarias, J.C., Reeds, J.A., Wright, M.H. and Wright, P.E., 1998. [Convergence properties of the Nelder--Mead simplex method in low dimensions](https://epubs.siam.org/doi/abs/10.1137/S1052623496303470). SIAM Journal on Optimization, 9(1), pp.112-147.
  * Powell, M.J., 1998. [Direct search algorithms for optimization calculations](https://www.cambridge.org/core/journals/acta-numerica/article/abs/direct-search-algorithms-for-optimization-calculations/23FA5B19EAF122E02D3724DB1841238C). Acta Numerica, 7, pp.287-336.
  * Torczon, V., 1997. [On the convergence of pattern search algorithms](https://epubs.siam.org/doi/abs/10.1137/S1052623493250780). SIAM Journal on Optimization, 7(1), pp.1-25.
  * Barton, R.R. and Ivey Jr, J.S., 1996. [Nelder-Mead simplex modifications for simulation optimization](https://pubsonline.informs.org/doi/abs/10.1287/mnsc.42.7.954). Management Science, 42(7), pp.954-973.
  * Wright, M.H., 1996. [Direct search methods: Once scorned, now respectable](https://nyuscholars.nyu.edu/en/publications/direct-search-methods-once-scorned-now-respectable). Pitman Research Notes in Mathematics Series, pp.191-208.
  * Jones, D.R., Perttunen, C.D. and Stuckman, B.E., 1993. [Lipschitzian optimization without the Lipschitz constant](https://link.springer.com/article/10.1007/BF00941892). Journal of Optimization Theory and Applications, 79(1), pp.157-181.
  * Nelder, J.A. and Mead, R., 1965. [A simplex method for function minimization](https://academic.oup.com/comjnl/article-abstract/7/4/308/354237). The Computer Journal, 7(4), pp.308-313.
  * Powell, M.J., 1964. [An efficient method for finding the minimum of a function of several variables without calculating derivatives](https://academic.oup.com/comjnl/article-abstract/7/2/155/335330). The Computer Journal, 7(2), pp.155-162.
  * Kaupe Jr, A.F., 1963. [Algorithm 178: Direct search](https://dl.acm.org/doi/pdf/10.1145/366604.366632). Communications of the ACM, 6(6), pp.313-314.
  * Spang, III, H.A., 1962. [A review of minimization techniques for nonlinear functions](https://epubs.siam.org/doi/abs/10.1137/1004089). SIAM Review, 4(4), pp.343-365.
  * Hooke, R. and Jeeves, T.A., 1961. [“Direct search” solution of numerical and statistical problems](https://dl.acm.org/doi/10.1145/321062.321069). Journal of the ACM, 8(2), pp.212-229. [ [Python - pymoo](https://pymoo.org/algorithms/soo/pattern.html) | [This Week's Citation Classic](http://garfield.library.upenn.edu/classics1980/A1980JK10100001.pdf) ]
  * Box, G.E., 1957. [Evolutionary operation: A method for increasing industrial productivity](https://rss.onlinelibrary.wiley.com/doi/abs/10.2307/2985505). Journal of the Royal Statistical Society: Series C (Applied Statistics), 6(2), pp.81-101.
  * Fermi, E. and Metropolis N., 1952. [Numerical solution of a minimum problem](https://www.osti.gov/servlets/purl/4377177). Los Alamos Scientific Lab., Los Alamos, NM.
    * Anderson, H.L., Davidon, W.C., Glicksman, M. and Kruse, U.E., 1955. [Scattering of positive pions by hydrogen at 189 MeV](https://journals.aps.org/pr/abstract/10.1103/PhysRev.100.279). Physical Review, 100(1), p.279.
  * Applications: e.g., [NM: [Gokhale et al., 2023, PNAS](https://www.pnas.org/doi/abs/10.1073/pnas.2207595120); [Hayashi, 2022, Bernoulli](https://doi.org/10.3150/21-BEJ1344); [Vanunu et al., 2021, PNAS](https://www.pnas.org/doi/abs/10.1073/pnas.2025646118); [Williams et al., 2021, PNAS](https://www.pnas.org/doi/abs/10.1073/pnas.2106237118); [Fleishman et al., Science, 2020](https://www.science.org/doi/abs/10.1126/science.aax6874); [Nanni et al., 2020, PNAS](https://www.pnas.org/doi/abs/10.1073/pnas.2023757118); [Steinrücken et al., 2019, PNAS](https://www.pnas.org/doi/abs/10.1073/pnas.1905060116); [Omran et al., 2019, Science](https://www.science.org/doi/10.1126/science.aax9743); [Sparrow et al., 2018, Nature](https://www.nature.com/articles/s41586-018-0152-9); [Prochazka&Vogl, 2017, PNAS](https://www.pnas.org/doi/abs/10.1073/pnas.1617252114); [Murphy&Brincke, 2017, MS](https://pubsonline.informs.org/doi/abs/10.1287/mnsc.2016.2591); [Gillon et al., 2017, Nature](https://www.nature.com/articles/nature21360); [Aghaeepour et al., 2017, Science Immunol.](https://www.science.org/doi/full/10.1126/sciimmunol.aan2946); [Sayegh et al., 2017, TS](https://pubsonline.informs.org/doi/10.1287/trsc.2017.0787); [Landis&Schraiber, 2017, PNAS](https://www.pnas.org/doi/abs/10.1073/pnas.1710920114); [Kim et al., 2016, PNAS](https://www.pnas.org/doi/abs/10.1073/pnas.1618883114); [Chan et al., 2014, MS](https://pubsonline.informs.org/doi/abs/10.1287/mnsc.2013.1840); [Chan et al., 2014, MKSC](https://pubsonline.informs.org/doi/abs/10.1287/mksc.2013.0831); [Bajikar et al., 2014, PNAS](https://www.pnas.org/doi/abs/10.1073/pnas.1311647111); [Lee et al., 2014, PNAS](https://www.pnas.org/doi/abs/10.1073/pnas.1313039111); [Wang et al., 2012, PRL](https://journals.aps.org/prl/abstract/10.1103/PhysRevLett.109.175502); [Lau&Rubinstein, Nature, 2012](https://www.nature.com/articles/nature10699); [Brown et al., 2012, MS](https://pubsonline.informs.org/doi/abs/10.1287/mnsc.1120.1563); [Contreras et al., 2012, PNAS](https://www.pnas.org/doi/abs/10.1073/pnas.1216953109); [Morlon et al., 2011, PNAS](https://www.pnas.org/doi/abs/10.1073/pnas.1102543108); [Forstmann et al., 2010, PNAS](https://www.pnas.org/doi/abs/10.1073/pnas.1004932107); [Balachander et al., 2009, MS](https://pubsonline.informs.org/doi/abs/10.1287/mnsc.1090.1056); [Jayanthi et al., 2009, MS](https://pubsonline.informs.org/doi/abs/10.1287/mnsc.1090.1002); [Farrell et al., 2009, PNAS](https://www.pnas.org/doi/abs/10.1073/pnas.0906845107); [Forstmann et al., 2008, PNAS](https://www.pnas.org/doi/abs/10.1073/pnas.0805903105); [Rouder et al., 2008, PNAS](https://www.pnas.org/doi/abs/10.1073/pnas.0711295105); [Sapir et al., 2005, PNAS](https://www.pnas.org/doi/abs/10.1073/pnas.0504678102); [Amonlirdviman et al., 2005, Science](https://www.science.org/doi/abs/10.1126/science.1105471); [Cowan et al., 2004, PS](https://journals.sagepub.com/doi/abs/10.1111/j.0956-7976.2004.00732.x); [Zhou et al., 2004, PS](https://journals.sagepub.com/doi/abs/10.1111/j.0963-7214.2004.01502007.x); [Draganska&Jain, 2004, MS](https://pubsonline.informs.org/doi/abs/10.1287/mnsc.1040.0227); [Fain&Levitt, 2003, PNAS](https://www.pnas.org/doi/abs/10.1073/pnas.1732312100); [Dennis et al., 2002, PNAS](https://www.pnas.org/doi/abs/10.1073/pnas.062398499); [Sudhir, 2001, MKSC](https://pubsonline.informs.org/doi/abs/10.1287/mksc.20.1.42.10196); [Rouder, 2001, PS](https://journals.sagepub.com/doi/abs/10.1111/1467-9280.00358); [Wolszczan, 1994, Science](https://www.science.org/doi/abs/10.1126/science.264.5158.538); [Polvani et al., 1990, Science](https://www.science.org/doi/abs/10.1126/science.249.4975.1393); [Lee et al., 1987, PNAS](https://www.pnas.org/doi/abs/10.1073/pnas.84.9.2590); [Sabath et al., 1986, PNAS](https://www.pnas.org/doi/abs/10.1073/pnas.83.13.4739); [Burch et al., 1985, PNAS](https://www.pnas.org/doi/abs/10.1073/pnas.82.21.7434); [Regoeczi et al., 1982, PNAS](https://www.pnas.org/doi/abs/10.1073/pnas.79.7.2226); [Brasseur et al., 1982, PNAS](); [Korn et al., 1981, Science](https://www.science.org/doi/abs/10.1126/science.6266015); [Dean et al., 1975, Science](https://www.science.org/doi/10.1126/science.189.4205.805)]; [HJ: [Kalita et al., 2023, Nature](https://www.nature.com/articles/s41586-023-06283-2); [Washington et al, 2023](https://arxiv.org/pdf/2301.07822.pdf); [Huynh et al., 2023](https://arxiv.org/ftp/arxiv/papers/2304/2304.05603.pdf); [Kucher et al., 2022, SPLC](https://dl.acm.org/doi/abs/10.1145/3503229.3547041); [Pepe et al., 2021, IEEE-TASLP](https://ieeexplore.ieee.org/abstract/document/9729497); [Khaledian et al., 2018, IEEE-TMTT](https://ieeexplore.ieee.org/abstract/document/8335770); [Luhar et al., 2015, JFM](https://www.cambridge.org/core/journals/journal-of-fluid-mechanics/article/framework-for-studying-the-effect-of-compliant-surfaces-on-wall-turbulence/47B8C79442109015690B6B16F891DF09); [Paxton et al., 2013, ApJS](https://iopscience.iop.org/article/10.1088/0067-0049/208/1/4); [Schneider&Excoffier, 1999, Genetics](https://academic.oup.com/genetics/article/152/3/1079/6034947); [Ditchfield et al., 1971, JCP](https://aip.scitation.org/doi/10.1063/1.1674902)].
* Random Search (RS)
  * Bergstra, J. and Bengio, Y., 2012. [Random search for hyper-parameter optimization](https://www.jmlr.org/papers/v13/bergstra12a.html). Journal of Machine Learning Research, 13(2).
  * Nemirovski, A., Juditsky, A., Lan, G. and Shapiro, A., 2009. [Robust stochastic approximation approach to stochastic programming](https://epubs.siam.org/doi/abs/10.1137/070704277). SIAM Journal on Optimization, 19(4), pp.1574-1609.
  * Schmidhuber, J., Hochreiter, S. and Bengio, Y., 2001. [Evaluating benchmark problems by random guessing](https://ml.jku.at/publications/older/ch9.pdf). A Field Guide to Dynamical Recurrent Networks, pp.231-235.
  * Rosenstein, M.T. and Barto, A.G., 2001, August. [Robot weightlifting by direct policy search](https://dl.acm.org/doi/abs/10.5555/1642194.1642206). In International Joint Conference on Artificial Intelligence (pp. 839-846).
  * Cvijović, D. and Klinowski, J., 1995. [Taboo search: An approach to the multiple minima problem](https://www.science.org/doi/abs/10.1126/science.267.5198.664). Science, 267(5198), pp.664-666.
  * Polyak, B.T., 1987. Introduction to optimization. Optimization Software. New York.
  * Dorea, C.C.Y., 1983. [Expected number of steps of a random optimization method](https://link.springer.com/article/10.1007/BF00934526). Journal of Optimization Theory and Applications, 39(2), pp.165-171. [ Sarma, M.S., 1990. [On the convergence of the Baba and Dorea random optimization methods](https://link.springer.com/article/10.1007/BF00939542). Journal of Optimization Theory and Applications, 66, pp.337-343. ] + [ Appel, M.J., Labarre, R. and Radulovic, D., 2004. [On accelerated random search](https://epubs.siam.org/doi/abs/10.1137/S105262340240063X). SIAM Journal on Optimization, 14(3), pp.708-731. ]
  * Solis, F.J. and Wets, R.J.B., 1981. [Minimization by random search techniques](https://pubsonline.informs.org/doi/abs/10.1287/moor.6.1.19). Mathematics of Operations Research, 6(1), pp.19-30.
  * Schumer, M.A. and Steiglitz, K., 1968. [Adaptive step size random search](https://ieeexplore.ieee.org/abstract/document/1098903). IEEE Transactions on Automatic Control, 13(3), pp.270-276.
  * Rastrigin, L.A., 1963. [The convergence of the random search method in the extremal control of a many parameter system](https://tinyurl.com/djfdnpx4). Automaton & Remote Control, 24, pp.1337-1342. [ Rastrigin, L.A., 1986. [Random search as a method for optimization and adaptation](https://link.springer.com/chapter/10.1007/BFb0007129). In Stochastic Optimization. ]
  * Brooks, S.H., 1958. [A discussion of random methods for seeking maxima](https://pubsonline.informs.org/doi/abs/10.1287/opre.6.2.244). Operations Research, 6(2), pp.244-251.
* Bayesian Optimization (BO)
  * [https://bayesoptbook.com/](https://bayesoptbook.com/)
  * [https://bayesopt-tutorial.github.io/](https://bayesopt-tutorial.github.io/)
  * Wang, L., Fonseca, R. and Tian, Y., 2020. [Learning search space partition for black-box optimization using monte carlo tree search](https://proceedings.neurips.cc/paper/2020/hash/e2ce14e81dba66dbff9cbc35ecfdb704-Abstract.html). Advances in Neural Information Processing Systems, 33, pp.19511-19522. [ [Python](https://github.com/facebookresearch/LaMCTS) ]
  * Jones, D.R., Schonlau, M. and Welch, W.J., 1998. [Efficient global optimization of expensive black-box functions](https://link.springer.com/article/10.1023/A:1008306431147). Journal of Global Optimization, 13(4), pp.455-492.
* Automated Machine Learning (AutoML)
  * Hutter, F., Kotthoff, L. and Vanschoren, J., 2019. [Automated machine learning: Methods, systems, challenges](https://www.automl.org/wp-content/uploads/2019/05/AutoML_Book.pdf). Springer Nature.
  * Hoos, H.H., 2011. [Automated algorithm configuration and parameter tuning](https://link.springer.com/chapter/10.1007/978-3-642-21434-9_3). In Autonomous Search (pp. 37-71). Springer, Berlin, Heidelberg.
* Software
  * Custódio, A.L., Scheinberg, K. and Nunes Vicente, L., 2017. [Methodologies and software for derivative-free optimization](https://epubs.siam.org/doi/abs/10.1137/1.9781611974683.ch37). Advances and Trends in Optimization with Engineering Applications, pp.495-506.
  * Rios, L.M. and Sahinidis, N.V., 2013. [Derivative-free optimization: A review of algorithms and comparison of software implementations](https://link.springer.com/article/10.1007/s10898-012-9951-y). Journal of Global Optimization, 56, pp.1247-1293.
  * Schaul, T., Bayer, J., Wierstra, D., Sun, Y., Felder, M., Sehnke, F., Rückstieß, T. and Schmidhuber, J., 2010. [PyBrain](https://jmlr.org/papers/v11/schaul10a.html). Journal of Machine Learning Research, 11(24), pp.743-746.
  * Press, W.H., Teukolsky, S.A., Vetterling, W.T. and Flannery, B.P., 2007. [Numerical recipes: The art of scientific computing](http://numerical.recipes/). Cambridge University Press. (See Chapter 10. Minimization or maximization of functions.)

## Sponsor for PyPop7

This open-source Python library for black-box optimization is now supported by Shenzhen Fundamental Research Program under Grant No. JCYJ20200109141235597 (￥2,000,000 from 2021 to 2023), granted to Prof. Shi (CSE, SUSTech @ Shenzhen, China), and actively developed by three of his group members (e.g., *Q.Q. Duan*, *C. Shao*, *G.C. Zhou*).

## Citation for PyPop7

If this open-source pure-Python library is used in your paper or project, it is highly welcomed to cite the following arXiv [preprint](https://arxiv.org/abs/2212.05652) paper:

**Duan, Q., Zhou, G., Shao, C., Wang, Z., Feng, M., Yang, Y., Zhao, Q. and Shi, Y., 2022. PyPop7: A pure-Python library for population-based black-box optimization. arXiv preprint arXiv:2212.05652.**
