"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SlackNotifier = void 0;
const path = require("path");
const aws_events_1 = require("@aws-cdk/aws-events");
const aws_events_targets_1 = require("@aws-cdk/aws-events-targets");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const core_1 = require("@aws-cdk/core");
const channel_types_1 = require("./channel-types");
class SlackNotifier extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const { slackBotToken, slackSigningSecret, slackChannel, slackChannelId, slackChannelTypes, slackBotName, slackBotIcon, pipeline, stageNames, } = props;
        this.environment = {
            SLACK_BOT_TOKEN: slackBotToken,
            SLACK_SIGNING_SECRET: slackSigningSecret,
            SLACK_CHANNEL: slackChannel || '',
            SLACK_CHANNEL_ID: slackChannelId || '',
            SLACK_CHANNEL_TYPES: (slackChannelTypes || [channel_types_1.ChannelTypes.PUBLIC]).join(','),
        };
        if (slackBotName) {
            this.environment.SLACK_BOT_NAME = slackBotName;
        }
        if (slackBotIcon) {
            this.environment.SLACK_BOT_ICON = slackBotIcon;
        }
        const notifier = new aws_lambda_1.Function(scope, 'SlackNotifierFunction', {
            runtime: aws_lambda_1.Runtime.NODEJS_12_X,
            handler: 'index.handler',
            code: aws_lambda_1.Code.asset(path.join(__dirname, 'lambdas', 'notifier')),
            environment: this.environment,
        });
        notifier.addToRolePolicy(new aws_iam_1.PolicyStatement({
            resources: [pipeline.pipelineArn],
            actions: ['codepipeline:GetPipelineState', 'codepipeline:GetPipelineExecution'],
        }));
        pipeline.onStateChange('SlackPipelineNotifierRule', {
            target: new aws_events_targets_1.LambdaFunction(notifier),
        });
        const stageRule = new aws_events_1.Rule(this, 'SlackStageNotifierRule');
        stageRule.addTarget(new aws_events_targets_1.LambdaFunction(notifier));
        stageRule.addEventPattern({
            source: ['aws.codepipeline'],
            resources: [pipeline.pipelineArn],
            detailType: ['CodePipeline Stage Execution State Change'],
        });
        if (stageNames) {
            stageRule.addEventPattern({
                detail: {
                    stage: stageNames,
                },
            });
        }
    }
    /**
     * Validate the current construct.
     *
     * This method can be implemented by derived constructs in order to perform
     * validation logic. It is called on all constructs before synthesis.
     */
    validate() {
        if (this.environment.SLACK_CHANNEL && this.environment.SLACK_CHANNEL_ID) {
            return ['Redundant Configuration: Please configure slackChannel by id (prop slackChannelId) OR name (prop slackChannel)'];
        }
        if (!this.environment.SLACK_CHANNEL && !this.environment.SLACK_CHANNEL_ID) {
            return ['Missing Configuration: Please configure slackChannel by id (prop slackChannelId) or name (prop slackChannel)'];
        }
        return [];
    }
}
exports.SlackNotifier = SlackNotifier;
//# sourceMappingURL=data:application/json;base64,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