from typing import Optional

from highcharts_core.decorators import class_sensitive

from highcharts_stock.options.plot_options.indicators import ParameterBase, IndicatorOptions, ComparableIndicatorOptions


class WilliamsRParameters(ParameterBase):

    @property
    def index(self):
        """Does not apply, so raises an :exc:`AttributeError <python:AttributeError>`."""
        raise AttributeError(f"{self.__class__.__name__} has no attribute 'index'")

    @index.setter
    def index(self, value):
        raise AttributeError(f"{self.__class__.__name__} has no attribute 'index'")

    @classmethod
    def _get_kwargs_from_dict(cls, as_dict):
        kwargs = {
            'period': as_dict.get('period', None),
        }

        return kwargs

    def _to_untrimmed_dict(self, in_cls = None) -> dict:
        untrimmed = {
            'period': self.period,
        }

        return untrimmed


class AroonOscillatorOptions(IndicatorOptions):
    """Options to configure an Aroon Oscillator series, which is an :term:`oscillator`
    that is used to gauge the strength of a current trend and the likelihood that it will
    continue.

    .. figure:: ../../../../_static/aroon-oscillator-example.png
      :alt: Aroon Oscillator Example Chart
      :align: center

    """
    pass


class APOOptions(ComparableIndicatorOptions):
    """Options to configure a Absolute Price
    :term:`Oscillator <oscillator>` (APO), which is an :term:`oscillator` that displays
    the difference between two exponential moving averages of an asset's price expressed
    as an absolute value

    .. figure:: ../../../../_static/apo-example.png
      :alt: Absolute Price Oscillator Example Chart
      :align: center

    """
    pass


class CCIOptions(ComparableIndicatorOptions):
    """Options to configure a Commodity Channel Index
    :term:`indicator <technical indicator>`, which is an :term:`oscillator` that measures
    the difference between the current price and the historical average price using a
    look-back
    period determined by the configuration of
    :meth:`.params <highcharts_stock.options.plot_options.atr.CCIOptions.params`.

    .. figure:: ../../../../_static/cci-example.png
      :alt: CCI Example Chart
      :align: center

    """
    pass


class ChaikinOptions(ComparableIndicatorOptions):
    """Options to configure a Chaikin :term:`oscillator`, which measures the
    accummulation-distribution line of moving average convergence-divergence by
    subtracting a 10-day exponential moving average from a 3-day moving average of the
    accumulation-distribution line.

    .. figure:: ../../../../_static/chaikin-example.png
      :alt: Chaikin Oscillator Example Chart
      :align: center

    """
    pass


class CMOOptions(ComparableIndicatorOptions):
    """Options to configure a Chande Momentum :term:`Oscillator`, which uses momentum to
    identify relative strength or weakness in a market.

    .. caution::

      The chosen time frame - configured by
      :meth:`.params.period <highcharts_stock.options.plot_options.atr.CCIOptions.params`
      and defaulting to a value of ``20`` - has a significant impact on the signals
      generated by the indicator.

    .. figure:: ../../../../_static/cmo-example.png
      :alt: Chande Momentum Oscillator Example Chart
      :align: center

    """
    pass


class DPOOptions(IndicatorOptions):
    """Options to configure a Detrended Price :term:`Oscillator`, which
    strips out price trends in an effort to estimate the length of price cycles.

    .. figure:: ../../../../_static/dpo-example.png
      :alt: DPO Example Chart
      :align: center

    """
    pass


class TRIXOptions(IndicatorOptions):
    """Options to configure a Triple Exponential Average :term:`Oscillator`, which
    can be used to suggest increasing or decreasing momentum.

    .. figure:: ../../../../_static/trix-example.png
      :alt: TRIX Example Chart
      :align: center

    """
    pass


class WilliamsROptions(ComparableIndicatorOptions):
    """Options to configure a Williams %R :term:`oscillator`, which uses the current
    closing price in relation to the high and low of the past ``n`` days to indicate
    overbought and oversold levels.

    .. figure:: ../../../../_static/williamsr-example.png
      :alt: Williams %R Example Chart
      :align: center

    """
    @property
    def params(self) -> Optional[WilliamsRParameters]:
        """Parameters used in calculating the indicator's data points.

        :rtype: :class:`WilliamsRParameters` or :obj:`None <python:None>`
        """
        return self._params

    @params.setter
    @class_sensitive(WilliamsRParameters)
    def params(self, value):
        self._params = value
