from typing import Optional, List

from validator_collection import validators

from highcharts_core.decorators import class_sensitive
from highcharts_core.metaclasses import HighchartsMeta

from highcharts_stock import errors
from highcharts_stock.options.plot_options.indicators import ParameterBase, ComparableIndicatorOptions
from highcharts_stock.utility_classes.line_styles import LineStylesColorWidth


class StochasticParameters(ParameterBase):

    def __init__(self, **kwargs):
        self._periods = None

        self.periods = kwargs.get('periods', None)

        super().__init__(**kwargs)

    @property
    def index(self):
        """Does not apply, so raises an :exc:`AttributeError <python:AttributeError>`."""
        raise AttributeError(f"{self.__class__.__name__} has no attribute 'index'")

    @index.setter
    def index(self, value):
        raise AttributeError(f"{self.__class__.__name__} has no attribute 'index'")

    @property
    def period(self):
        """Does not apply, so raises an :exc:`AttributeError <python:AttributeError>`."""
        raise AttributeError(f"{self.__class__.__name__} has no attribute 'period'")

    @period.setter
    def period(self, value):
        raise AttributeError(f"{self.__class__.__name__} has no attribute 'period'")

    @property
    def periods(self) -> Optional[List[int]]:
        """The periods to use when calculating the Stochastic Oscillator. Defaults
        to ``[14, 3]``.

        .. note::

          Expects a two-member iterable of :class:`int <python:int>` instances.

        :rtype: :class:`list <python:list>` of :class:`int <python:int>`, or
          :obj:`None <python:None>`
        """
        return self._periods

    @periods.setter
    def periods(self, value):
        if not value:
            self._periods = None
        else:
            value = validators.iterable(value)
            if len(value) != 2:
                raise errors.HighchartsValueError(f'periods expects a 2-member iterable. '
                                                  f'Received a {len(value)}-member '
                                                  f'iterable.')
            self._periods = [validators.integer(x, minimum = 0) for x in value]

    @classmethod
    def _get_kwargs_from_dict(cls, as_dict):
        kwargs = {
            'periods': as_dict.get('periods', None),
        }

        return kwargs

    def _to_untrimmed_dict(self, in_cls = None) -> dict:
        untrimmed = {
            'periods': self.periods,
        }

        return untrimmed


class SlowStochasticParameters(StochasticParameters):

    def __init__(self, **kwargs):
        super().__init__(**kwargs)

    @property
    def periods(self) -> Optional[List[int]]:
        """The periods to use when calculating the Slow Stochastic Oscillator. Defaults
        to ``[14, 3, 3]``.

        .. note::

          Expects a three-member iterable of :class:`int <python:int>` instances.

        :rtype: :class:`list <python:list>` of :class:`int <python:int>`, or
          :obj:`None <python:None>`
        """
        return self._periods

    @periods.setter
    def periods(self, value):
        if not value:
            self._periods = None
        else:
            value = validators.iterable(value)
            if len(value) != 3:
                raise errors.HighchartsValueError(f'periods expects a 3-member iterable. '
                                                  f'Received a {len(value)}-member '
                                                  f'iterable.')
            self._periods = [validators.integer(x, minimum = 0) for x in value]

    @classmethod
    def _get_kwargs_from_dict(cls, as_dict):
        kwargs = {
            'periods': as_dict.get('periods', None),
        }

        return kwargs

    def _to_untrimmed_dict(self, in_cls = None) -> dict:
        untrimmed = {
            'periods': self.periods,
        }

        return untrimmed


class StochasticStyleOptions(HighchartsMeta):
    """Styles for the Stochastic lines."""

    def __init__(self, **kwargs):
        self._styles = None

        self.styles = kwargs.get('styles', None)

    @property
    def styles(self) -> Optional[LineStylesColorWidth]:
        """Styles for the line.

        :rtype: :class:`LineStylesColorWidth` or :obj:`None <python:None>`
        """
        return self._styles

    @styles.setter
    @class_sensitive(LineStylesColorWidth)
    def styles(self, value):
        self._styles = value

    @classmethod
    def _get_kwargs_from_dict(cls, as_dict):
        kwargs = {
            'styles': as_dict.get('styles', None)
        }

        return kwargs

    def _to_untrimmed_dict(self, in_cls = None) -> dict:
        untrimmed = {
            'styles': self.styles
        }

        return untrimmed


class StochasticOptions(ComparableIndicatorOptions):
    """Configuration for a Stochastic :term:`Oscillator`.

    .. figure:: ../../../../_static/stochastic-example.png
      :alt: Stochsatic Oscillator Example Chart
      :align: center

    """

    def __init__(self, **kwargs):
        self._smoothed_line = None

        self.smoothed_line = kwargs.get('smoothed_line', None)

        super().__init__(**kwargs)

    @property
    def smoothed_line(self) -> Optional[StochasticStyleOptions]:
        """Styles for the smoothed line.

        :rtype: :class:`StochasticStyleOptions` or :obj:`None <python:None>`
        """
        return self._smoothed_line

    @smoothed_line.setter
    @class_sensitive(StochasticStyleOptions)
    def smoothed_line(self, value):
        self._smoothed_line = value

    @property
    def params(self) -> Optional[StochasticParameters]:
        """Parameters used in calculating the indicator's data points.

        :rtype: :class:`StochasticParameters` or :obj:`None <python:None>`
        """
        return self._params

    @params.setter
    @class_sensitive(StochasticParameters)
    def params(self, value):
        self._params = value

    @classmethod
    def _get_kwargs_from_dict(cls, as_dict):
        kwargs = {
            'accessibility': as_dict.get('accessibility', None),
            'allow_point_select': as_dict.get('allowPointSelect', None),
            'animation': as_dict.get('animation', None),
            'class_name': as_dict.get('className', None),
            'clip': as_dict.get('clip', None),
            'color': as_dict.get('color', None),
            'cursor': as_dict.get('cursor', None),
            'custom': as_dict.get('custom', None),
            'dash_style': as_dict.get('dashStyle', None),
            'data_labels': as_dict.get('dataLabels', None),
            'description': as_dict.get('description', None),
            'enable_mouse_tracking': as_dict.get('enableMouseTracking', None),
            'events': as_dict.get('events', None),
            'include_in_data_export': as_dict.get('includeInDataExport', None),
            'keys': as_dict.get('keys', None),
            'label': as_dict.get('label', None),
            'linked_to': as_dict.get('linkedTo', None),
            'marker': as_dict.get('marker', None),
            'on_point': as_dict.get('onPoint', None),
            'opacity': as_dict.get('opacity', None),
            'point': as_dict.get('point', None),
            'point_description_formatter': as_dict.get('pointDescriptionFormatter', None),
            'selected': as_dict.get('selected', None),
            'show_checkbox': as_dict.get('showCheckbox', None),
            'show_in_legend': as_dict.get('showInLegend', None),
            'skip_keyboard_navigation': as_dict.get('skipKeyboardNavigation', None),
            'states': as_dict.get('states', None),
            'sticky_tracking': as_dict.get('stickyTracking', None),
            'threshold': as_dict.get('threshold', None),
            'tooltip': as_dict.get('tooltip', None),
            'turbo_threshold': as_dict.get('turboThreshold', None),
            'visible': as_dict.get('visible', None),

            'animation_limit': as_dict.get('animationLimit', None),
            'boost_blending': as_dict.get('boostBlending', None),
            'boost_threshold': as_dict.get('boostThreshold', None),
            'color_index': as_dict.get('colorIndex', None),
            'color_key': as_dict.get('colorKey', None),
            'connect_nulls': as_dict.get('connectNulls', None),
            'crisp': as_dict.get('crisp', None),
            'crop_threshold': as_dict.get('cropThreshold', None),
            'data_sorting': as_dict.get('dataSorting', None),
            'find_nearest_point_by': as_dict.get('findNearestPointBy', None),
            'get_extremes_from_all': as_dict.get('getExtremesFromAll', None),
            'linecap': as_dict.get('linecap', None),
            'line_width': as_dict.get('lineWidth', None),
            'relative_x_value': as_dict.get('relativeXValue', None),
            'shadow': as_dict.get('shadow', None),
            'soft_threshold': as_dict.get('softThreshold', None),
            'step': as_dict.get('step', None),
            'zone_axis': as_dict.get('zoneAxis', None),
            'zones': as_dict.get('zones', None),

            'color_axis': as_dict.get('colorAxis', None),
            'connect_ends': as_dict.get('connectEnds', None),
            'drag_drop': as_dict.get('dragDrop', None),
            'negative_color': as_dict.get('negativeColor', None),
            'point_interval': as_dict.get('pointInterval', None),
            'point_interval_unit': as_dict.get('pointIntervalUnit', None),
            'point_placement': as_dict.get('pointPlacement', None),
            'point_start': as_dict.get('pointStart', None),
            'stacking': as_dict.get('stacking', None),

            'compare_start': as_dict.get('compareStart', None),
            'compare_to_main': as_dict.get('compareToMain', None),
            'cumulative': as_dict.get('cumulative', None),
            'data_as_columns': as_dict.get('dataAsColumns', None),
            'data_grouping': as_dict.get('dataGrouping', None),
            'gap_size': as_dict.get('gapSize', None),
            'gap_unit': as_dict.get('gapUnit', None),
            'last_price': as_dict.get('lastPrice', None),
            'last_visible_price': as_dict.get('lastVisiblePrice', None),
            'name': as_dict.get('name', None),
            'params': as_dict.get('params', None),

            'compare': as_dict.get('compare', None),
            'compare_base': as_dict.get('compareBase', None),

            'smoothed_line': as_dict.get('smoothedLine', None),
        }

        return kwargs

    def _to_untrimmed_dict(self, in_cls = None) -> dict:
        untrimmed = {
            'smoothedLine': self.smoothed_line,
        }

        parent_as_dict = super()._to_untrimmed_dict(in_cls = in_cls)

        for key in parent_as_dict:
            untrimmed[key] = parent_as_dict[key]

        return untrimmed


class SlowStochasticOptions(StochasticOptions):
    """Configuration for a Slow Stochastic :term:`Oscillator`.

    .. figure:: ../../../../_static/slow-stochastic-example.png
      :alt: Slow Stochsatic Oscillator Example Chart
      :align: center

    """
    @property
    def params(self) -> Optional[SlowStochasticParameters]:
        """Parameters used in calculating the indicator's data points.

        :rtype: :class:`SlowStochasticParameters` or :obj:`None <python:None>`
        """
        return self._params

    @params.setter
    @class_sensitive(SlowStochasticParameters)
    def params(self, value):
        self._params = value
