"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Archiver = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk = require("aws-cdk-lib/");
const aws_cdk_lib_1 = require("aws-cdk-lib/");
const codebuild = require("aws-cdk-lib/aws-codebuild");
const iam = require("aws-cdk-lib/aws-iam");
const kms = require("aws-cdk-lib/aws-kms");
const logs = require("aws-cdk-lib/aws-logs");
const s3 = require("aws-cdk-lib/aws-s3");
const snsNotifications = require("aws-cdk-lib/aws-s3-notifications");
const sns = require("aws-cdk-lib/aws-sns");
const constructs_1 = require("constructs");
class Archiver extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.props = props;
        this.logGroupKmsKey = this.createLogGroupKey();
        this.logGroup = this.createLogGroup();
        const topic = new sns.Topic(this, 'notifications', {
            displayName: 'archiver-notifications',
        });
        this.bucket = this.createArchiveBucket();
        this.bucket.addEventNotification(s3.EventType.OBJECT_CREATED, new snsNotifications.SnsDestination(topic));
        this.createProjects();
        new cdk.CfnOutput(this, 's3-bucket-arn', {
            description: 'ARN of the S3 bucket storing the repositories.',
            value: this.bucket.bucketArn,
        });
        new cdk.CfnOutput(this, 'log-group-arn', {
            description: 'ARN of the Cloudwatch Log group storing the code build logs.',
            value: this.logGroup.logGroupArn,
        });
        new cdk.CfnOutput(this, 'log-group-key', {
            description: 'ARN of the KMS key used to encrypt the Cloudwatch logs.',
            value: this.logGroupKmsKey.keyArn,
        });
        new cdk.CfnOutput(this, 'sns-topic-arn', {
            description: 'ARN of the SNS topic.',
            value: topic.topicArn,
        });
    }
    createLogGroup() {
        const loggroup = new logs.LogGroup(this, 'loggroup', {
            encryptionKey: this.logGroupKmsKey,
            retention: this.props.retentionDays ? this.props.retentionDays : logs.RetentionDays.ONE_MONTH,
        });
        loggroup.node.addDependency(this.logGroupKmsKey);
        return loggroup;
    }
    /**
     *Create the S3 bucket that will later store the repositories.
     *
     * @private
     * @return {*}
     * @memberof Archiver
     */
    createArchiveBucket() {
        return new s3.Bucket(this, 'destination', {
            blockPublicAccess: s3.BlockPublicAccess.BLOCK_ALL,
            encryption: s3.BucketEncryption.S3_MANAGED,
            enforceSSL: true,
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.RETAIN,
            lifecycleRules: [
                {
                    expiration: cdk.Duration.days(360),
                    transitions: [
                        {
                            storageClass: s3.StorageClass.INFREQUENT_ACCESS,
                            transitionAfter: cdk.Duration.days(30),
                        },
                        {
                            storageClass: s3.StorageClass.GLACIER,
                            transitionAfter: cdk.Duration.days(90),
                        },
                        {
                            storageClass: s3.StorageClass.DEEP_ARCHIVE,
                            transitionAfter: cdk.Duration.days(180),
                        },
                    ],
                },
            ],
            publicReadAccess: false,
            versioned: true,
        });
    }
    createLogGroupKey() {
        const key = new kms.Key(this, 'loggroupKey', {
            description: 'Repository Archiver',
            enableKeyRotation: true,
            pendingWindow: cdk.Duration.days(7),
            keyUsage: kms.KeyUsage.ENCRYPT_DECRYPT,
            keySpec: kms.KeySpec.SYMMETRIC_DEFAULT,
            alias: 'archiver-loggroup-key',
        });
        key.grantEncryptDecrypt(new iam.ServicePrincipal(`logs.${cdk.Stack.of(this).region}.amazonaws.com`));
        return key;
    }
    /**
     * Creates for each backup configuration a separate CodeBuild project.
     *
     * @private
     * @memberof Archiver
     */
    createProjects() {
        this.props.backupConfigurations.forEach((element) => {
            const project = this.createProject(element);
            project.enableBatchBuilds();
            this.bucket.grantReadWrite(project);
        });
    }
    /**
     * Create a CodeBuild project.
     *
     * @private
     * @param {BackupConfiguration} element
     * @return {*}
     * @memberof Archiver
     */
    createProject(element) {
        return new codebuild.Project(this, 'archiver-' + element.organizationName + '-' + element.projectName, {
            timeout: cdk.Duration.hours(5),
            projectName: 'AzureDevOpsGitBackup' +
                '-' +
                element.organizationName +
                '-' +
                element.projectName,
            description: 'Backup Azure DevOps git repositories to an S3 bucket.',
            checkSecretsInPlainTextEnvVariables: true,
            concurrentBuildLimit: 90,
            environment: {
                environmentVariables: {
                    TOKEN: {
                        value: element.secretArn + ':pat',
                        type: codebuild.BuildEnvironmentVariableType.SECRETS_MANAGER,
                    },
                    ORGANIZATION: { value: element.organizationName },
                    PROJECT: { value: element.projectName },
                },
                buildImage: codebuild.LinuxBuildImage.STANDARD_6_0,
            },
            logging: {
                cloudWatch: {
                    enabled: true,
                    logGroup: this.logGroup,
                },
            },
            buildSpec: codebuild.BuildSpec.fromObject({
                version: 0.2,
                batch: {
                    'fail-fast': false,
                    'build-list': this.createBatchConfiguration(element.repositoryNames),
                },
                phases: {
                    build: {
                        commands: [
                            'git clone --mirror "https://${TOKEN}@dev.azure.com/${ORGANIZATION}/${PROJECT}/_git/${REPOSITORY}"',
                            'tar czf ${REPOSITORY}.tgz ./${REPOSITORY}.git',
                            'aws s3 cp ./${REPOSITORY}.tgz ' +
                                this.bucket.s3UrlForObject() +
                                '/${ORGANIZATION}/${PROJECT}/${REPOSITORY}.tgz',
                        ],
                    },
                },
            }),
        });
    }
    createBatchConfiguration(repositoryNames) {
        const output = [];
        repositoryNames.forEach((element) => {
            output.push({
                identifier: 'build_' + element.replace(/-/g, '_'),
                env: {
                    variables: {
                        REPOSITORY: element,
                    },
                },
            });
        });
        return output;
    }
}
exports.Archiver = Archiver;
_a = JSII_RTTI_SYMBOL_1;
Archiver[_a] = { fqn: "azure-devops-repository-archiver.Archiver", version: "0.0.8" };
//# sourceMappingURL=data:application/json;base64,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