from __future__ import annotations

from datetime import datetime
from typing import Literal, Optional

from pydantic import BaseModel, Field

from slingshot import schemas

from .base_graphql import BaseGraphQLEntity


class BlobArtifactShallow(BaseGraphQLEntity):
    _depends_on = []
    _fragment = """
        fragment BlobArtifactShallow on BlobArtifacts {
          blobArtifactId
          createdAt
          updatedAt
          bytesHash
          bytesSize
          name
          tag
        } """

    blob_artifact_id: str = Field(..., alias="blobArtifactId")
    name: str = Field(..., alias="name")
    tag: Optional[str] = Field(..., alias="tag")
    created_at: datetime = Field(..., alias="createdAt")
    updated_at: datetime = Field(..., alias="updatedAt")
    bytes_hash: Optional[str] = Field(..., alias="bytesHash")
    bytes_size: Optional[int] = Field(..., alias="bytesSize")
    blob_artifact_name: Optional[str] = Field(None, alias="name")
    blob_artifact_tag: Optional[str] = Field(None, alias="tag")


class Mount(BaseGraphQLEntity):
    _fragment = """
        fragment Mount on Mount {
          mountId
          deploymentId
          trainingRunId
          mountPath
          mountMode
          downloadedBlobArtifact {
            ...BlobArtifactShallow
          }
          uploadedBlobArtifact {
            ...BlobArtifactShallow
          }
        } """

    _depends_on = [BlobArtifactShallow]

    mount_id: str = Field(..., alias="mountId")
    deployment_id: Optional[str] = Field(None, alias="deploymentId")
    run_id: Optional[str] = Field(None, alias="trainingRunId")
    mount_path: str = Field(..., alias="mountPath")
    mount_mode: Literal["DOWNLOAD", "UPLOAD", "VOLUME"] = Field(..., alias="mountMode")
    downloaded_blob_artifact: Optional[BlobArtifactShallow] = Field(..., alias="downloadedBlobArtifact")
    uploaded_blob_artifact: Optional[BlobArtifactShallow] = Field(..., alias="uploadedBlobArtifact")


class BlobArtifact(BaseGraphQLEntity):
    _fragment = """
        fragment BlobArtifact on BlobArtifacts {
          blobArtifactId
          createdAt
          updatedAt
          bytesHash
          bytesSize
          name
          tag
          isDraft
          originMount {
            ...Mount
          }
        } """
    _depends_on = [Mount]

    blob_artifact_id: str = Field(..., alias="blobArtifactId")
    name: str = Field(..., alias="name")
    tag: Optional[str] = Field(..., alias="tag")
    created_at: datetime = Field(..., alias="createdAt")
    updated_at: datetime = Field(..., alias="updatedAt")
    bytes_hash: Optional[str] = Field(..., alias="bytesHash")
    bytes_size: Optional[int] = Field(..., alias="bytesSize")
    blob_artifact_name: Optional[str] = Field(None, alias="name")
    blob_artifact_tag: Optional[str] = Field(None, alias="tag")
    origin_mount: Optional[Mount] = Field(..., alias="originMount")
    is_draft: bool = Field(..., alias="isDraft")


class SourceCodeArtifact(BaseGraphQLEntity):
    _depends_on = [BlobArtifactShallow]
    _fragment = """
        fragment SourceCodeArtifact on SourceCodes {
          sourceCodeId
          sourceCodeName
          description
          createdAt
          projectId
          blobArtifact {
            ...BlobArtifactShallow
          }
        } """

    source_code_id: str = Field(..., alias="sourceCodeId")
    source_code_name: str = Field(..., alias="sourceCodeName")
    description: Optional[str] = Field(..., alias="description")
    created_at: datetime = Field(..., alias="createdAt")
    project_id: str = Field(..., alias="projectId")
    blob_artifact: BlobArtifactShallow = Field(..., alias="blobArtifact")


class Deployment(BaseGraphQLEntity):
    _depends_on = [BlobArtifact, SourceCodeArtifact]
    _fragment = """
        fragment Deployment on Deployments {
          createdAt
          deploymentId
          deploymentStatus
          projectId
          machineSize
          sourceCode {
            ...SourceCodeArtifact
          }
        } """

    deployment_id: str = Field(..., alias="deploymentId")
    created_at: datetime = Field(..., alias="createdAt")
    deployment_status: schemas.AppInstanceStatus = Field(..., alias="deploymentStatus")
    project_id: str = Field(..., alias="projectId")
    source_code: SourceCodeArtifact = Field(..., alias="sourceCode")


class Build(BaseGraphQLEntity):
    _depends_on = []
    _fragment = """
        fragment Build on Builds{
          buildId
          buildStatus
        } """
    build_id: str = Field(..., alias="buildId")
    build_status: str = Field(..., alias="buildStatus")  # TODO: Replace with enum or Literal


class ExecutionEnvironment(BaseGraphQLEntity):
    _depends_on = [Build]
    _fragment = """
        fragment ExecutionEnvironment on ExecutionEnvironments {
          executionEnvironmentId
          build {
            ...Build
          }
          createdAt
          requirements
          errorMessage
          status
          gpuDrivers
          requestedAptPackages
          requestedPythonRequirements
        } """
    execution_environment_id: str = Field(..., alias="executionEnvironmentId")
    status: schemas.ExecEnvStatus
    created_at: datetime = Field(..., alias="createdAt")
    build: Optional[Build]
    requirements: Optional[str]
    error_message: Optional[str] = Field(..., alias="errorMessage")
    gpu_drivers: bool = Field(..., alias="gpuDrivers")
    requested_apt_packages: list[dict[str, str]] = Field(..., alias="requestedAptPackages")
    requested_python_requirements: list[dict[str, Optional[str]]] = Field(
        ..., alias="requestedPythonRequirements", repr=False
    )


class Run(BaseGraphQLEntity):
    _depends_on = [ExecutionEnvironment, Mount, SourceCodeArtifact]
    _fragment = """
        fragment Run on TrainingRuns {
          trainingRunId
          runSpecId
          trainingRunName
          jobStatus
          createdAt
          startTime
          endTime
          machineSize
          hyperparameters
          cmd
          sourceCode {
            ...SourceCodeArtifact
          }
          executionEnvironment {
            ...ExecutionEnvironment
          }
          mounts {
            ...Mount
          }
        }
        """
    run_id: str = Field(..., alias="trainingRunId")
    run_spec_id: Optional[str] = Field(..., alias="runSpecId")
    run_name: str = Field(..., alias="trainingRunName")
    job_status: schemas.JobStatus = Field(..., alias="jobStatus")
    created_at: datetime = Field(..., alias="createdAt")
    start_time: Optional[datetime] = Field(..., alias="startTime")
    end_time: Optional[datetime] = Field(..., alias="endTime")
    machine_size: schemas.MachineSize = Field(..., alias="machineSize")
    hyperparameters: Optional[str]
    cmd: Optional[str]
    source_code: SourceCodeArtifact = Field(..., alias="sourceCode")
    # TODO: ENG-1062 projects don't have permissions for outdated envs so this might be returned as None
    execution_environment: Optional[ExecutionEnvironment] = Field(..., alias="executionEnvironment")
    mounts: list[Mount] = Field(..., alias="mounts")


class Volume(BaseGraphQLEntity):
    _depends_on = []
    _fragment = """
        fragment Volume on Volumes {
          volumeId
          volumeName
          createdAt
        }
        """
    volume_id: str = Field(..., alias="volumeId")
    volume_name: str = Field(..., alias="volumeName")
    created_at: datetime = Field(..., alias="createdAt")


class ProjectFields(BaseGraphQLEntity):
    _depends_on = []
    _fragment = """
        fragment ProjectFields on Projects {
          projectId
          displayName
        } """
    project_id: str = Field(..., alias="projectId")
    display_name: str = Field(..., alias="displayName")


class BillingLineItem(BaseGraphQLEntity):
    _depends_on = []
    _fragment = """
        fragment BillingLineItem on BillingLineItems {
          appInstanceId
          deploymentId
          runId
          computeCostCredits
          computeCostCreditsInProgress
        } """
    app_instance_id: Optional[str] = Field(..., alias="appInstanceId")
    deployment_id: Optional[str] = Field(..., alias="deploymentId")
    run_id: Optional[str] = Field(..., alias="runId")
    compute_cost_credits: Optional[int] = Field(..., alias="computeCostCredits")
    compute_cost_credits_in_progress: int = Field(..., alias="computeCostCreditsInProgress")


class ProjectProjection(BaseModel):
    project: Optional[ProjectFields]


class UserWithProjects(BaseGraphQLEntity):
    _depends_on = [ProjectFields]
    _fragment = """
        fragment UserWithProjects on Users {
          displayName
          username
          sshPublicKey
          userId
          isActivated
          userProjectAcls(where: {project: {isArchived: {_eq: false}}}) {
            project {
              ...ProjectFields
            }
          }
        } """
    display_name: str = Field(..., alias="displayName")
    username: str = Field(..., alias="username")
    ssh_public_key: Optional[str] = Field(..., alias="sshPublicKey")
    user_id: str = Field(..., alias="userId")
    user_project_acls: list[ProjectProjection] = Field(..., alias="userProjectAcls")
    is_activated: bool = Field(..., alias="isActivated")

    @property
    def projects(self) -> list[ProjectFields]:
        return [acl.project for acl in self.user_project_acls if acl.project]  # Filter out None values


class ServiceAccount(BaseGraphQLEntity):
    _depends_on = []
    _fragment = """
        fragment ServiceAccount on ServiceAccounts {
              serviceAccountId
              nickname
              lastFour
              apiKeyHash
              createdAt
        } """
    service_account_id: str = Field(..., alias="serviceAccountId")
    nickname: Optional[str] = Field(..., alias="nickname")
    last_four: str = Field(..., alias="lastFour")
    api_key_hash: str = Field(..., alias="apiKeyHash")
    created_at: datetime = Field(..., alias="createdAt")


class ServiceAccountWithProjects(ServiceAccount):
    _depends_on = [ProjectFields, ServiceAccount]
    _fragment = """
        fragment ServiceAccountWithProjects on ServiceAccounts {
          ...ServiceAccount
          serviceAccountProjectAcls {
            project {
              ...ProjectFields
            }
          }
        } """

    service_account_project_acls: list[ProjectProjection] = Field(..., alias="serviceAccountProjectAcls")

    @property
    def projects(self) -> list[ProjectFields]:
        return [acl.project for acl in self.service_account_project_acls if acl.project]  # Filter out None values


class MeResponse(BaseModel):
    projects: list[ProjectFields]
    user: Optional[UserWithProjects] = None
    service_account: Optional[ServiceAccountWithProjects] = None

    @classmethod
    def from_user(cls, user: UserWithProjects) -> MeResponse:
        return cls(projects=user.projects, user=user)

    @classmethod
    def from_service_account(cls, service_account: ServiceAccountWithProjects) -> MeResponse:
        return cls(projects=service_account.projects, service_account=service_account)


class MountSpec(BaseGraphQLEntity):
    _depends_on = []
    _fragment = """
        fragment MountSpec on MountSpecs {
            path
            mode
            tag
            name
        }
    """
    path: str = Field(..., alias="path")
    mode: Literal["DOWNLOAD", "UPLOAD", "VOLUME_READONLY", "VOLUME"] = Field(..., alias="mode")
    tag: Optional[str] = Field(None, alias="tag")
    name: Optional[str] = Field(None, alias="name")


class ExecutionEnvironmentSpec(BaseGraphQLEntity):
    _depends_on = [ExecutionEnvironment]
    _fragment = """
        fragment ExecutionEnvironmentSpec on ExecutionEnvironmentSpecs {
            executionEnvironmentSpecId
            executionEnvironmentSpecName
            createdAt
            gpuDrivers
            requestedAptPackages
            requestedPythonRequirements
            isArchived
            executionEnvironment {
                ...ExecutionEnvironment
            }
        }"""

    execution_environment_spec_id: str = Field(..., alias="executionEnvironmentSpecId")
    execution_environment_spec_name: str = Field(..., alias="executionEnvironmentSpecName")
    created_at: datetime = Field(..., alias="createdAt")
    execution_environment: ExecutionEnvironment = Field(..., alias="executionEnvironment")
    gpu_drivers: bool = Field(..., alias="gpuDrivers")
    requested_apt_packages: list[dict[str, str]] = Field(..., alias="requestedAptPackages")
    requested_python_requirements: list[dict[str, Optional[str]]] = Field(
        ..., alias="requestedPythonRequirements", repr=False
    )
    is_archived: bool = Field(..., alias="isArchived")


class _AppInstanceProjection(BaseModel):
    app_instance_status: schemas.AppInstanceStatus = Field(..., alias="appInstanceStatus")
    app_instance_url: Optional[str] = Field(..., alias="appInstanceUrl")
    created_at: datetime = Field(..., alias="createdAt")
    ssh_port: Optional[int] = Field(..., alias="sshPort")


class AppSpec(BaseGraphQLEntity):
    _depends_on = [ExecutionEnvironmentSpec, MountSpec, Deployment]
    _fragment = """
        fragment SlingshotAppSpec on AppSpecs {
            appSpecId
            appSpecName
            appSpecCommand
            appType
            appSubType
            projectId
            machineSize
            configVariables
            serviceAccount
            appPort
            batchSize
            batchInterval
            executionEnvironmentSpec {
                ...ExecutionEnvironmentSpec
            }
            mountSpecs {
                ...MountSpec
            }
            appInstances(orderBy:{ createdAt:DESC }, limit:1) {
                appInstanceStatus
                appInstanceUrl
                createdAt
                sshPort
            }
            deployments(orderBy:{ createdAt:DESC }, limit:1) {
                ...Deployment
            }
        }
    """
    app_spec_id: str = Field(..., alias="appSpecId")
    app_spec_name: str = Field(..., alias="appSpecName")
    app_spec_command: Optional[str] = Field(..., alias="appSpecCommand")
    app_type: schemas.AppType = Field(..., alias="appType")
    app_sub_type: Optional[schemas.AppSubType] = Field(..., alias="appSubType")
    app_port: Optional[int] = Field(..., alias="appPort")
    config_variables: Optional[str] = Field(..., alias="configVariables")
    batch_size: Optional[int] = Field(..., alias="batchSize")
    batch_interval: Optional[int] = Field(..., alias="batchInterval")
    project_id: str = Field(..., alias="projectId")
    machine_size: schemas.MachineSize = Field(..., alias="machineSize")
    service_account: bool = Field(..., alias="serviceAccount")
    execution_environment_spec: ExecutionEnvironmentSpec = Field(..., alias="executionEnvironmentSpec")
    mount_specs: list[MountSpec] = Field(..., alias="mountSpecs")
    app_instances: list[_AppInstanceProjection] = Field(..., alias="appInstances")
    deployments: list[Deployment] = Field(..., alias="deployments")

    @property
    def app_instance_status(self) -> schemas.AppInstanceStatus | None:
        if self.app_instances:
            return self.app_instances[0].app_instance_status
        return None

    @property
    def app_instance_url(self) -> str | None:
        if self.app_instances:
            return self.app_instances[0].app_instance_url
        return None

    @property
    def last_created_at(self) -> datetime | None:
        if self.app_instances:
            return self.app_instances[0].created_at
        if self.deployments:
            return self.deployments[0].created_at
        return None

    @property
    def deployment_status(self) -> schemas.AppInstanceStatus | None:
        if self.deployments:
            return self.deployments[0].deployment_status
        return None


class AppInstance(BaseGraphQLEntity):
    _depends_on = [AppSpec]
    _fragment = """
        fragment AppInstance on AppInstances {
            appInstanceId
            appInstanceStatus
            appInstanceUrl
            appType
            appSubType
            appPort
            sshPort
            executionEnvironmentId
            createdAt
            appSpecCommand
            appSpecId
            appSpec {
                ...SlingshotAppSpec
            }
        }
    """
    app_instance_id: str = Field(..., alias="appInstanceId")
    app_instance_status: schemas.AppInstanceStatus = Field(..., alias="appInstanceStatus")
    app_instance_url: Optional[str] = Field(..., alias="appInstanceUrl")
    app_type: schemas.AppType = Field(..., alias="appType")
    app_sub_type: Optional[schemas.AppSubType] = Field(..., alias="appSubType")
    app_port: Optional[int] = Field(..., alias="appPort")
    ssh_port: Optional[int] = Field(..., alias="sshPort")
    execution_environment_id: str = Field(..., alias="executionEnvironmentId")
    created_at: datetime = Field(..., alias="createdAt")
    app_spec_command: str = Field(..., alias="appSpecCommand")
    app_spec_id: str = Field(..., alias="appSpecId")
    app_spec: AppSpec = Field(..., alias="appSpec")


class ProjectSecret(BaseGraphQLEntity):
    _depends_on = []
    _fragment = """
        fragment ProjectSecret on ProjectSecrets {
          secretName
        }
    """
    secret_name: str = Field(..., alias="secretName")


class DeploymentInstance(BaseGraphQLEntity):
    _depends_on = [AppSpec, SourceCodeArtifact]
    _fragment = """
        fragment DeploymentInstance on Deployments {
          createdAt
          deploymentId
          deploymentStatus
          projectId
          machineSize
          appSpecId
          sourceCode {
            ...SourceCodeArtifact
          }
          appSpec {
            ...SlingshotAppSpec
          }
        } """

    deployment_id: str = Field(..., alias="deploymentId")
    created_at: datetime = Field(..., alias="createdAt")
    deployment_status: schemas.AppInstanceStatus = Field(..., alias="deploymentStatus")
    project_id: str = Field(..., alias="projectId")
    source_code: SourceCodeArtifact = Field(..., alias="sourceCode")
    app_spec_id: str = Field(..., alias="appSpecId")
    app_spec: AppSpec = Field(..., alias="appSpec")
