"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const core_1 = require("@aws-cdk/core");
const logs_generated_1 = require("./logs.generated");
/**
 * A new CloudWatch Logs Destination for use in cross-account scenarios
 *
 * CrossAccountDestinations are used to subscribe a Kinesis stream in a
 * different account to a CloudWatch Subscription.
 *
 * Consumers will hardly ever need to use this class. Instead, directly
 * subscribe a Kinesis stream using the integration class in the
 * `@aws-cdk/aws-logs-destinations` package; if necessary, a
 * `CrossAccountDestination` will be created automatically.
 *
 * @resource AWS::Logs::Destination
 */
class CrossAccountDestination extends cdk.Resource {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.destinationName ||
                // In the underlying model, the name is not optional, but we make it so anyway.
                core_1.Lazy.stringValue({ produce: () => this.generateUniqueName() }),
        });
        /**
         * Policy object of this CrossAccountDestination object
         */
        this.policyDocument = new iam.PolicyDocument();
        this.resource = new logs_generated_1.CfnDestination(this, 'Resource', {
            destinationName: this.physicalName,
            // Must be stringified policy
            destinationPolicy: this.lazyStringifiedPolicyDocument(),
            roleArn: props.role.roleArn,
            targetArn: props.targetArn
        });
        this.destinationArn = this.getResourceArnAttribute(this.resource.attrArn, {
            service: 'logs',
            resource: 'destination',
            resourceName: this.physicalName,
            sep: ':',
        });
        this.destinationName = this.getResourceNameAttribute(this.resource.ref);
    }
    addToPolicy(statement) {
        this.policyDocument.addStatements(statement);
    }
    bind(_scope, _sourceLogGroup) {
        return { arn: this.destinationArn };
    }
    /**
     * Generate a unique Destination name in case the user didn't supply one
     */
    generateUniqueName() {
        // Combination of stack name and LogicalID, which are guaranteed to be unique.
        return core_1.Stack.of(this).stackName + '-' + this.resource.logicalId;
    }
    /**
     * Return a stringified JSON version of the PolicyDocument
     */
    lazyStringifiedPolicyDocument() {
        return core_1.Lazy.stringValue({ produce: () => this.policyDocument.isEmpty ? '' : core_1.Stack.of(this).toJsonString(this.policyDocument)
        });
    }
}
exports.CrossAccountDestination = CrossAccountDestination;
//# sourceMappingURL=data:application/json;base64,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