# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['vietnam_provinces', 'vietnam_provinces.enums']

package_data = \
{'': ['*'], 'vietnam_provinces': ['data/*']}

install_requires = \
['Unidecode>=1.1.2,<2.0.0',
 'fast-enum>=1.3.0,<2.0.0',
 'logbook>=1.5.3,<2.0.0',
 'single-version>=1.5.1,<2.0.0']

setup_kwargs = {
    'name': 'vietnam-provinces',
    'version': '0.4.0',
    'description': 'Library to provide list of Vietnam administrative divisions (tỉnh thành, quận huyện, phường xã).',
    'long_description': '================\nVietnamProvinces\n================\n\n|image love| |image pypi|\n\n[`Tiếng Việt <vietnamese_>`_]\n\nLibrary to provide list of Vietnam administrative divisions (tỉnh thành, quận huyện, phường xã) with the name and code as defined by `General Statistics Office of Viet Nam <gso_vn_>`_ (Tổng cục Thống kê).\n\nExample:\n\n.. code-block:: json\n\n    {\n        "name": "Tỉnh Cà Mau",\n        "code": 96,\n        "codename": "tinh_ca_mau",\n        "division_type": "tỉnh",\n        "phone_code": 290,\n        "districts": [\n            {\n                "name": "Huyện Đầm Dơi",\n                "code": 970,\n                "codename": "huyen_dam_doi",\n                "division_type": "huyện",\n                "wards": [\n                    {\n                        "name": "Thị trấn Đầm Dơi",\n                        "code": 32152,\n                        "codename": "thi_tran_dam_doi",\n                        "division_type": "thị trấn"\n                    },\n                    {\n                        "name": "Xã Tạ An Khương",\n                        "code": 32155,\n                        "codename": "xa_ta_an_khuong",\n                        "division_type": "xã"\n                    },\n                ]\n            }\n        ]\n    }\n\nThis library provides data in these forms:\n\n1. JSON\n\nThis data is suitable for applications which don\'t need to access the data often. They are fine with loading JSON and extract information from it. The JSON files are saved in *data* folder. You can get the file path via ``vietnam_provinces.NESTED_DIVISIONS_JSON_PATH`` variable.\n\nNote that this variable only returns the path of the file, not the content. It is up to application developer to use any method to parse the JSON. For example:\n\n.. code-block:: python\n\n    import orjson\n    import rapidjson\n    from vietnam_provinces import NESTED_DIVISIONS_JSON_PATH\n\n    # With rapidjson\n    with NESTED_DIVISIONS_JSON_PATH.open() as f:\n        rapidjson.load(f)\n\n    # With orjson\n    orjson.loads(NESTED_DIVISIONS_JSON_PATH.read_bytes())\n\nDue to the big amount of data (10609 wards all over Viet Nam), this loading will be slow.\n\n\n2. Python data type\n\nThis data is useful for some applications which need to access the data more often. They are built as ``Enum``, where you can import in Python code:\n\n.. code-block:: python\n\n    >>> from vietnam_provinces.enums import ProvinceEnum, ProvinceDEnum, DistrictEnum, DistrictDEnum\n\n    >>> ProvinceEnum.P_77\n    <ProvinceEnum.P_77: Province(name=\'Tỉnh Bà Rịa - Vũng Tàu\', code=77, division_type=<VietNamDivisionType.TINH: \'tỉnh\'>, codename=\'tinh_ba_ria_vung_tau\', phone_code=254)>\n\n    >>> ProvinceDEnum.BA_RIA_VUNG_TAU\n    <ProvinceDEnum.BA_RIA_VUNG_TAU: Province(name=\'Tỉnh Bà Rịa - Vũng Tàu\', code=77, division_type=<VietNamDivisionType.TINH: \'tỉnh\'>, codename=\'tinh_ba_ria_vung_tau\', phone_code=254)>\n\n    >>> DistrictEnum.D_624\n    >>> <DistrictEnum.D_624: District(name=\'Thị xã Ayun Pa\', code=624, division_type=<VietNamDivisionType.THI_XA: \'thị xã\'>, codename=\'thi_xa_ayun_pa\', province_code=64)>\n\n    >>> DistrictDEnum.AYUN_PA_GL\n    <DistrictDEnum.AYUN_PA_GL: District(name=\'Thị xã Ayun Pa\', code=624, division_type=<VietNamDivisionType.THI_XA: \'thị xã\'>, codename=\'thi_xa_ayun_pa\', province_code=64)>\n\n    >>> from vietnam_provinces.enums.wards import WardEnum, WardDEnum\n\n    >>> WardEnum.W_7450\n    <WardEnum.W_7450: Ward(name=\'Xã Đông Hưng\', code=7450, division_type=<VietNamDivisionType.XA: \'xã\'>, codename=\'xa_dong_hung\', district_code=218)>\n\n    >>> WardDEnum.BG_DONG_HUNG_7450\n    <WardDEnum.BG_DONG_HUNG_7450: Ward(name=\'Xã Đông Hưng\', code=7450, division_type=<VietNamDivisionType.XA: \'xã\'>, codename=\'xa_dong_hung\', district_code=218)>\n\n\nLoading wards this way is far more faster than the JSON option.\n\nThey are made as ``Enum``, so that library user can take advantage of auto-complete feature of IDE/code editors in development. It prevents typo mistake.\n\nThe Ward Enum has two variants:\n\n- ``WardEnum``: Has member name in form of numeric ward code (``W_28912``). It helps look up a ward by its code (which is a most-seen use case).\n\n- ``WardDEnum``: Has more readable member name (``D`` means "descriptive"), to help the application code easier to reason about. For example, looking at ``WardDEnum.BT_PHAN_RI_CUA_22972``, the programmer can guess that this ward is "Phan Rí Cửa", of "Bình Thuận" province.\n\nSimilarly, other levels (District, Province) also have two variants of Enum.\n\nExample of looking up ``Ward``, ``District``, ``Province`` with theirs numeric code:\n\n.. code-block:: python\n\n    # Assume that you are loading user info from your database\n    user_info = load_user_info()\n\n    province_code = user_info[\'province_code\']\n    province = ProvinceEnum[f\'P_{province_code}\'].value\n\nUnlike ``ProvinceDEnum``, ``DistrictDEnum``, the ``WardDEnum`` has ward code in member name. It is because there are too many Vietnamese wards with the same name. There is no way to build unique ID for wards, with pure Latin letters (Vietnamese punctuations stripped), even if we add district and province info to the ID. Let\'s take "Xã Đông Thành" and "Xã Đông Thạnh" as example. Both belong to "Huyện Bình Minh" of "Vĩnh Long", both produces ID name "DONG_THANH". Although Python allows Unicode as ID name, like "ĐÔNG_THẠNH", but it is not practical yet because the code formatter tool (`Black`_) will still normalizes it to Latin form.\n\nBecause the ``WardEnum`` has many records (10609 in February 2021) and may not be needed in some applications, I move it to separate module, to avoid loading automatically to application.\n\n\nMember of these enums, the ``Province``, ``District`` and ``Ward`` data types, can be imported from top-level of ``vietnam_provinces``.\n\n.. code-block:: python\n\n    >>> from vietnam_provinces import Province, District, Ward\n\n\nInstall\n-------\n\n.. code-block:: sh\n\n    pip3 install vietnam-provinces\n\n\nThis library is compatible with Python 3.7+ (due to the use of *dataclass*).\n\n\nDevelopment\n-----------\n\nIn development, this project has a tool to convert data from government sources.\n\nThe tool doesn\'t directly crawl data from government websites because the data rarely change (it doesn\'t worth developing the feature which you only need to use each ten years), and because those websites provide data in unfriendly Microsoft Office formats.\n\nUpdate data\n~~~~~~~~~~~\n\nIn the future, when the authority reorganize administrative divisions, we need to collect this data again from GSOVN website. Do:\n\n- Go to: https://danhmuchanhchinh.gso.gov.vn/ (this URL may change when `GSOVN <gso_vn_>`_ replaces their software).\n- Find the button "Xuất Excel".\n- Tick the "Quận Huyện Phường Xã" checkbox.\n- Click the button to export and download list of units in Excel (XLS) file.\n- Use LibreOffice to convert Excel file to CSV file. For example, we name it *Xa_2021-02-03.csv*.\n- Run this tool to compute data to JSON format:\n\n.. code-block:: sh\n\n    python3 -m dev -i dev/seed-data/Xa_2021-02-03.csv -o vietnam_provinces/data/nested-divisions.json\n\nYou can run\n\n.. code-block:: sh\n\n    python3 -m dev --help\n\nto see more options of that tool.\n\nNote that this tool is only available in the source folder (cloned from Git). It is not included in the distributable Python package.\n\n\nGenerate Python code\n~~~~~~~~~~~~~~~~~~~~\n\n.. code-block:: sh\n\n    python3 -m dev -i dev/seed-data/Xa_2021-02-03.csv -f python\n\n\nData source\n~~~~~~~~~~~\n\n- Name and code of provinces, districts and wards:  `General Statistics Office of Viet Nam <gso_vn_>`_.\n- Phone area code: `Thái Bình province\'s department of Information and Communication <tb_ic_>`_.\n\n\nCredit\n------\n\nGiven to you by `Nguyễn Hồng Quân <quan_>`_, after nights and weekends.\n\n\n.. |image love| image:: https://madewithlove.now.sh/vn?heart=true&colorA=%23ffcd00&colorB=%23da251d\n.. |image pypi| image:: https://badgen.net/pypi/v/vietnam-provinces\n   :target: https://pypi.org/project/vietnam-provinces/\n.. _vietnamese: README.vi_VN.rst\n.. _gso_vn: https://www.gso.gov.vn/\n.. _tb_ic: https://sotttt.thaibinh.gov.vn/tin-tuc/buu-chinh-vien-thong/tra-cuu-ma-vung-dien-thoai-co-dinh-mat-dat-ma-mang-dien-thoa2.html\n.. _dataclass: https://docs.python.org/3/library/dataclasses.html\n.. _fast-enum: https://pypi.org/project/fast-enum/\n.. _pydantic: https://pypi.org/project/pydantic/\n.. _Black: https://github.com/psf/black\n.. _quan: https://quan.hoabinh.vn\n',
    'author': 'Nguyễn Hồng Quân',
    'author_email': 'ng.hong.quan@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/sunshine-tech/VietnamProvinces.git',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
