"Inherit LiveSlides class from here. It adds useful attributes and methods."
import os, io, re
from IPython import get_ipython
from contextlib import suppress
from .widgets import Widgets
from .screenshot import ScreenShot
from .navigation import Navigation
from .settings import LayoutSettings
from .notes import Notes
from .export_html import _HhtmlExporter

class BaseLiveSlides:
    def __init__(self):
        self.__widgets = Widgets()
        self.__screenshot = ScreenShot(self.__widgets)
        self.__navigation = Navigation(self.__widgets) # Not accessed later, just for actions
        self.__settings = LayoutSettings(self.__widgets)
        self.__export = _HhtmlExporter(self)
        self.notes = Notes(self, self.__widgets) # Needs main class for access to notes
        
        self._md_content = 'Slides not loaded from markdown.'
        
        self._toasts = {} #Store notifications
        self.toast_html = self.widgets.htmls.toast
        
        self.widgets.checks.toast.observe(self.__toggle_notify,names=['value'])
    
    @property
    def widgets(self):
        return self.__widgets
    
    @property
    def export(self):
        return self.__export
    
    @property
    def screenshot(self):
        return self.__screenshot
    
    @property
    def settings(self):
        return self.__settings
    
    def notify(self,content,title='IPySlides Notification',timeout=5):
        "Send inside notifications for user to know whats happened on some button click. Set `title = None` if need only content. Remain invisible in screenshot."
        return self.widgets._push_toast(content,title=title,timeout=timeout)
    
    def __toggle_notify(self,change):
        "Blocks notifications."
        if self.widgets.checks.toast.value:
            self.toast_html.layout.visibility = 'hidden' 
        else:
            self.toast_html.layout.visibility = 'visible'
    
    @property
    def css_styles(self):
        """CSS styles for write(..., className = style)."""
        # self.html will be added from Chid class
        return self.html('pre', '''Use any or combinations of these styles in className argument of writing functions:
        className = 'Center'            ------Text------
        className = 'Left'              Text------------
        className = 'Right'             ------------Text
        className = 'RTL'               ------ اردو عربی 
        className = 'Info'              Blue Text
        className = 'Warning'           Orange Text
        className = 'Success'           Green Text
        className = 'Error'             Red Text
        className = 'Note'              Text with info icon
        className = 'slides-only'       Text will not appear in exported html with `build_report`
        className = 'report-only'       Text will not appear on slides. Useful to fill content in report. 
        ''',className= 'Info')

    
    def notify_later(self, title='IPySlides Notification', timeout=5):
        """Decorator to push notification at slide under which it is run. 
        It should return a string that will be content of notifictaion.
        The content is dynamically generated by underlying function, 
        so you can set timer as well. Remains invisible in screenshot through app itself.
        ```python
        @notify_at(title='Notification Title', timeout=5)
        def push_notification():
            time = datetime.now()
            return f'Notification at {time}'
        ```
        """
        def _notify(func): 
            self._toasts[f'{self._current_slide}'] = dict(func = func, kwargs = dict(title=title, timeout=timeout))
        return _notify
    
    def clear_notifications(self):
        "Remove all redundent notifications that show up."
        self._toasts = {} # Free up
    
    @property
    def notifications(self):
        "See all stored notifications."
        return self._toasts
    
    def _display_toast(self):
        toast = self._toasts.get(self._access_key,None) #_access_key is current slide's number from LiveSlides
        if toast:
            # clear previous content of notification as new one is about to be shown, this will ensure not to see on wrong slide
            self.widgets.htmls.toast.value = ''
            self.notify(content = toast['func'](), **toast['kwargs'])
    
    @property
    def md_content(self):
        "Get markdown content from loaded file."
        return self._md_content
        
    
    def from_markdown(self, path, trusted = False):
        """You can create slides from a markdown file or StringIO object as well. It creates slides 1,2,3... in order.
        You should add more slides by higher number than the number of slides in the file, or it will overwrite.
        Slides separator should be --- (three dashes) in start of line.
        Frames separator should be ___ (three underscores) in start of line. All markdown before first ___ will be written on all frames.
        **Markdown File Content**
        ```markdown
        # Talk Title
        ---
        # Slide 1 
        || Inline - Column A || Inline - Column B ||
        {{some_var}} that will be replaced by it's html value.
         ```python run source
         from ipyslides import parsers as prs # import parser functions from this module (1.5.6+)
         # code here will be executed and it's output will be shown in slide.
         ```
         {{source}} from above code block will be replaced by it's html value.
        ---
        # Slide 2
        ___
        ## First Frame
         ```multicol 40 60
        # Block column 1
        +++
        # Block column 2
        || Mini - Column A || Mini - Column B ||
         ```
        ___
        ## Second Frame
        ```
        This will create two slides along with title page. Second slide will have two frames.
        
        Content of each slide from imported file is stored as list in `slides.md_content`. You can append content to it like this:
        ```python
        with slides.slide(2):
            self.parse_xmd(slides.md_content[2]) # Instead of write, parse_xmd take cares of code blocks
            plot_something()
            write_something()
        ```
        
        > Note: With this method you can add more slides besides created ones.
        
        Starting from version 1.6.2, only those slides will be updated whose content is changed from last run of this function. This increases speed.
        """
        if self.shell is None or self.shell.__class__.__name__ == 'TerminalInteractiveShell':
            raise Exception('Python/IPython REPL cannot show slides. Use IPython notebook instead.')
        
        if not trusted:
            if isinstance(path, io.StringIO):
                lines = path.readlines()
            else:
                with open(path, 'r') as f:
                    lines = f.readlines()
                    
            untrusted_lines = []
            for i, line in enumerate(lines, start = 1):
                if re.match(r'{{|```python\s+run', line):
                    untrusted_lines.append(i)
            
            if untrusted_lines:
                raise Exception(f'File {path!r} may contain unsafe code to be executed at lines: {untrusted_lines}'
                    ' Verify code is safe and try again with argument `trusted = True`.'
                    ' Never run files that you did not create yourself or not verified by you.')
        
                    
        self._check_computed('add slides from markdown file')
        if not (isinstance(path, io.StringIO) or os.path.isfile(path)): #check path later or it will throw error
            raise ValueError(f"File {path!r} does not exist or not a io.StringIO object.")
        
        self.convert2slides(True)
        
        if isinstance(path, io.StringIO):
            chunks = _parse_md_file(path)
        else:
            with open(path, 'r') as fp:
                chunks = _parse_md_file(fp)
        
        if hasattr(self,'_loaded_other') and self._loaded_other:
            self.clear()
            self._loaded_other = False
        
        if hasattr(self, '_md_content'):
            if len(self._md_content) > len(chunks):
                self.clear() # If there are more chunks, they will overwrite previous slides automatically, if less then clear previous content
        
        for i,chunk in enumerate(chunks):
            # Must run under this to create frames with triple underscore (___)
            if hasattr(self, '_md_content'):
                if self._md_content[i:] and chunk == self._md_content[i]:
                    pass # Skip if content is same as previous one
                else:
                    self.shell.run_cell_magic('slide', f'{i} -m', chunk)
            else:
                self.shell.run_cell_magic('slide', f'{i} -m', chunk)
            
        self._md_content = chunks # Store for later use
        
        return self
    
    def _clean_markdown_loaded(self):
        "Use in other than from_markdown methods."
        self.clear() # Clear previous content
        if hasattr(self, '_md_content'): # If loaded from markdown file, clear it
            self._md_content = []
        self._loaded_other = True
    
    def demo(self):
        """Demo slides with a variety of content."""
        self._check_computed('load demo')
        self._clean_markdown_loaded()
            
        get_ipython().user_ns['_s_l_i_d_e_s_'] = self
        from .. import _demo # Import after assigning in user_ns
        import importlib
        _demo = importlib.reload(_demo) # Reload is must if other contexts loaded before this
        slides = _demo.slides # or it is self
        with slides.slide(100):
            slides.write('## This is all code to generate slides')
            slides.write(_demo)
            slides.write(self.demo)
        with slides.slide(101,background='#9ACD32'):
            with slides.source.context() as s:
                slides.write_citations()
            s.display()
        
        slides.progress_slider.index = 0 # back to title
        return slides
    
    def load_docs(self):
        "Create presentation from docs of IPySlides."
        self._check_computed('load docs')
        self._clean_markdown_loaded()
        
        from ..core import LiveSlides
        from ..__version__ import __version__
        
        self.settings.set_footer('IPySlides Documentation')
        
        with self.title(): # Title
            self.write(f'## IPySlides {__version__} Documentation\n### Creating slides with IPySlides')
            self.write(self.doc(LiveSlides))
        
        with self.slide(1):
            self.write('## Adding Slides')
            self.write('Besides functions below, you can add slides with `%%title`,  `%%slide <slide number>` and `%%slide <slide number>` -m` magics as well.\n{.Note .Info}')
            self.write([self.doc(self.title,'LiveSlides'),self.doc(self.slide,'LiveSlides'),self.doc(self.frames,'LiveSlides')])
        
        with self.slide(2):
            self.write('## Adding Content')
            self.write('Besides functions below, you can add content to slides with `%%xmd`,`%xmd`, `display(obj)` as well.\n{.Note .Info}')
            self.write([self.doc(self.write,'LiveSlides'),self.doc(self.iwrite,'LiveSlides'), self.doc(self.parse_xmd,'LiveSlides'),self.doc(self.cite,'LiveSlides')])
        
        with self.slide(3):
            self.write('## Adding Speaker Notes')
            self.write('You can use line magic `%notes` to add notes as well.\n{.Note .Success}')
            self.doc(self.notes,'LiveSlides', members = True).display()
                   
        with self.slide(4):
            self.write('## Displaying Source Code')
            self.doc(self.source,'LiveSlides', members=True).display()
        
        with self.slide(5):
            self.write('## Layout and Theme Settings')
            self.doc(self.settings,'LiveSlides', members=True).display()
                
        with self.slide(6):
            self.write('## Useful Functions for Rich Content')
            for attr in dir(self):
                if not attr.startswith('_'):
                    if not attr in ['write','iwrite','parse_xmd','source','export','notes','settings','title','slide','frames','css_styles','load_docs','demo','from_markdown']:
                        with suppress(Exception):
                            if not 'block_' in attr:
                                self.write(self.doc(getattr(self,attr),'LiveSlides'))
                        if attr == 'block':
                            self.write(f"`block` has other shortcut colored versions {', '.join(f'`block_{c}`' for c in 'rgbycmkowp')}.\n{{.Note .Info}}")
        
        with self.slide(7):
            self.write('## Content Styling')
            with self.source.context() as c:
                self.write(('You can **style**{.Error} your *content*{: style="color:hotpink;"} with `className` attribute in writing/content functions. ' 
                       'Provide **CSS**{.Info} for that using `.format_css` or use some of the available styles. '
                       'See these **styles**{.Success} with `.css_styles` property as below:'))
                self.css_styles.display()
                c.display()
        
        with self.slide(8):
            self.write('## Highlighting Code')
            with self.source.context() as s:
                self.write(('You can **highlight**{.Error} code using `highlight` function or within markdown like this: \n'
                        '```python\n'
                        'import ipyslides as isd\n```\n'
                        '```javascript\n'
                        'import React, { Component } from "react";\n```\n'))
                s.display()
        
        with self.slide(9):
            self.write('## Loading from File/Exporting to HTML')
            self.write('You can parse and view a markdown file with `ipyslides.display_markdown` as well. The output you can save by exporting notebook in other formats.\n{.Note .Info}')
            self.write([self.doc(self.from_markdown,'LiveSlides'), 
                        self.doc(self.demo,'LiveSlides'), 
                        self.doc(self.load_docs,'LiveSlides'),
                        self.doc(self.export.slides,'LiveSlides.export'),
                        self.doc(self.export.report,'LiveSlides.export')])
        
        with self.slide(10):
            self.write('## Adding User defined Objects')
            self.write('If you need to serialize your own or third party objects not serialized by this module, you can use `@LiveSlides.serializer.register` to serialize them to html.\n{.Note .Info}')
            self.write(self.doc(self.serializer.register,'LiveSlides.serializer'))
        
        with self.slide(11):
            self.write(['## Presentation Code',self.load_docs])
        
        self.progress_slider.index = 0 # back to title
        return self


def _parse_md_file(fp):
    "Parse a Markdown file or StringIO to put in slides and returns text for title and each slide."
    lines = fp.readlines()
    breaks = [-1] # start, will add +1 next
    for i,line in enumerate(lines):
        if line and line.strip() =='---':
            breaks.append(i)
    breaks.append(len(lines)) # Last one
    
    ranges = [range(j+1,k) for j,k in zip(breaks[:-1],breaks[1:])]
    return [''.join(lines[x.start:x.stop]) for x in ranges]
        