"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ServerlessClamscan = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
const path = require("path");
const aws_ec2_1 = require("monocdk/aws-ec2");
const aws_efs_1 = require("monocdk/aws-efs");
const aws_events_1 = require("monocdk/aws-events");
const aws_events_targets_1 = require("monocdk/aws-events-targets");
const aws_iam_1 = require("monocdk/aws-iam");
const aws_lambda_1 = require("monocdk/aws-lambda");
const aws_lambda_destinations_1 = require("monocdk/aws-lambda-destinations");
const aws_lambda_event_sources_1 = require("monocdk/aws-lambda-event-sources");
const aws_s3_1 = require("monocdk/aws-s3");
const aws_sqs_1 = require("monocdk/aws-sqs");
const monocdk_1 = require("monocdk");
const monocdk_nag_1 = require("monocdk-nag");
/**
  An [aws-cdk](https://github.com/aws/aws-cdk) construct that uses [ClamAV®](https://www.clamav.net/).
  to scan objects in Amazon S3 for viruses. The construct provides a flexible interface for a system
  to act based on the results of a ClamAV virus scan.

  The construct creates a Lambda function with EFS integration to support larger files.
  A VPC with isolated subnets, a S3 Gateway endpoint will also be created.

  Additionally creates an twice-daily job to download the latest ClamAV definition files to the
  Virus Definitions S3 Bucket by utilizing an EventBridge rule and a Lambda function and
  publishes CloudWatch Metrics to the 'serverless-clamscan' namespace.

  __Important O&M__:
  When ClamAV publishes updates to the scanner you will see “Your ClamAV installation is OUTDATED” in your scan results.
  While the construct creates a system to keep the database definitions up to date, you must update the scanner to
  detect all the latest Viruses.

  Update the docker images of the Lambda functions with the latest version of ClamAV by re-running `cdk deploy`.

  Successful Scan Event format
  ```json
  {
     "source": "serverless-clamscan",
     "input_bucket": <input_bucket_name>,
     "input_key": <object_key>,
     "status": <"CLEAN"|"INFECTED"|"N/A">,
     "message": <scan_summary>,
   }
  ```

  Note: The Virus Definitions bucket policy will likely cause a deletion error if you choose to delete
  the stack associated in the construct. However since the bucket itself gets deleted, you can delete
  the stack again to resolve the error.
 */
class ServerlessClamscan extends monocdk_1.Construct {
    /**
     * Creates a ServerlessClamscan construct.
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param props A `ServerlessClamscanProps` interface.
     */
    constructor(scope, id, props) {
        var _b, _c;
        super(scope, id);
        this._efsRootPath = '/lambda';
        this._efsMountPath = `/mnt${this._efsRootPath}`;
        this._efsDefsPath = 'virus_database/';
        if (!props.onResult) {
            this.resultBus = new aws_events_1.EventBus(this, 'ScanResultBus');
            this.resultDest = new aws_lambda_destinations_1.EventBridgeDestination(this.resultBus);
            this.infectedRule = new aws_events_1.Rule(this, 'InfectedRule', {
                eventBus: this.resultBus,
                description: 'Event for when a file is marked INFECTED',
                eventPattern: {
                    detail: {
                        responsePayload: {
                            source: ['serverless-clamscan'],
                            status: ['INFECTED'],
                        },
                    },
                },
            });
            this.cleanRule = new aws_events_1.Rule(this, 'CleanRule', {
                eventBus: this.resultBus,
                description: 'Event for when a file is marked CLEAN',
                eventPattern: {
                    detail: {
                        responsePayload: {
                            source: ['serverless-clamscan'],
                            status: ['CLEAN'],
                        },
                    },
                },
            });
        }
        else {
            this.resultDest = props.onResult;
        }
        if (!props.onError) {
            this.errorDeadLetterQueue = new aws_sqs_1.Queue(this, 'ScanErrorDeadLetterQueue', {
                encryption: aws_sqs_1.QueueEncryption.KMS_MANAGED,
            });
            this.errorDeadLetterQueue.addToResourcePolicy(new aws_iam_1.PolicyStatement({
                actions: ['sqs:*'],
                effect: aws_iam_1.Effect.DENY,
                principals: [new aws_iam_1.AnyPrincipal()],
                conditions: { Bool: { 'aws:SecureTransport': false } },
                resources: [this.errorDeadLetterQueue.queueArn],
            }));
            this.errorQueue = new aws_sqs_1.Queue(this, 'ScanErrorQueue', {
                encryption: aws_sqs_1.QueueEncryption.KMS_MANAGED,
                deadLetterQueue: {
                    maxReceiveCount: 3,
                    queue: this.errorDeadLetterQueue,
                },
            });
            this.errorQueue.addToResourcePolicy(new aws_iam_1.PolicyStatement({
                actions: ['sqs:*'],
                effect: aws_iam_1.Effect.DENY,
                principals: [new aws_iam_1.AnyPrincipal()],
                conditions: { Bool: { 'aws:SecureTransport': false } },
                resources: [this.errorQueue.queueArn],
            }));
            this.errorDest = new aws_lambda_destinations_1.SqsDestination(this.errorQueue);
            monocdk_nag_1.NagSuppressions.addResourceSuppressions(this.errorDeadLetterQueue, [
                { id: 'AwsSolutions-SQS3', reason: 'This queue is a DLQ.' },
            ]);
        }
        else {
            this.errorDest = props.onError;
        }
        const vpc = new aws_ec2_1.Vpc(this, 'ScanVPC', {
            subnetConfiguration: [
                {
                    subnetType: aws_ec2_1.SubnetType.PRIVATE_ISOLATED,
                    name: 'Isolated',
                },
            ],
        });
        vpc.addFlowLog('FlowLogs');
        this._s3Gw = vpc.addGatewayEndpoint('S3Endpoint', {
            service: aws_ec2_1.GatewayVpcEndpointAwsService.S3,
        });
        const fileSystem = new aws_efs_1.FileSystem(this, 'ScanFileSystem', {
            vpc: vpc,
            encrypted: props.efsEncryption === false ? false : true,
            lifecyclePolicy: aws_efs_1.LifecyclePolicy.AFTER_7_DAYS,
            performanceMode: aws_efs_1.PerformanceMode.GENERAL_PURPOSE,
            removalPolicy: monocdk_1.RemovalPolicy.DESTROY,
            securityGroup: new aws_ec2_1.SecurityGroup(this, 'ScanFileSystemSecurityGroup', {
                vpc: vpc,
                allowAllOutbound: false,
            }),
        });
        const lambda_ap = fileSystem.addAccessPoint('ScanLambdaAP', {
            createAcl: {
                ownerGid: '1000',
                ownerUid: '1000',
                permissions: '755',
            },
            posixUser: {
                gid: '1000',
                uid: '1000',
            },
            path: this._efsRootPath,
        });
        const logs_bucket = (_b = props.defsBucketAccessLogsConfig) === null || _b === void 0 ? void 0 : _b.logsBucket;
        const logs_bucket_prefix = (_c = props.defsBucketAccessLogsConfig) === null || _c === void 0 ? void 0 : _c.logsPrefix;
        if (logs_bucket === true || logs_bucket === undefined) {
            this.defsAccessLogsBucket = new aws_s3_1.Bucket(this, 'VirusDefsAccessLogsBucket', {
                encryption: aws_s3_1.BucketEncryption.S3_MANAGED,
                removalPolicy: monocdk_1.RemovalPolicy.RETAIN,
                serverAccessLogsPrefix: 'access-logs-bucket-logs',
                blockPublicAccess: {
                    blockPublicAcls: true,
                    blockPublicPolicy: true,
                    ignorePublicAcls: true,
                    restrictPublicBuckets: true,
                },
            });
            this.defsAccessLogsBucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
                effect: aws_iam_1.Effect.DENY,
                actions: ['s3:*'],
                resources: [
                    this.defsAccessLogsBucket.arnForObjects('*'),
                    this.defsAccessLogsBucket.bucketArn,
                ],
                principals: [new aws_iam_1.AnyPrincipal()],
                conditions: {
                    Bool: {
                        'aws:SecureTransport': false,
                    },
                },
            }));
        }
        else if (logs_bucket != false) {
            this.defsAccessLogsBucket = logs_bucket;
        }
        const defs_bucket = new aws_s3_1.Bucket(this, 'VirusDefsBucket', {
            encryption: aws_s3_1.BucketEncryption.S3_MANAGED,
            removalPolicy: monocdk_1.RemovalPolicy.DESTROY,
            autoDeleteObjects: true,
            serverAccessLogsBucket: this.defsAccessLogsBucket,
            serverAccessLogsPrefix: logs_bucket === false ? undefined : logs_bucket_prefix,
            blockPublicAccess: {
                blockPublicAcls: true,
                blockPublicPolicy: true,
                ignorePublicAcls: true,
                restrictPublicBuckets: true,
            },
        });
        defs_bucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.DENY,
            actions: ['s3:*'],
            resources: [defs_bucket.arnForObjects('*'), defs_bucket.bucketArn],
            principals: [new aws_iam_1.AnyPrincipal()],
            conditions: {
                Bool: {
                    'aws:SecureTransport': false,
                },
            },
        }));
        defs_bucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.ALLOW,
            actions: ['s3:GetObject', 's3:ListBucket'],
            resources: [defs_bucket.arnForObjects('*'), defs_bucket.bucketArn],
            principals: [new aws_iam_1.AnyPrincipal()],
            conditions: {
                StringEquals: {
                    'aws:SourceVpce': this._s3Gw.vpcEndpointId,
                },
            },
        }));
        defs_bucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.DENY,
            actions: ['s3:PutBucketPolicy', 's3:DeleteBucketPolicy'],
            resources: [defs_bucket.bucketArn],
            notPrincipals: [new aws_iam_1.AccountRootPrincipal()],
        }));
        this._s3Gw.addToPolicy(new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.ALLOW,
            actions: ['s3:GetObject', 's3:ListBucket'],
            resources: [defs_bucket.arnForObjects('*'), defs_bucket.bucketArn],
            principals: [new aws_iam_1.AnyPrincipal()],
        }));
        this._scanFunction = new aws_lambda_1.DockerImageFunction(this, 'ServerlessClamscan', {
            code: aws_lambda_1.DockerImageCode.fromImageAsset(path.join(__dirname, '../assets/lambda/code/scan'), {
                buildArgs: {
                    // Only force update the docker layer cache once a day
                    CACHE_DATE: new Date().toDateString(),
                },
                extraHash: Date.now().toString(),
            }),
            onSuccess: this.resultDest,
            onFailure: this.errorDest,
            filesystem: aws_lambda_1.FileSystem.fromEfsAccessPoint(lambda_ap, this._efsMountPath),
            vpc: vpc,
            vpcSubnets: { subnets: vpc.isolatedSubnets },
            allowAllOutbound: false,
            timeout: monocdk_1.Duration.minutes(15),
            memorySize: 10240,
            reservedConcurrentExecutions: props.reservedConcurrency,
            environment: {
                EFS_MOUNT_PATH: this._efsMountPath,
                EFS_DEF_PATH: this._efsDefsPath,
                DEFS_URL: defs_bucket.virtualHostedUrlForObject(),
                POWERTOOLS_METRICS_NAMESPACE: 'serverless-clamscan',
                POWERTOOLS_SERVICE_NAME: 'virus-scan',
            },
        });
        if (this._scanFunction.role) {
            monocdk_nag_1.NagSuppressions.addResourceSuppressions(this._scanFunction.role, [
                {
                    id: 'AwsSolutions-IAM4',
                    reason: 'The AWSLambdaBasicExecutionRole does not provide permissions beyond uploading logs to CloudWatch. The AWSLambdaVPCAccessExecutionRole is required for functions with VPC access to manage elastic network interfaces.',
                },
            ]);
            monocdk_nag_1.NagSuppressions.addResourceSuppressions(this._scanFunction.role, [
                {
                    id: 'AwsSolutions-IAM5',
                    reason: 'The EFS mount point permissions are controlled through a condition which limit the scope of the * resources.',
                },
            ], true);
        }
        this._scanFunction.connections.allowToAnyIpv4(aws_ec2_1.Port.tcp(443), 'Allow outbound HTTPS traffic for S3 access.');
        defs_bucket.grantRead(this._scanFunction);
        const download_defs = new aws_lambda_1.DockerImageFunction(this, 'DownloadDefs', {
            code: aws_lambda_1.DockerImageCode.fromImageAsset(path.join(__dirname, '../assets/lambda/code/download_defs'), {
                buildArgs: {
                    // Only force update the docker layer cache once a day
                    CACHE_DATE: new Date().toDateString(),
                },
                extraHash: Date.now().toString(),
            }),
            timeout: monocdk_1.Duration.minutes(5),
            memorySize: 1024,
            environment: {
                DEFS_BUCKET: defs_bucket.bucketName,
                POWERTOOLS_SERVICE_NAME: 'freshclam-update',
            },
        });
        const stack = monocdk_1.Stack.of(this);
        if (download_defs.role) {
            const download_defs_role = `arn:${stack.partition}:sts::${stack.account}:assumed-role/${download_defs.role.roleName}/${download_defs.functionName}`;
            const download_defs_assumed_principal = new aws_iam_1.ArnPrincipal(download_defs_role);
            defs_bucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
                effect: aws_iam_1.Effect.DENY,
                actions: ['s3:PutObject*'],
                resources: [defs_bucket.arnForObjects('*')],
                notPrincipals: [download_defs.role, download_defs_assumed_principal],
            }));
            defs_bucket.grantReadWrite(download_defs);
            monocdk_nag_1.NagSuppressions.addResourceSuppressions(download_defs.role, [
                {
                    id: 'AwsSolutions-IAM4',
                    reason: 'The AWSLambdaBasicExecutionRole does not provide permissions beyond uploading logs to CloudWatch.',
                },
            ]);
            monocdk_nag_1.NagSuppressions.addResourceSuppressions(download_defs.role, [
                {
                    id: 'AwsSolutions-IAM5',
                    reason: 'The function is allowed to invoke the download defs Lambda function.',
                },
            ], true);
        }
        new aws_events_1.Rule(this, 'VirusDefsUpdateRule', {
            schedule: aws_events_1.Schedule.rate(monocdk_1.Duration.hours(12)),
            targets: [new aws_events_targets_1.LambdaFunction(download_defs)],
        });
        const init_defs_cr = new aws_lambda_1.Function(this, 'InitDefs', {
            runtime: aws_lambda_1.Runtime.PYTHON_3_8,
            code: aws_lambda_1.Code.fromAsset(path.join(__dirname, '../assets/lambda/code/initialize_defs_cr')),
            handler: 'lambda.lambda_handler',
            timeout: monocdk_1.Duration.minutes(5),
        });
        download_defs.grantInvoke(init_defs_cr);
        if (init_defs_cr.role) {
            monocdk_nag_1.NagSuppressions.addResourceSuppressions(init_defs_cr.role, [
                {
                    id: 'AwsSolutions-IAM4',
                    reason: 'The AWSLambdaBasicExecutionRole does not provide permissions beyond uploading logs to CloudWatch.',
                },
                {
                    id: 'AwsSolutions-IAM5',
                    reason: 'The AWSLambdaBasicExecutionRole does not provide permissions beyond uploading logs to CloudWatch.',
                },
            ], true);
        }
        new monocdk_1.CustomResource(this, 'InitDefsCr', {
            serviceToken: init_defs_cr.functionArn,
            properties: {
                FnName: download_defs.functionName,
            },
        });
        if (props.buckets) {
            props.buckets.forEach((bucket) => {
                this.addSourceBucket(bucket);
            });
        }
    }
    /**
     * Sets the specified S3 Bucket as a s3:ObjectCreate* for the ClamAV function.
       Grants the ClamAV function permissions to get and tag objects.
       Adds a bucket policy to disallow GetObject operations on files that are tagged 'IN PROGRESS', 'INFECTED', or 'ERROR'.
     * @param bucket The bucket to add the scanning bucket policy and s3:ObjectCreate* trigger to.
     */
    addSourceBucket(bucket) {
        this._scanFunction.addEventSource(new aws_lambda_event_sources_1.S3EventSource(bucket, { events: [aws_s3_1.EventType.OBJECT_CREATED] }));
        bucket.grantRead(this._scanFunction);
        this._scanFunction.addToRolePolicy(new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.ALLOW,
            actions: ['s3:PutObjectTagging', 's3:PutObjectVersionTagging'],
            resources: [bucket.arnForObjects('*')],
        }));
        if (this._scanFunction.role) {
            const stack = monocdk_1.Stack.of(this);
            const scan_assumed_role = `arn:${stack.partition}:sts::${stack.account}:assumed-role/${this._scanFunction.role.roleName}/${this._scanFunction.functionName}`;
            const scan_assumed_principal = new aws_iam_1.ArnPrincipal(scan_assumed_role);
            this._s3Gw.addToPolicy(new aws_iam_1.PolicyStatement({
                effect: aws_iam_1.Effect.ALLOW,
                actions: ['s3:GetObject*', 's3:GetBucket*', 's3:List*'],
                resources: [bucket.bucketArn, bucket.arnForObjects('*')],
                principals: [this._scanFunction.role, scan_assumed_principal],
            }));
            this._s3Gw.addToPolicy(new aws_iam_1.PolicyStatement({
                effect: aws_iam_1.Effect.ALLOW,
                actions: ['s3:PutObjectTagging', 's3:PutObjectVersionTagging'],
                resources: [bucket.arnForObjects('*')],
                principals: [this._scanFunction.role, scan_assumed_principal],
            }));
            // Need the assumed role for the not Principal Action with Lambda
            bucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
                effect: aws_iam_1.Effect.DENY,
                actions: ['s3:GetObject'],
                resources: [bucket.arnForObjects('*')],
                notPrincipals: [this._scanFunction.role, scan_assumed_principal],
                conditions: {
                    StringEquals: {
                        's3:ExistingObjectTag/scan-status': [
                            'IN PROGRESS',
                            'INFECTED',
                            'ERROR',
                        ],
                    },
                },
            }));
        }
    }
}
exports.ServerlessClamscan = ServerlessClamscan;
_a = JSII_RTTI_SYMBOL_1;
ServerlessClamscan[_a] = { fqn: "monocdk-serverless-clamscan.ServerlessClamscan", version: "1.1.51" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaW5kZXguanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi9zcmMvaW5kZXgudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7QUFBQSxxRUFBcUU7QUFDckUsc0NBQXNDO0FBRXRDLDZCQUE2QjtBQUM3Qiw2Q0FPeUI7QUFDekIsNkNBQStFO0FBQy9FLG1EQUE4RDtBQUM5RCxtRUFBNEQ7QUFDNUQsNkNBTXlCO0FBQ3pCLG1EQVE0QjtBQUM1Qiw2RUFHeUM7QUFDekMsK0VBQWlFO0FBQ2pFLDJDQUE4RTtBQUM5RSw2Q0FBeUQ7QUFDekQscUNBTWlCO0FBQ2pCLDZDQUE4QztBQThDOUM7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7OztHQWlDRztBQUNILE1BQWEsa0JBQW1CLFNBQVEsbUJBQVM7SUErQy9DOzs7OztPQUtHO0lBQ0gsWUFBWSxLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUE4Qjs7UUFDdEUsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztRQVhYLGlCQUFZLEdBQUcsU0FBUyxDQUFDO1FBQ3pCLGtCQUFhLEdBQUcsT0FBTyxJQUFJLENBQUMsWUFBWSxFQUFFLENBQUM7UUFDM0MsaUJBQVksR0FBRyxpQkFBaUIsQ0FBQztRQVd2QyxJQUFJLENBQUMsS0FBSyxDQUFDLFFBQVEsRUFBRTtZQUNuQixJQUFJLENBQUMsU0FBUyxHQUFHLElBQUkscUJBQVEsQ0FBQyxJQUFJLEVBQUUsZUFBZSxDQUFDLENBQUM7WUFDckQsSUFBSSxDQUFDLFVBQVUsR0FBRyxJQUFJLGdEQUFzQixDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsQ0FBQztZQUM3RCxJQUFJLENBQUMsWUFBWSxHQUFHLElBQUksaUJBQUksQ0FBQyxJQUFJLEVBQUUsY0FBYyxFQUFFO2dCQUNqRCxRQUFRLEVBQUUsSUFBSSxDQUFDLFNBQVM7Z0JBQ3hCLFdBQVcsRUFBRSwwQ0FBMEM7Z0JBQ3ZELFlBQVksRUFBRTtvQkFDWixNQUFNLEVBQUU7d0JBQ04sZUFBZSxFQUFFOzRCQUNmLE1BQU0sRUFBRSxDQUFDLHFCQUFxQixDQUFDOzRCQUMvQixNQUFNLEVBQUUsQ0FBQyxVQUFVLENBQUM7eUJBQ3JCO3FCQUNGO2lCQUNGO2FBQ0YsQ0FBQyxDQUFDO1lBQ0gsSUFBSSxDQUFDLFNBQVMsR0FBRyxJQUFJLGlCQUFJLENBQUMsSUFBSSxFQUFFLFdBQVcsRUFBRTtnQkFDM0MsUUFBUSxFQUFFLElBQUksQ0FBQyxTQUFTO2dCQUN4QixXQUFXLEVBQUUsdUNBQXVDO2dCQUNwRCxZQUFZLEVBQUU7b0JBQ1osTUFBTSxFQUFFO3dCQUNOLGVBQWUsRUFBRTs0QkFDZixNQUFNLEVBQUUsQ0FBQyxxQkFBcUIsQ0FBQzs0QkFDL0IsTUFBTSxFQUFFLENBQUMsT0FBTyxDQUFDO3lCQUNsQjtxQkFDRjtpQkFDRjthQUNGLENBQUMsQ0FBQztTQUNKO2FBQU07WUFDTCxJQUFJLENBQUMsVUFBVSxHQUFHLEtBQUssQ0FBQyxRQUFRLENBQUM7U0FDbEM7UUFFRCxJQUFJLENBQUMsS0FBSyxDQUFDLE9BQU8sRUFBRTtZQUNsQixJQUFJLENBQUMsb0JBQW9CLEdBQUcsSUFBSSxlQUFLLENBQUMsSUFBSSxFQUFFLDBCQUEwQixFQUFFO2dCQUN0RSxVQUFVLEVBQUUseUJBQWUsQ0FBQyxXQUFXO2FBQ3hDLENBQUMsQ0FBQztZQUNILElBQUksQ0FBQyxvQkFBb0IsQ0FBQyxtQkFBbUIsQ0FDM0MsSUFBSSx5QkFBZSxDQUFDO2dCQUNsQixPQUFPLEVBQUUsQ0FBQyxPQUFPLENBQUM7Z0JBQ2xCLE1BQU0sRUFBRSxnQkFBTSxDQUFDLElBQUk7Z0JBQ25CLFVBQVUsRUFBRSxDQUFDLElBQUksc0JBQVksRUFBRSxDQUFDO2dCQUNoQyxVQUFVLEVBQUUsRUFBRSxJQUFJLEVBQUUsRUFBRSxxQkFBcUIsRUFBRSxLQUFLLEVBQUUsRUFBRTtnQkFDdEQsU0FBUyxFQUFFLENBQUMsSUFBSSxDQUFDLG9CQUFvQixDQUFDLFFBQVEsQ0FBQzthQUNoRCxDQUFDLENBQ0gsQ0FBQztZQUNGLElBQUksQ0FBQyxVQUFVLEdBQUcsSUFBSSxlQUFLLENBQUMsSUFBSSxFQUFFLGdCQUFnQixFQUFFO2dCQUNsRCxVQUFVLEVBQUUseUJBQWUsQ0FBQyxXQUFXO2dCQUN2QyxlQUFlLEVBQUU7b0JBQ2YsZUFBZSxFQUFFLENBQUM7b0JBQ2xCLEtBQUssRUFBRSxJQUFJLENBQUMsb0JBQW9CO2lCQUNqQzthQUNGLENBQUMsQ0FBQztZQUNILElBQUksQ0FBQyxVQUFVLENBQUMsbUJBQW1CLENBQ2pDLElBQUkseUJBQWUsQ0FBQztnQkFDbEIsT0FBTyxFQUFFLENBQUMsT0FBTyxDQUFDO2dCQUNsQixNQUFNLEVBQUUsZ0JBQU0sQ0FBQyxJQUFJO2dCQUNuQixVQUFVLEVBQUUsQ0FBQyxJQUFJLHNCQUFZLEVBQUUsQ0FBQztnQkFDaEMsVUFBVSxFQUFFLEVBQUUsSUFBSSxFQUFFLEVBQUUscUJBQXFCLEVBQUUsS0FBSyxFQUFFLEVBQUU7Z0JBQ3RELFNBQVMsRUFBRSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsUUFBUSxDQUFDO2FBQ3RDLENBQUMsQ0FDSCxDQUFDO1lBQ0YsSUFBSSxDQUFDLFNBQVMsR0FBRyxJQUFJLHdDQUFjLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxDQUFDO1lBQ3JELDZCQUFlLENBQUMsdUJBQXVCLENBQUMsSUFBSSxDQUFDLG9CQUFvQixFQUFFO2dCQUNqRSxFQUFFLEVBQUUsRUFBRSxtQkFBbUIsRUFBRSxNQUFNLEVBQUUsc0JBQXNCLEVBQUU7YUFDNUQsQ0FBQyxDQUFDO1NBQ0o7YUFBTTtZQUNMLElBQUksQ0FBQyxTQUFTLEdBQUcsS0FBSyxDQUFDLE9BQU8sQ0FBQztTQUNoQztRQUVELE1BQU0sR0FBRyxHQUFHLElBQUksYUFBRyxDQUFDLElBQUksRUFBRSxTQUFTLEVBQUU7WUFDbkMsbUJBQW1CLEVBQUU7Z0JBQ25CO29CQUNFLFVBQVUsRUFBRSxvQkFBVSxDQUFDLGdCQUFnQjtvQkFDdkMsSUFBSSxFQUFFLFVBQVU7aUJBQ2pCO2FBQ0Y7U0FDRixDQUFDLENBQUM7UUFFSCxHQUFHLENBQUMsVUFBVSxDQUFDLFVBQVUsQ0FBQyxDQUFDO1FBRTNCLElBQUksQ0FBQyxLQUFLLEdBQUcsR0FBRyxDQUFDLGtCQUFrQixDQUFDLFlBQVksRUFBRTtZQUNoRCxPQUFPLEVBQUUsc0NBQTRCLENBQUMsRUFBRTtTQUN6QyxDQUFDLENBQUM7UUFFSCxNQUFNLFVBQVUsR0FBRyxJQUFJLG9CQUFVLENBQUMsSUFBSSxFQUFFLGdCQUFnQixFQUFFO1lBQ3hELEdBQUcsRUFBRSxHQUFHO1lBQ1IsU0FBUyxFQUFFLEtBQUssQ0FBQyxhQUFhLEtBQUssS0FBSyxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLElBQUk7WUFDdkQsZUFBZSxFQUFFLHlCQUFlLENBQUMsWUFBWTtZQUM3QyxlQUFlLEVBQUUseUJBQWUsQ0FBQyxlQUFlO1lBQ2hELGFBQWEsRUFBRSx1QkFBYSxDQUFDLE9BQU87WUFDcEMsYUFBYSxFQUFFLElBQUksdUJBQWEsQ0FBQyxJQUFJLEVBQUUsNkJBQTZCLEVBQUU7Z0JBQ3BFLEdBQUcsRUFBRSxHQUFHO2dCQUNSLGdCQUFnQixFQUFFLEtBQUs7YUFDeEIsQ0FBQztTQUNILENBQUMsQ0FBQztRQUVILE1BQU0sU0FBUyxHQUFHLFVBQVUsQ0FBQyxjQUFjLENBQUMsY0FBYyxFQUFFO1lBQzFELFNBQVMsRUFBRTtnQkFDVCxRQUFRLEVBQUUsTUFBTTtnQkFDaEIsUUFBUSxFQUFFLE1BQU07Z0JBQ2hCLFdBQVcsRUFBRSxLQUFLO2FBQ25CO1lBQ0QsU0FBUyxFQUFFO2dCQUNULEdBQUcsRUFBRSxNQUFNO2dCQUNYLEdBQUcsRUFBRSxNQUFNO2FBQ1o7WUFDRCxJQUFJLEVBQUUsSUFBSSxDQUFDLFlBQVk7U0FDeEIsQ0FBQyxDQUFDO1FBRUgsTUFBTSxXQUFXLFNBQUcsS0FBSyxDQUFDLDBCQUEwQiwwQ0FBRSxVQUFVLENBQUM7UUFDakUsTUFBTSxrQkFBa0IsU0FBRyxLQUFLLENBQUMsMEJBQTBCLDBDQUFFLFVBQVUsQ0FBQztRQUN4RSxJQUFJLFdBQVcsS0FBSyxJQUFJLElBQUksV0FBVyxLQUFLLFNBQVMsRUFBRTtZQUNyRCxJQUFJLENBQUMsb0JBQW9CLEdBQUcsSUFBSSxlQUFNLENBQ3BDLElBQUksRUFDSiwyQkFBMkIsRUFDM0I7Z0JBQ0UsVUFBVSxFQUFFLHlCQUFnQixDQUFDLFVBQVU7Z0JBQ3ZDLGFBQWEsRUFBRSx1QkFBYSxDQUFDLE1BQU07Z0JBQ25DLHNCQUFzQixFQUFFLHlCQUF5QjtnQkFDakQsaUJBQWlCLEVBQUU7b0JBQ2pCLGVBQWUsRUFBRSxJQUFJO29CQUNyQixpQkFBaUIsRUFBRSxJQUFJO29CQUN2QixnQkFBZ0IsRUFBRSxJQUFJO29CQUN0QixxQkFBcUIsRUFBRSxJQUFJO2lCQUM1QjthQUNGLENBQ0YsQ0FBQztZQUNGLElBQUksQ0FBQyxvQkFBb0IsQ0FBQyxtQkFBbUIsQ0FDM0MsSUFBSSx5QkFBZSxDQUFDO2dCQUNsQixNQUFNLEVBQUUsZ0JBQU0sQ0FBQyxJQUFJO2dCQUNuQixPQUFPLEVBQUUsQ0FBQyxNQUFNLENBQUM7Z0JBQ2pCLFNBQVMsRUFBRTtvQkFDVCxJQUFJLENBQUMsb0JBQW9CLENBQUMsYUFBYSxDQUFDLEdBQUcsQ0FBQztvQkFDNUMsSUFBSSxDQUFDLG9CQUFvQixDQUFDLFNBQVM7aUJBQ3BDO2dCQUNELFVBQVUsRUFBRSxDQUFDLElBQUksc0JBQVksRUFBRSxDQUFDO2dCQUNoQyxVQUFVLEVBQUU7b0JBQ1YsSUFBSSxFQUFFO3dCQUNKLHFCQUFxQixFQUFFLEtBQUs7cUJBQzdCO2lCQUNGO2FBQ0YsQ0FBQyxDQUNILENBQUM7U0FDSDthQUFNLElBQUksV0FBVyxJQUFJLEtBQUssRUFBRTtZQUMvQixJQUFJLENBQUMsb0JBQW9CLEdBQUcsV0FBVyxDQUFDO1NBQ3pDO1FBRUQsTUFBTSxXQUFXLEdBQUcsSUFBSSxlQUFNLENBQUMsSUFBSSxFQUFFLGlCQUFpQixFQUFFO1lBQ3RELFVBQVUsRUFBRSx5QkFBZ0IsQ0FBQyxVQUFVO1lBQ3ZDLGFBQWEsRUFBRSx1QkFBYSxDQUFDLE9BQU87WUFDcEMsaUJBQWlCLEVBQUUsSUFBSTtZQUN2QixzQkFBc0IsRUFBRSxJQUFJLENBQUMsb0JBQW9CO1lBQ2pELHNCQUFzQixFQUNwQixXQUFXLEtBQUssS0FBSyxDQUFDLENBQUMsQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDLGtCQUFrQjtZQUN4RCxpQkFBaUIsRUFBRTtnQkFDakIsZUFBZSxFQUFFLElBQUk7Z0JBQ3JCLGlCQUFpQixFQUFFLElBQUk7Z0JBQ3ZCLGdCQUFnQixFQUFFLElBQUk7Z0JBQ3RCLHFCQUFxQixFQUFFLElBQUk7YUFDNUI7U0FDRixDQUFDLENBQUM7UUFFSCxXQUFXLENBQUMsbUJBQW1CLENBQzdCLElBQUkseUJBQWUsQ0FBQztZQUNsQixNQUFNLEVBQUUsZ0JBQU0sQ0FBQyxJQUFJO1lBQ25CLE9BQU8sRUFBRSxDQUFDLE1BQU0sQ0FBQztZQUNqQixTQUFTLEVBQUUsQ0FBQyxXQUFXLENBQUMsYUFBYSxDQUFDLEdBQUcsQ0FBQyxFQUFFLFdBQVcsQ0FBQyxTQUFTLENBQUM7WUFDbEUsVUFBVSxFQUFFLENBQUMsSUFBSSxzQkFBWSxFQUFFLENBQUM7WUFDaEMsVUFBVSxFQUFFO2dCQUNWLElBQUksRUFBRTtvQkFDSixxQkFBcUIsRUFBRSxLQUFLO2lCQUM3QjthQUNGO1NBQ0YsQ0FBQyxDQUNILENBQUM7UUFDRixXQUFXLENBQUMsbUJBQW1CLENBQzdCLElBQUkseUJBQWUsQ0FBQztZQUNsQixNQUFNLEVBQUUsZ0JBQU0sQ0FBQyxLQUFLO1lBQ3BCLE9BQU8sRUFBRSxDQUFDLGNBQWMsRUFBRSxlQUFlLENBQUM7WUFDMUMsU0FBUyxFQUFFLENBQUMsV0FBVyxDQUFDLGFBQWEsQ0FBQyxHQUFHLENBQUMsRUFBRSxXQUFXLENBQUMsU0FBUyxDQUFDO1lBQ2xFLFVBQVUsRUFBRSxDQUFDLElBQUksc0JBQVksRUFBRSxDQUFDO1lBQ2hDLFVBQVUsRUFBRTtnQkFDVixZQUFZLEVBQUU7b0JBQ1osZ0JBQWdCLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxhQUFhO2lCQUMzQzthQUNGO1NBQ0YsQ0FBQyxDQUNILENBQUM7UUFDRixXQUFXLENBQUMsbUJBQW1CLENBQzdCLElBQUkseUJBQWUsQ0FBQztZQUNsQixNQUFNLEVBQUUsZ0JBQU0sQ0FBQyxJQUFJO1lBQ25CLE9BQU8sRUFBRSxDQUFDLG9CQUFvQixFQUFFLHVCQUF1QixDQUFDO1lBQ3hELFNBQVMsRUFBRSxDQUFDLFdBQVcsQ0FBQyxTQUFTLENBQUM7WUFDbEMsYUFBYSxFQUFFLENBQUMsSUFBSSw4QkFBb0IsRUFBRSxDQUFDO1NBQzVDLENBQUMsQ0FDSCxDQUFDO1FBQ0YsSUFBSSxDQUFDLEtBQUssQ0FBQyxXQUFXLENBQ3BCLElBQUkseUJBQWUsQ0FBQztZQUNsQixNQUFNLEVBQUUsZ0JBQU0sQ0FBQyxLQUFLO1lBQ3BCLE9BQU8sRUFBRSxDQUFDLGNBQWMsRUFBRSxlQUFlLENBQUM7WUFDMUMsU0FBUyxFQUFFLENBQUMsV0FBVyxDQUFDLGFBQWEsQ0FBQyxHQUFHLENBQUMsRUFBRSxXQUFXLENBQUMsU0FBUyxDQUFDO1lBQ2xFLFVBQVUsRUFBRSxDQUFDLElBQUksc0JBQVksRUFBRSxDQUFDO1NBQ2pDLENBQUMsQ0FDSCxDQUFDO1FBRUYsSUFBSSxDQUFDLGFBQWEsR0FBRyxJQUFJLGdDQUFtQixDQUFDLElBQUksRUFBRSxvQkFBb0IsRUFBRTtZQUN2RSxJQUFJLEVBQUUsNEJBQWUsQ0FBQyxjQUFjLENBQ2xDLElBQUksQ0FBQyxJQUFJLENBQUMsU0FBUyxFQUFFLDRCQUE0QixDQUFDLEVBQ2xEO2dCQUNFLFNBQVMsRUFBRTtvQkFDVCxzREFBc0Q7b0JBQ3RELFVBQVUsRUFBRSxJQUFJLElBQUksRUFBRSxDQUFDLFlBQVksRUFBRTtpQkFDdEM7Z0JBQ0QsU0FBUyxFQUFFLElBQUksQ0FBQyxHQUFHLEVBQUUsQ0FBQyxRQUFRLEVBQUU7YUFDakMsQ0FDRjtZQUNELFNBQVMsRUFBRSxJQUFJLENBQUMsVUFBVTtZQUMxQixTQUFTLEVBQUUsSUFBSSxDQUFDLFNBQVM7WUFDekIsVUFBVSxFQUFFLHVCQUFnQixDQUFDLGtCQUFrQixDQUM3QyxTQUFTLEVBQ1QsSUFBSSxDQUFDLGFBQWEsQ0FDbkI7WUFDRCxHQUFHLEVBQUUsR0FBRztZQUNSLFVBQVUsRUFBRSxFQUFFLE9BQU8sRUFBRSxHQUFHLENBQUMsZUFBZSxFQUFFO1lBQzVDLGdCQUFnQixFQUFFLEtBQUs7WUFDdkIsT0FBTyxFQUFFLGtCQUFRLENBQUMsT0FBTyxDQUFDLEVBQUUsQ0FBQztZQUM3QixVQUFVLEVBQUUsS0FBSztZQUNqQiw0QkFBNEIsRUFBRSxLQUFLLENBQUMsbUJBQW1CO1lBQ3ZELFdBQVcsRUFBRTtnQkFDWCxjQUFjLEVBQUUsSUFBSSxDQUFDLGFBQWE7Z0JBQ2xDLFlBQVksRUFBRSxJQUFJLENBQUMsWUFBWTtnQkFDL0IsUUFBUSxFQUFFLFdBQVcsQ0FBQyx5QkFBeUIsRUFBRTtnQkFDakQsNEJBQTRCLEVBQUUscUJBQXFCO2dCQUNuRCx1QkFBdUIsRUFBRSxZQUFZO2FBQ3RDO1NBQ0YsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxJQUFJLENBQUMsYUFBYSxDQUFDLElBQUksRUFBRTtZQUMzQiw2QkFBZSxDQUFDLHVCQUF1QixDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsSUFBSSxFQUFFO2dCQUMvRDtvQkFDRSxFQUFFLEVBQUUsbUJBQW1CO29CQUN2QixNQUFNLEVBQ0osdU5BQXVOO2lCQUMxTjthQUNGLENBQUMsQ0FBQztZQUNILDZCQUFlLENBQUMsdUJBQXVCLENBQ3JDLElBQUksQ0FBQyxhQUFhLENBQUMsSUFBSSxFQUN2QjtnQkFDRTtvQkFDRSxFQUFFLEVBQUUsbUJBQW1CO29CQUN2QixNQUFNLEVBQ0osOEdBQThHO2lCQUNqSDthQUNGLEVBQ0QsSUFBSSxDQUNMLENBQUM7U0FDSDtRQUNELElBQUksQ0FBQyxhQUFhLENBQUMsV0FBVyxDQUFDLGNBQWMsQ0FDM0MsY0FBSSxDQUFDLEdBQUcsQ0FBQyxHQUFHLENBQUMsRUFDYiw2Q0FBNkMsQ0FDOUMsQ0FBQztRQUNGLFdBQVcsQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxDQUFDO1FBRTFDLE1BQU0sYUFBYSxHQUFHLElBQUksZ0NBQW1CLENBQUMsSUFBSSxFQUFFLGNBQWMsRUFBRTtZQUNsRSxJQUFJLEVBQUUsNEJBQWUsQ0FBQyxjQUFjLENBQ2xDLElBQUksQ0FBQyxJQUFJLENBQUMsU0FBUyxFQUFFLHFDQUFxQyxDQUFDLEVBQzNEO2dCQUNFLFNBQVMsRUFBRTtvQkFDVCxzREFBc0Q7b0JBQ3RELFVBQVUsRUFBRSxJQUFJLElBQUksRUFBRSxDQUFDLFlBQVksRUFBRTtpQkFDdEM7Z0JBQ0QsU0FBUyxFQUFFLElBQUksQ0FBQyxHQUFHLEVBQUUsQ0FBQyxRQUFRLEVBQUU7YUFDakMsQ0FDRjtZQUNELE9BQU8sRUFBRSxrQkFBUSxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUM7WUFDNUIsVUFBVSxFQUFFLElBQUk7WUFDaEIsV0FBVyxFQUFFO2dCQUNYLFdBQVcsRUFBRSxXQUFXLENBQUMsVUFBVTtnQkFDbkMsdUJBQXVCLEVBQUUsa0JBQWtCO2FBQzVDO1NBQ0YsQ0FBQyxDQUFDO1FBQ0gsTUFBTSxLQUFLLEdBQUcsZUFBSyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUU3QixJQUFJLGFBQWEsQ0FBQyxJQUFJLEVBQUU7WUFDdEIsTUFBTSxrQkFBa0IsR0FBRyxPQUFPLEtBQUssQ0FBQyxTQUFTLFNBQVMsS0FBSyxDQUFDLE9BQU8saUJBQWlCLGFBQWEsQ0FBQyxJQUFJLENBQUMsUUFBUSxJQUFJLGFBQWEsQ0FBQyxZQUFZLEVBQUUsQ0FBQztZQUNwSixNQUFNLCtCQUErQixHQUFHLElBQUksc0JBQVksQ0FDdEQsa0JBQWtCLENBQ25CLENBQUM7WUFDRixXQUFXLENBQUMsbUJBQW1CLENBQzdCLElBQUkseUJBQWUsQ0FBQztnQkFDbEIsTUFBTSxFQUFFLGdCQUFNLENBQUMsSUFBSTtnQkFDbkIsT0FBTyxFQUFFLENBQUMsZUFBZSxDQUFDO2dCQUMxQixTQUFTLEVBQUUsQ0FBQyxXQUFXLENBQUMsYUFBYSxDQUFDLEdBQUcsQ0FBQyxDQUFDO2dCQUMzQyxhQUFhLEVBQUUsQ0FBQyxhQUFhLENBQUMsSUFBSSxFQUFFLCtCQUErQixDQUFDO2FBQ3JFLENBQUMsQ0FDSCxDQUFDO1lBQ0YsV0FBVyxDQUFDLGNBQWMsQ0FBQyxhQUFhLENBQUMsQ0FBQztZQUMxQyw2QkFBZSxDQUFDLHVCQUF1QixDQUFDLGFBQWEsQ0FBQyxJQUFJLEVBQUU7Z0JBQzFEO29CQUNFLEVBQUUsRUFBRSxtQkFBbUI7b0JBQ3ZCLE1BQU0sRUFDSixtR0FBbUc7aUJBQ3RHO2FBQ0YsQ0FBQyxDQUFDO1lBQ0gsNkJBQWUsQ0FBQyx1QkFBdUIsQ0FDckMsYUFBYSxDQUFDLElBQUksRUFDbEI7Z0JBQ0U7b0JBQ0UsRUFBRSxFQUFFLG1CQUFtQjtvQkFDdkIsTUFBTSxFQUNKLHNFQUFzRTtpQkFDekU7YUFDRixFQUNELElBQUksQ0FDTCxDQUFDO1NBQ0g7UUFFRCxJQUFJLGlCQUFJLENBQUMsSUFBSSxFQUFFLHFCQUFxQixFQUFFO1lBQ3BDLFFBQVEsRUFBRSxxQkFBUSxDQUFDLElBQUksQ0FBQyxrQkFBUSxDQUFDLEtBQUssQ0FBQyxFQUFFLENBQUMsQ0FBQztZQUMzQyxPQUFPLEVBQUUsQ0FBQyxJQUFJLG1DQUFjLENBQUMsYUFBYSxDQUFDLENBQUM7U0FDN0MsQ0FBQyxDQUFDO1FBRUgsTUFBTSxZQUFZLEdBQUcsSUFBSSxxQkFBUSxDQUFDLElBQUksRUFBRSxVQUFVLEVBQUU7WUFDbEQsT0FBTyxFQUFFLG9CQUFPLENBQUMsVUFBVTtZQUMzQixJQUFJLEVBQUUsaUJBQUksQ0FBQyxTQUFTLENBQ2xCLElBQUksQ0FBQyxJQUFJLENBQUMsU0FBUyxFQUFFLDBDQUEwQyxDQUFDLENBQ2pFO1lBQ0QsT0FBTyxFQUFFLHVCQUF1QjtZQUNoQyxPQUFPLEVBQUUsa0JBQVEsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDO1NBQzdCLENBQUMsQ0FBQztRQUNILGFBQWEsQ0FBQyxXQUFXLENBQUMsWUFBWSxDQUFDLENBQUM7UUFDeEMsSUFBSSxZQUFZLENBQUMsSUFBSSxFQUFFO1lBQ3JCLDZCQUFlLENBQUMsdUJBQXVCLENBQ3JDLFlBQVksQ0FBQyxJQUFJLEVBQ2pCO2dCQUNFO29CQUNFLEVBQUUsRUFBRSxtQkFBbUI7b0JBQ3ZCLE1BQU0sRUFDSixtR0FBbUc7aUJBQ3RHO2dCQUNEO29CQUNFLEVBQUUsRUFBRSxtQkFBbUI7b0JBQ3ZCLE1BQU0sRUFDSixtR0FBbUc7aUJBQ3RHO2FBQ0YsRUFDRCxJQUFJLENBQ0wsQ0FBQztTQUNIO1FBQ0QsSUFBSSx3QkFBYyxDQUFDLElBQUksRUFBRSxZQUFZLEVBQUU7WUFDckMsWUFBWSxFQUFFLFlBQVksQ0FBQyxXQUFXO1lBQ3RDLFVBQVUsRUFBRTtnQkFDVixNQUFNLEVBQUUsYUFBYSxDQUFDLFlBQVk7YUFDbkM7U0FDRixDQUFDLENBQUM7UUFFSCxJQUFJLEtBQUssQ0FBQyxPQUFPLEVBQUU7WUFDakIsS0FBSyxDQUFDLE9BQU8sQ0FBQyxPQUFPLENBQUMsQ0FBQyxNQUFNLEVBQUUsRUFBRTtnQkFDL0IsSUFBSSxDQUFDLGVBQWUsQ0FBQyxNQUFNLENBQUMsQ0FBQztZQUMvQixDQUFDLENBQUMsQ0FBQztTQUNKO0lBQ0gsQ0FBQztJQUVEOzs7OztPQUtHO0lBQ0gsZUFBZSxDQUFDLE1BQWM7UUFDNUIsSUFBSSxDQUFDLGFBQWEsQ0FBQyxjQUFjLENBQy9CLElBQUksd0NBQWEsQ0FBQyxNQUFNLEVBQUUsRUFBRSxNQUFNLEVBQUUsQ0FBQyxrQkFBUyxDQUFDLGNBQWMsQ0FBQyxFQUFFLENBQUMsQ0FDbEUsQ0FBQztRQUNGLE1BQU0sQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxDQUFDO1FBQ3JDLElBQUksQ0FBQyxhQUFhLENBQUMsZUFBZSxDQUNoQyxJQUFJLHlCQUFlLENBQUM7WUFDbEIsTUFBTSxFQUFFLGdCQUFNLENBQUMsS0FBSztZQUNwQixPQUFPLEVBQUUsQ0FBQyxxQkFBcUIsRUFBRSw0QkFBNEIsQ0FBQztZQUM5RCxTQUFTLEVBQUUsQ0FBQyxNQUFNLENBQUMsYUFBYSxDQUFDLEdBQUcsQ0FBQyxDQUFDO1NBQ3ZDLENBQUMsQ0FDSCxDQUFDO1FBRUYsSUFBSSxJQUFJLENBQUMsYUFBYSxDQUFDLElBQUksRUFBRTtZQUMzQixNQUFNLEtBQUssR0FBRyxlQUFLLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDO1lBQzdCLE1BQU0saUJBQWlCLEdBQUcsT0FBTyxLQUFLLENBQUMsU0FBUyxTQUFTLEtBQUssQ0FBQyxPQUFPLGlCQUFpQixJQUFJLENBQUMsYUFBYSxDQUFDLElBQUksQ0FBQyxRQUFRLElBQUksSUFBSSxDQUFDLGFBQWEsQ0FBQyxZQUFZLEVBQUUsQ0FBQztZQUM3SixNQUFNLHNCQUFzQixHQUFHLElBQUksc0JBQVksQ0FBQyxpQkFBaUIsQ0FBQyxDQUFDO1lBQ25FLElBQUksQ0FBQyxLQUFLLENBQUMsV0FBVyxDQUNwQixJQUFJLHlCQUFlLENBQUM7Z0JBQ2xCLE1BQU0sRUFBRSxnQkFBTSxDQUFDLEtBQUs7Z0JBQ3BCLE9BQU8sRUFBRSxDQUFDLGVBQWUsRUFBRSxlQUFlLEVBQUUsVUFBVSxDQUFDO2dCQUN2RCxTQUFTLEVBQUUsQ0FBQyxNQUFNLENBQUMsU0FBUyxFQUFFLE1BQU0sQ0FBQyxhQUFhLENBQUMsR0FBRyxDQUFDLENBQUM7Z0JBQ3hELFVBQVUsRUFBRSxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsSUFBSSxFQUFFLHNCQUFzQixDQUFDO2FBQzlELENBQUMsQ0FDSCxDQUFDO1lBQ0YsSUFBSSxDQUFDLEtBQUssQ0FBQyxXQUFXLENBQ3BCLElBQUkseUJBQWUsQ0FBQztnQkFDbEIsTUFBTSxFQUFFLGdCQUFNLENBQUMsS0FBSztnQkFDcEIsT0FBTyxFQUFFLENBQUMscUJBQXFCLEVBQUUsNEJBQTRCLENBQUM7Z0JBQzlELFNBQVMsRUFBRSxDQUFDLE1BQU0sQ0FBQyxhQUFhLENBQUMsR0FBRyxDQUFDLENBQUM7Z0JBQ3RDLFVBQVUsRUFBRSxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsSUFBSSxFQUFFLHNCQUFzQixDQUFDO2FBQzlELENBQUMsQ0FDSCxDQUFDO1lBRUYsaUVBQWlFO1lBQ2pFLE1BQU0sQ0FBQyxtQkFBbUIsQ0FDeEIsSUFBSSx5QkFBZSxDQUFDO2dCQUNsQixNQUFNLEVBQUUsZ0JBQU0sQ0FBQyxJQUFJO2dCQUNuQixPQUFPLEVBQUUsQ0FBQyxjQUFjLENBQUM7Z0JBQ3pCLFNBQVMsRUFBRSxDQUFDLE1BQU0sQ0FBQyxhQUFhLENBQUMsR0FBRyxDQUFDLENBQUM7Z0JBQ3RDLGFBQWEsRUFBRSxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsSUFBSSxFQUFFLHNCQUFzQixDQUFDO2dCQUNoRSxVQUFVLEVBQUU7b0JBQ1YsWUFBWSxFQUFFO3dCQUNaLGtDQUFrQyxFQUFFOzRCQUNsQyxhQUFhOzRCQUNiLFVBQVU7NEJBQ1YsT0FBTzt5QkFDUjtxQkFDRjtpQkFDRjthQUNGLENBQUMsQ0FDSCxDQUFDO1NBQ0g7SUFDSCxDQUFDOztBQTVkSCxnREE2ZEMiLCJzb3VyY2VzQ29udGVudCI6WyIvLyBDb3B5cmlnaHQgQW1hem9uLmNvbSwgSW5jLiBvciBpdHMgYWZmaWxpYXRlcy4gQWxsIFJpZ2h0cyBSZXNlcnZlZC5cbi8vIFNQRFgtTGljZW5zZS1JZGVudGlmaWVyOiBBcGFjaGUtMi4wXG5cbmltcG9ydCAqIGFzIHBhdGggZnJvbSAncGF0aCc7XG5pbXBvcnQge1xuICBWcGMsXG4gIFN1Ym5ldFR5cGUsXG4gIEdhdGV3YXlWcGNFbmRwb2ludCxcbiAgR2F0ZXdheVZwY0VuZHBvaW50QXdzU2VydmljZSxcbiAgUG9ydCxcbiAgU2VjdXJpdHlHcm91cCxcbn0gZnJvbSAnbW9ub2Nkay9hd3MtZWMyJztcbmltcG9ydCB7IEZpbGVTeXN0ZW0sIExpZmVjeWNsZVBvbGljeSwgUGVyZm9ybWFuY2VNb2RlIH0gZnJvbSAnbW9ub2Nkay9hd3MtZWZzJztcbmltcG9ydCB7IEV2ZW50QnVzLCBSdWxlLCBTY2hlZHVsZSB9IGZyb20gJ21vbm9jZGsvYXdzLWV2ZW50cyc7XG5pbXBvcnQgeyBMYW1iZGFGdW5jdGlvbiB9IGZyb20gJ21vbm9jZGsvYXdzLWV2ZW50cy10YXJnZXRzJztcbmltcG9ydCB7XG4gIEVmZmVjdCxcbiAgUG9saWN5U3RhdGVtZW50LFxuICBBcm5QcmluY2lwYWwsXG4gIEFueVByaW5jaXBhbCxcbiAgQWNjb3VudFJvb3RQcmluY2lwYWwsXG59IGZyb20gJ21vbm9jZGsvYXdzLWlhbSc7XG5pbXBvcnQge1xuICBEb2NrZXJJbWFnZUNvZGUsXG4gIERvY2tlckltYWdlRnVuY3Rpb24sXG4gIEZ1bmN0aW9uLFxuICBJRGVzdGluYXRpb24sXG4gIEZpbGVTeXN0ZW0gYXMgTGFtYmRhRmlsZVN5c3RlbSxcbiAgUnVudGltZSxcbiAgQ29kZSxcbn0gZnJvbSAnbW9ub2Nkay9hd3MtbGFtYmRhJztcbmltcG9ydCB7XG4gIEV2ZW50QnJpZGdlRGVzdGluYXRpb24sXG4gIFNxc0Rlc3RpbmF0aW9uLFxufSBmcm9tICdtb25vY2RrL2F3cy1sYW1iZGEtZGVzdGluYXRpb25zJztcbmltcG9ydCB7IFMzRXZlbnRTb3VyY2UgfSBmcm9tICdtb25vY2RrL2F3cy1sYW1iZGEtZXZlbnQtc291cmNlcyc7XG5pbXBvcnQgeyBJQnVja2V0LCBCdWNrZXQsIEJ1Y2tldEVuY3J5cHRpb24sIEV2ZW50VHlwZSB9IGZyb20gJ21vbm9jZGsvYXdzLXMzJztcbmltcG9ydCB7IFF1ZXVlLCBRdWV1ZUVuY3J5cHRpb24gfSBmcm9tICdtb25vY2RrL2F3cy1zcXMnO1xuaW1wb3J0IHtcbiAgQ29uc3RydWN0LFxuICBEdXJhdGlvbixcbiAgQ3VzdG9tUmVzb3VyY2UsXG4gIFJlbW92YWxQb2xpY3ksXG4gIFN0YWNrLFxufSBmcm9tICdtb25vY2RrJztcbmltcG9ydCB7IE5hZ1N1cHByZXNzaW9ucyB9IGZyb20gJ21vbm9jZGstbmFnJztcbi8qKlxuICogSW50ZXJmYWNlIGZvciBTZXJ2ZXJsZXNzQ2xhbXNjYW4gVmlydXMgRGVmaW5pdGlvbnMgUzMgQnVja2V0IExvZ2dpbmcuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgU2VydmVybGVzc0NsYW1zY2FuTG9nZ2luZ1Byb3BzIHtcbiAgLyoqXG4gICAqIERlc3RpbmF0aW9uIGJ1Y2tldCBmb3IgdGhlIHNlcnZlciBhY2Nlc3MgbG9ncyAoRGVmYXVsdDogQ3JlYXRlcyBhIG5ldyBTMyBCdWNrZXQgZm9yIGFjY2VzcyBsb2dzICkuXG4gICAqL1xuICByZWFkb25seSBsb2dzQnVja2V0PzogYm9vbGVhbiB8IElCdWNrZXQ7XG4gIC8qKlxuICAgKiBPcHRpb25hbCBsb2cgZmlsZSBwcmVmaXggdG8gdXNlIGZvciB0aGUgYnVja2V0J3MgYWNjZXNzIGxvZ3MsIG9wdGlvbiBpcyBpZ25vcmVkIGlmIGxvZ3NfYnVja2V0IGlzIHNldCB0byBmYWxzZS5cbiAgICovXG4gIHJlYWRvbmx5IGxvZ3NQcmVmaXg/OiBzdHJpbmc7XG59XG5cbi8qKlxuICogSW50ZXJmYWNlIGZvciBjcmVhdGluZyBhIFNlcnZlcmxlc3NDbGFtc2Nhbi5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBTZXJ2ZXJsZXNzQ2xhbXNjYW5Qcm9wcyB7XG4gIC8qKlxuICAgKiBBbiBvcHRpb25hbCBsaXN0IG9mIFMzIGJ1Y2tldHMgdG8gY29uZmlndXJlIGZvciBDbGFtQVYgVmlydXMgU2Nhbm5pbmc7IGJ1Y2tldHMgY2FuIGJlIGFkZGVkIGxhdGVyIGJ5IGNhbGxpbmcgYWRkU291cmNlQnVja2V0LlxuICAgKi9cbiAgcmVhZG9ubHkgYnVja2V0cz86IEJ1Y2tldFtdO1xuICAvKipcbiAgICogT3B0aW9uYWxseSBzZXQgYSByZXNlcnZlZCBjb25jdXJyZW5jeSBmb3IgdGhlIHZpcnVzIHNjYW5uaW5nIExhbWJkYS5cbiAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGFtYmRhL2xhdGVzdC9vcGVyYXRvcmd1aWRlL3Jlc2VydmVkLWNvbmN1cnJlbmN5Lmh0bWxcbiAgICovXG4gIHJlYWRvbmx5IHJlc2VydmVkQ29uY3VycmVuY3k/OiBudW1iZXI7XG4gIC8qKlxuICAgKiBUaGUgTGFtYmRhIERlc3RpbmF0aW9uIGZvciBmaWxlcyBtYXJrZWQgJ0NMRUFOJyBvciAnSU5GRUNURUQnIGJhc2VkIG9uIHRoZSBDbGFtQVYgVmlydXMgc2NhbiBvciAnTi9BJyBmb3Igc2NhbnMgdHJpZ2dlcmVkIGJ5IFMzIGZvbGRlciBjcmVhdGlvbiBldmVudHMgbWFya2VkIChEZWZhdWx0OiBDcmVhdGVzIGFuZCBwdWJsaXNoZXMgdG8gYSBuZXcgRXZlbnQgQnJpZGdlIEJ1cyBpZiB1bnNwZWNpZmllZCkuXG4gICAqL1xuICByZWFkb25seSBvblJlc3VsdD86IElEZXN0aW5hdGlvbjtcbiAgLyoqXG4gICAqIFRoZSBMYW1iZGEgRGVzdGluYXRpb24gZm9yIGZpbGVzIHRoYXQgZmFpbCB0byBzY2FuIGFuZCBhcmUgbWFya2VkICdFUlJPUicgb3Igc3R1Y2sgJ0lOIFBST0dSRVNTJyBkdWUgdG8gYSBMYW1iZGEgdGltZW91dCAoRGVmYXVsdDogQ3JlYXRlcyBhbmQgcHVibGlzaGVzIHRvIGEgbmV3IFNRUyBxdWV1ZSBpZiB1bnNwZWNpZmllZCkuXG4gICAqL1xuICByZWFkb25seSBvbkVycm9yPzogSURlc3RpbmF0aW9uO1xuICAvKipcbiAgICogV2hldGhlciBvciBub3QgdG8gZW5hYmxlIGVuY3J5cHRpb24gb24gRUZTIGZpbGVzeXN0ZW0gKERlZmF1bHQ6IGVuYWJsZWQpLlxuICAgKi9cbiAgcmVhZG9ubHkgZWZzRW5jcnlwdGlvbj86IGJvb2xlYW47XG4gIC8qKlxuICAgKiBXaGV0aGVyIG9yIG5vdCB0byBlbmFibGUgQWNjZXNzIExvZ2dpbmcgZm9yIHRoZSBWaXJ1cyBEZWZpbml0aW9ucyBidWNrZXQsIHlvdSBjYW4gc3BlY2lmeSBhbiBleGlzdGluZyBidWNrZXQgYW5kIHByZWZpeCAoRGVmYXVsdDogQ3JlYXRlcyBhIG5ldyBTMyBCdWNrZXQgZm9yIGFjY2VzcyBsb2dzICkuXG4gICAqL1xuICByZWFkb25seSBkZWZzQnVja2V0QWNjZXNzTG9nc0NvbmZpZz86IFNlcnZlcmxlc3NDbGFtc2NhbkxvZ2dpbmdQcm9wcztcbn1cblxuLyoqXG4gIEFuIFthd3MtY2RrXShodHRwczovL2dpdGh1Yi5jb20vYXdzL2F3cy1jZGspIGNvbnN0cnVjdCB0aGF0IHVzZXMgW0NsYW1BVsKuXShodHRwczovL3d3dy5jbGFtYXYubmV0LykuXG4gIHRvIHNjYW4gb2JqZWN0cyBpbiBBbWF6b24gUzMgZm9yIHZpcnVzZXMuIFRoZSBjb25zdHJ1Y3QgcHJvdmlkZXMgYSBmbGV4aWJsZSBpbnRlcmZhY2UgZm9yIGEgc3lzdGVtXG4gIHRvIGFjdCBiYXNlZCBvbiB0aGUgcmVzdWx0cyBvZiBhIENsYW1BViB2aXJ1cyBzY2FuLlxuXG4gIFRoZSBjb25zdHJ1Y3QgY3JlYXRlcyBhIExhbWJkYSBmdW5jdGlvbiB3aXRoIEVGUyBpbnRlZ3JhdGlvbiB0byBzdXBwb3J0IGxhcmdlciBmaWxlcy5cbiAgQSBWUEMgd2l0aCBpc29sYXRlZCBzdWJuZXRzLCBhIFMzIEdhdGV3YXkgZW5kcG9pbnQgd2lsbCBhbHNvIGJlIGNyZWF0ZWQuXG5cbiAgQWRkaXRpb25hbGx5IGNyZWF0ZXMgYW4gdHdpY2UtZGFpbHkgam9iIHRvIGRvd25sb2FkIHRoZSBsYXRlc3QgQ2xhbUFWIGRlZmluaXRpb24gZmlsZXMgdG8gdGhlXG4gIFZpcnVzIERlZmluaXRpb25zIFMzIEJ1Y2tldCBieSB1dGlsaXppbmcgYW4gRXZlbnRCcmlkZ2UgcnVsZSBhbmQgYSBMYW1iZGEgZnVuY3Rpb24gYW5kXG4gIHB1Ymxpc2hlcyBDbG91ZFdhdGNoIE1ldHJpY3MgdG8gdGhlICdzZXJ2ZXJsZXNzLWNsYW1zY2FuJyBuYW1lc3BhY2UuXG5cbiAgX19JbXBvcnRhbnQgTyZNX186XG4gIFdoZW4gQ2xhbUFWIHB1Ymxpc2hlcyB1cGRhdGVzIHRvIHRoZSBzY2FubmVyIHlvdSB3aWxsIHNlZSDigJxZb3VyIENsYW1BViBpbnN0YWxsYXRpb24gaXMgT1VUREFURUTigJ0gaW4geW91ciBzY2FuIHJlc3VsdHMuXG4gIFdoaWxlIHRoZSBjb25zdHJ1Y3QgY3JlYXRlcyBhIHN5c3RlbSB0byBrZWVwIHRoZSBkYXRhYmFzZSBkZWZpbml0aW9ucyB1cCB0byBkYXRlLCB5b3UgbXVzdCB1cGRhdGUgdGhlIHNjYW5uZXIgdG9cbiAgZGV0ZWN0IGFsbCB0aGUgbGF0ZXN0IFZpcnVzZXMuXG5cbiAgVXBkYXRlIHRoZSBkb2NrZXIgaW1hZ2VzIG9mIHRoZSBMYW1iZGEgZnVuY3Rpb25zIHdpdGggdGhlIGxhdGVzdCB2ZXJzaW9uIG9mIENsYW1BViBieSByZS1ydW5uaW5nIGBjZGsgZGVwbG95YC5cblxuICBTdWNjZXNzZnVsIFNjYW4gRXZlbnQgZm9ybWF0XG4gIGBgYGpzb25cbiAge1xuICAgICBcInNvdXJjZVwiOiBcInNlcnZlcmxlc3MtY2xhbXNjYW5cIixcbiAgICAgXCJpbnB1dF9idWNrZXRcIjogPGlucHV0X2J1Y2tldF9uYW1lPixcbiAgICAgXCJpbnB1dF9rZXlcIjogPG9iamVjdF9rZXk+LFxuICAgICBcInN0YXR1c1wiOiA8XCJDTEVBTlwifFwiSU5GRUNURURcInxcIk4vQVwiPixcbiAgICAgXCJtZXNzYWdlXCI6IDxzY2FuX3N1bW1hcnk+LFxuICAgfVxuICBgYGBcblxuICBOb3RlOiBUaGUgVmlydXMgRGVmaW5pdGlvbnMgYnVja2V0IHBvbGljeSB3aWxsIGxpa2VseSBjYXVzZSBhIGRlbGV0aW9uIGVycm9yIGlmIHlvdSBjaG9vc2UgdG8gZGVsZXRlXG4gIHRoZSBzdGFjayBhc3NvY2lhdGVkIGluIHRoZSBjb25zdHJ1Y3QuIEhvd2V2ZXIgc2luY2UgdGhlIGJ1Y2tldCBpdHNlbGYgZ2V0cyBkZWxldGVkLCB5b3UgY2FuIGRlbGV0ZVxuICB0aGUgc3RhY2sgYWdhaW4gdG8gcmVzb2x2ZSB0aGUgZXJyb3IuXG4gKi9cbmV4cG9ydCBjbGFzcyBTZXJ2ZXJsZXNzQ2xhbXNjYW4gZXh0ZW5kcyBDb25zdHJ1Y3Qge1xuICAvKipcbiAgICBUaGUgTGFtYmRhIERlc3RpbmF0aW9uIGZvciBmYWlsZWQgb24gZXJyZWQgc2NhbnMgW0VSUk9SLCBJTiBQUk9HUkVTUyAoSWYgZXJyb3IgaXMgZHVlIHRvIExhbWJkYSB0aW1lb3V0KV0uXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgZXJyb3JEZXN0OiBJRGVzdGluYXRpb247XG5cbiAgLyoqXG4gICAgVGhlIExhbWJkYSBEZXN0aW5hdGlvbiBmb3IgY29tcGxldGVkIENsYW1BViBzY2FucyBbQ0xFQU4sIElORkVDVEVEXS5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSByZXN1bHREZXN0OiBJRGVzdGluYXRpb247XG5cbiAgLyoqXG4gICAgQ29uZGl0aW9uYWw6IFRoZSBTUVMgUXVldWUgZm9yIGVycmVkIHNjYW5zIGlmIGEgZmFpbHVyZSAob25FcnJvcikgZGVzdGluYXRpb24gd2FzIG5vdCBzcGVjaWZpZWQuXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgZXJyb3JRdWV1ZT86IFF1ZXVlO1xuXG4gIC8qKlxuICAgIENvbmRpdGlvbmFsOiBUaGUgU1FTIERlYWQgTGV0dGVyIFF1ZXVlIGZvciB0aGUgZXJyb3JRdWV1ZSBpZiBhIGZhaWx1cmUgKG9uRXJyb3IpIGRlc3RpbmF0aW9uIHdhcyBub3Qgc3BlY2lmaWVkLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IGVycm9yRGVhZExldHRlclF1ZXVlPzogUXVldWU7XG5cbiAgLyoqXG4gICAgQ29uZGl0aW9uYWw6IFRoZSBFdmVudCBCcmlkZ2UgQnVzIGZvciBjb21wbGV0ZWQgQ2xhbUFWIHNjYW5zIGlmIGEgc3VjY2VzcyAob25SZXN1bHQpIGRlc3RpbmF0aW9uIHdhcyBub3Qgc3BlY2lmaWVkLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IHJlc3VsdEJ1cz86IEV2ZW50QnVzO1xuXG4gIC8qKlxuICAgIENvbmRpdGlvbmFsOiBBbiBFdmVudCBCcmlkZ2UgUnVsZSBmb3IgZmlsZXMgdGhhdCBhcmUgbWFya2VkICdDTEVBTicgYnkgQ2xhbUFWIGlmIGEgc3VjY2VzcyBkZXN0aW5hdGlvbiB3YXMgbm90IHNwZWNpZmllZC5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBjbGVhblJ1bGU/OiBSdWxlO1xuXG4gIC8qKlxuICAgIENvbmRpdGlvbmFsOiBBbiBFdmVudCBCcmlkZ2UgUnVsZSBmb3IgZmlsZXMgdGhhdCBhcmUgbWFya2VkICdJTkZFQ1RFRCcgYnkgQ2xhbUFWIGlmIGEgc3VjY2VzcyBkZXN0aW5hdGlvbiB3YXMgbm90IHNwZWNpZmllZC5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBpbmZlY3RlZFJ1bGU/OiBSdWxlO1xuXG4gIC8qKlxuICAgIENvbmRpdGlvbmFsOiBUaGUgQnVja2V0IGZvciBhY2Nlc3MgbG9ncyBmb3IgdGhlIHZpcnVzIGRlZmluaXRpb25zIGJ1Y2tldCBpZiBsb2dnaW5nIGlzIGVuYWJsZWQgKGRlZnNCdWNrZXRBY2Nlc3NMb2dzQ29uZmlnKS5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBkZWZzQWNjZXNzTG9nc0J1Y2tldD86IElCdWNrZXQ7XG5cbiAgcHJpdmF0ZSBfc2NhbkZ1bmN0aW9uOiBEb2NrZXJJbWFnZUZ1bmN0aW9uO1xuICBwcml2YXRlIF9zM0d3OiBHYXRld2F5VnBjRW5kcG9pbnQ7XG4gIHByaXZhdGUgX2Vmc1Jvb3RQYXRoID0gJy9sYW1iZGEnO1xuICBwcml2YXRlIF9lZnNNb3VudFBhdGggPSBgL21udCR7dGhpcy5fZWZzUm9vdFBhdGh9YDtcbiAgcHJpdmF0ZSBfZWZzRGVmc1BhdGggPSAndmlydXNfZGF0YWJhc2UvJztcblxuICAvKipcbiAgICogQ3JlYXRlcyBhIFNlcnZlcmxlc3NDbGFtc2NhbiBjb25zdHJ1Y3QuXG4gICAqIEBwYXJhbSBzY29wZSBUaGUgcGFyZW50IGNyZWF0aW5nIGNvbnN0cnVjdCAodXN1YWxseSBgdGhpc2ApLlxuICAgKiBAcGFyYW0gaWQgVGhlIGNvbnN0cnVjdCdzIG5hbWUuXG4gICAqIEBwYXJhbSBwcm9wcyBBIGBTZXJ2ZXJsZXNzQ2xhbXNjYW5Qcm9wc2AgaW50ZXJmYWNlLlxuICAgKi9cbiAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IFNlcnZlcmxlc3NDbGFtc2NhblByb3BzKSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkKTtcblxuICAgIGlmICghcHJvcHMub25SZXN1bHQpIHtcbiAgICAgIHRoaXMucmVzdWx0QnVzID0gbmV3IEV2ZW50QnVzKHRoaXMsICdTY2FuUmVzdWx0QnVzJyk7XG4gICAgICB0aGlzLnJlc3VsdERlc3QgPSBuZXcgRXZlbnRCcmlkZ2VEZXN0aW5hdGlvbih0aGlzLnJlc3VsdEJ1cyk7XG4gICAgICB0aGlzLmluZmVjdGVkUnVsZSA9IG5ldyBSdWxlKHRoaXMsICdJbmZlY3RlZFJ1bGUnLCB7XG4gICAgICAgIGV2ZW50QnVzOiB0aGlzLnJlc3VsdEJ1cyxcbiAgICAgICAgZGVzY3JpcHRpb246ICdFdmVudCBmb3Igd2hlbiBhIGZpbGUgaXMgbWFya2VkIElORkVDVEVEJyxcbiAgICAgICAgZXZlbnRQYXR0ZXJuOiB7XG4gICAgICAgICAgZGV0YWlsOiB7XG4gICAgICAgICAgICByZXNwb25zZVBheWxvYWQ6IHtcbiAgICAgICAgICAgICAgc291cmNlOiBbJ3NlcnZlcmxlc3MtY2xhbXNjYW4nXSxcbiAgICAgICAgICAgICAgc3RhdHVzOiBbJ0lORkVDVEVEJ10sXG4gICAgICAgICAgICB9LFxuICAgICAgICAgIH0sXG4gICAgICAgIH0sXG4gICAgICB9KTtcbiAgICAgIHRoaXMuY2xlYW5SdWxlID0gbmV3IFJ1bGUodGhpcywgJ0NsZWFuUnVsZScsIHtcbiAgICAgICAgZXZlbnRCdXM6IHRoaXMucmVzdWx0QnVzLFxuICAgICAgICBkZXNjcmlwdGlvbjogJ0V2ZW50IGZvciB3aGVuIGEgZmlsZSBpcyBtYXJrZWQgQ0xFQU4nLFxuICAgICAgICBldmVudFBhdHRlcm46IHtcbiAgICAgICAgICBkZXRhaWw6IHtcbiAgICAgICAgICAgIHJlc3BvbnNlUGF5bG9hZDoge1xuICAgICAgICAgICAgICBzb3VyY2U6IFsnc2VydmVybGVzcy1jbGFtc2NhbiddLFxuICAgICAgICAgICAgICBzdGF0dXM6IFsnQ0xFQU4nXSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgfSxcbiAgICAgICAgfSxcbiAgICAgIH0pO1xuICAgIH0gZWxzZSB7XG4gICAgICB0aGlzLnJlc3VsdERlc3QgPSBwcm9wcy5vblJlc3VsdDtcbiAgICB9XG5cbiAgICBpZiAoIXByb3BzLm9uRXJyb3IpIHtcbiAgICAgIHRoaXMuZXJyb3JEZWFkTGV0dGVyUXVldWUgPSBuZXcgUXVldWUodGhpcywgJ1NjYW5FcnJvckRlYWRMZXR0ZXJRdWV1ZScsIHtcbiAgICAgICAgZW5jcnlwdGlvbjogUXVldWVFbmNyeXB0aW9uLktNU19NQU5BR0VELFxuICAgICAgfSk7XG4gICAgICB0aGlzLmVycm9yRGVhZExldHRlclF1ZXVlLmFkZFRvUmVzb3VyY2VQb2xpY3koXG4gICAgICAgIG5ldyBQb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgICAgIGFjdGlvbnM6IFsnc3FzOionXSxcbiAgICAgICAgICBlZmZlY3Q6IEVmZmVjdC5ERU5ZLFxuICAgICAgICAgIHByaW5jaXBhbHM6IFtuZXcgQW55UHJpbmNpcGFsKCldLFxuICAgICAgICAgIGNvbmRpdGlvbnM6IHsgQm9vbDogeyAnYXdzOlNlY3VyZVRyYW5zcG9ydCc6IGZhbHNlIH0gfSxcbiAgICAgICAgICByZXNvdXJjZXM6IFt0aGlzLmVycm9yRGVhZExldHRlclF1ZXVlLnF1ZXVlQXJuXSxcbiAgICAgICAgfSksXG4gICAgICApO1xuICAgICAgdGhpcy5lcnJvclF1ZXVlID0gbmV3IFF1ZXVlKHRoaXMsICdTY2FuRXJyb3JRdWV1ZScsIHtcbiAgICAgICAgZW5jcnlwdGlvbjogUXVldWVFbmNyeXB0aW9uLktNU19NQU5BR0VELFxuICAgICAgICBkZWFkTGV0dGVyUXVldWU6IHtcbiAgICAgICAgICBtYXhSZWNlaXZlQ291bnQ6IDMsXG4gICAgICAgICAgcXVldWU6IHRoaXMuZXJyb3JEZWFkTGV0dGVyUXVldWUsXG4gICAgICAgIH0sXG4gICAgICB9KTtcbiAgICAgIHRoaXMuZXJyb3JRdWV1ZS5hZGRUb1Jlc291cmNlUG9saWN5KFxuICAgICAgICBuZXcgUG9saWN5U3RhdGVtZW50KHtcbiAgICAgICAgICBhY3Rpb25zOiBbJ3NxczoqJ10sXG4gICAgICAgICAgZWZmZWN0OiBFZmZlY3QuREVOWSxcbiAgICAgICAgICBwcmluY2lwYWxzOiBbbmV3IEFueVByaW5jaXBhbCgpXSxcbiAgICAgICAgICBjb25kaXRpb25zOiB7IEJvb2w6IHsgJ2F3czpTZWN1cmVUcmFuc3BvcnQnOiBmYWxzZSB9IH0sXG4gICAgICAgICAgcmVzb3VyY2VzOiBbdGhpcy5lcnJvclF1ZXVlLnF1ZXVlQXJuXSxcbiAgICAgICAgfSksXG4gICAgICApO1xuICAgICAgdGhpcy5lcnJvckRlc3QgPSBuZXcgU3FzRGVzdGluYXRpb24odGhpcy5lcnJvclF1ZXVlKTtcbiAgICAgIE5hZ1N1cHByZXNzaW9ucy5hZGRSZXNvdXJjZVN1cHByZXNzaW9ucyh0aGlzLmVycm9yRGVhZExldHRlclF1ZXVlLCBbXG4gICAgICAgIHsgaWQ6ICdBd3NTb2x1dGlvbnMtU1FTMycsIHJlYXNvbjogJ1RoaXMgcXVldWUgaXMgYSBETFEuJyB9LFxuICAgICAgXSk7XG4gICAgfSBlbHNlIHtcbiAgICAgIHRoaXMuZXJyb3JEZXN0ID0gcHJvcHMub25FcnJvcjtcbiAgICB9XG5cbiAgICBjb25zdCB2cGMgPSBuZXcgVnBjKHRoaXMsICdTY2FuVlBDJywge1xuICAgICAgc3VibmV0Q29uZmlndXJhdGlvbjogW1xuICAgICAgICB7XG4gICAgICAgICAgc3VibmV0VHlwZTogU3VibmV0VHlwZS5QUklWQVRFX0lTT0xBVEVELFxuICAgICAgICAgIG5hbWU6ICdJc29sYXRlZCcsXG4gICAgICAgIH0sXG4gICAgICBdLFxuICAgIH0pO1xuXG4gICAgdnBjLmFkZEZsb3dMb2coJ0Zsb3dMb2dzJyk7XG5cbiAgICB0aGlzLl9zM0d3ID0gdnBjLmFkZEdhdGV3YXlFbmRwb2ludCgnUzNFbmRwb2ludCcsIHtcbiAgICAgIHNlcnZpY2U6IEdhdGV3YXlWcGNFbmRwb2ludEF3c1NlcnZpY2UuUzMsXG4gICAgfSk7XG5cbiAgICBjb25zdCBmaWxlU3lzdGVtID0gbmV3IEZpbGVTeXN0ZW0odGhpcywgJ1NjYW5GaWxlU3lzdGVtJywge1xuICAgICAgdnBjOiB2cGMsXG4gICAgICBlbmNyeXB0ZWQ6IHByb3BzLmVmc0VuY3J5cHRpb24gPT09IGZhbHNlID8gZmFsc2UgOiB0cnVlLFxuICAgICAgbGlmZWN5Y2xlUG9saWN5OiBMaWZlY3ljbGVQb2xpY3kuQUZURVJfN19EQVlTLFxuICAgICAgcGVyZm9ybWFuY2VNb2RlOiBQZXJmb3JtYW5jZU1vZGUuR0VORVJBTF9QVVJQT1NFLFxuICAgICAgcmVtb3ZhbFBvbGljeTogUmVtb3ZhbFBvbGljeS5ERVNUUk9ZLFxuICAgICAgc2VjdXJpdHlHcm91cDogbmV3IFNlY3VyaXR5R3JvdXAodGhpcywgJ1NjYW5GaWxlU3lzdGVtU2VjdXJpdHlHcm91cCcsIHtcbiAgICAgICAgdnBjOiB2cGMsXG4gICAgICAgIGFsbG93QWxsT3V0Ym91bmQ6IGZhbHNlLFxuICAgICAgfSksXG4gICAgfSk7XG5cbiAgICBjb25zdCBsYW1iZGFfYXAgPSBmaWxlU3lzdGVtLmFkZEFjY2Vzc1BvaW50KCdTY2FuTGFtYmRhQVAnLCB7XG4gICAgICBjcmVhdGVBY2w6IHtcbiAgICAgICAgb3duZXJHaWQ6ICcxMDAwJyxcbiAgICAgICAgb3duZXJVaWQ6ICcxMDAwJyxcbiAgICAgICAgcGVybWlzc2lvbnM6ICc3NTUnLFxuICAgICAgfSxcbiAgICAgIHBvc2l4VXNlcjoge1xuICAgICAgICBnaWQ6ICcxMDAwJyxcbiAgICAgICAgdWlkOiAnMTAwMCcsXG4gICAgICB9LFxuICAgICAgcGF0aDogdGhpcy5fZWZzUm9vdFBhdGgsXG4gICAgfSk7XG5cbiAgICBjb25zdCBsb2dzX2J1Y2tldCA9IHByb3BzLmRlZnNCdWNrZXRBY2Nlc3NMb2dzQ29uZmlnPy5sb2dzQnVja2V0O1xuICAgIGNvbnN0IGxvZ3NfYnVja2V0X3ByZWZpeCA9IHByb3BzLmRlZnNCdWNrZXRBY2Nlc3NMb2dzQ29uZmlnPy5sb2dzUHJlZml4O1xuICAgIGlmIChsb2dzX2J1Y2tldCA9PT0gdHJ1ZSB8fCBsb2dzX2J1Y2tldCA9PT0gdW5kZWZpbmVkKSB7XG4gICAgICB0aGlzLmRlZnNBY2Nlc3NMb2dzQnVja2V0ID0gbmV3IEJ1Y2tldChcbiAgICAgICAgdGhpcyxcbiAgICAgICAgJ1ZpcnVzRGVmc0FjY2Vzc0xvZ3NCdWNrZXQnLFxuICAgICAgICB7XG4gICAgICAgICAgZW5jcnlwdGlvbjogQnVja2V0RW5jcnlwdGlvbi5TM19NQU5BR0VELFxuICAgICAgICAgIHJlbW92YWxQb2xpY3k6IFJlbW92YWxQb2xpY3kuUkVUQUlOLFxuICAgICAgICAgIHNlcnZlckFjY2Vzc0xvZ3NQcmVmaXg6ICdhY2Nlc3MtbG9ncy1idWNrZXQtbG9ncycsXG4gICAgICAgICAgYmxvY2tQdWJsaWNBY2Nlc3M6IHtcbiAgICAgICAgICAgIGJsb2NrUHVibGljQWNsczogdHJ1ZSxcbiAgICAgICAgICAgIGJsb2NrUHVibGljUG9saWN5OiB0cnVlLFxuICAgICAgICAgICAgaWdub3JlUHVibGljQWNsczogdHJ1ZSxcbiAgICAgICAgICAgIHJlc3RyaWN0UHVibGljQnVja2V0czogdHJ1ZSxcbiAgICAgICAgICB9LFxuICAgICAgICB9LFxuICAgICAgKTtcbiAgICAgIHRoaXMuZGVmc0FjY2Vzc0xvZ3NCdWNrZXQuYWRkVG9SZXNvdXJjZVBvbGljeShcbiAgICAgICAgbmV3IFBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgICAgZWZmZWN0OiBFZmZlY3QuREVOWSxcbiAgICAgICAgICBhY3Rpb25zOiBbJ3MzOionXSxcbiAgICAgICAgICByZXNvdXJjZXM6IFtcbiAgICAgICAgICAgIHRoaXMuZGVmc0FjY2Vzc0xvZ3NCdWNrZXQuYXJuRm9yT2JqZWN0cygnKicpLFxuICAgICAgICAgICAgdGhpcy5kZWZzQWNjZXNzTG9nc0J1Y2tldC5idWNrZXRBcm4sXG4gICAgICAgICAgXSxcbiAgICAgICAgICBwcmluY2lwYWxzOiBbbmV3IEFueVByaW5jaXBhbCgpXSxcbiAgICAgICAgICBjb25kaXRpb25zOiB7XG4gICAgICAgICAgICBCb29sOiB7XG4gICAgICAgICAgICAgICdhd3M6U2VjdXJlVHJhbnNwb3J0JzogZmFsc2UsXG4gICAgICAgICAgICB9LFxuICAgICAgICAgIH0sXG4gICAgICAgIH0pLFxuICAgICAgKTtcbiAgICB9IGVsc2UgaWYgKGxvZ3NfYnVja2V0ICE9IGZhbHNlKSB7XG4gICAgICB0aGlzLmRlZnNBY2Nlc3NMb2dzQnVja2V0ID0gbG9nc19idWNrZXQ7XG4gICAgfVxuXG4gICAgY29uc3QgZGVmc19idWNrZXQgPSBuZXcgQnVja2V0KHRoaXMsICdWaXJ1c0RlZnNCdWNrZXQnLCB7XG4gICAgICBlbmNyeXB0aW9uOiBCdWNrZXRFbmNyeXB0aW9uLlMzX01BTkFHRUQsXG4gICAgICByZW1vdmFsUG9saWN5OiBSZW1vdmFsUG9saWN5LkRFU1RST1ksXG4gICAgICBhdXRvRGVsZXRlT2JqZWN0czogdHJ1ZSxcbiAgICAgIHNlcnZlckFjY2Vzc0xvZ3NCdWNrZXQ6IHRoaXMuZGVmc0FjY2Vzc0xvZ3NCdWNrZXQsXG4gICAgICBzZXJ2ZXJBY2Nlc3NMb2dzUHJlZml4OlxuICAgICAgICBsb2dzX2J1Y2tldCA9PT0gZmFsc2UgPyB1bmRlZmluZWQgOiBsb2dzX2J1Y2tldF9wcmVmaXgsXG4gICAgICBibG9ja1B1YmxpY0FjY2Vzczoge1xuICAgICAgICBibG9ja1B1YmxpY0FjbHM6IHRydWUsXG4gICAgICAgIGJsb2NrUHVibGljUG9saWN5OiB0cnVlLFxuICAgICAgICBpZ25vcmVQdWJsaWNBY2xzOiB0cnVlLFxuICAgICAgICByZXN0cmljdFB1YmxpY0J1Y2tldHM6IHRydWUsXG4gICAgICB9LFxuICAgIH0pO1xuXG4gICAgZGVmc19idWNrZXQuYWRkVG9SZXNvdXJjZVBvbGljeShcbiAgICAgIG5ldyBQb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgICBlZmZlY3Q6IEVmZmVjdC5ERU5ZLFxuICAgICAgICBhY3Rpb25zOiBbJ3MzOionXSxcbiAgICAgICAgcmVzb3VyY2VzOiBbZGVmc19idWNrZXQuYXJuRm9yT2JqZWN0cygnKicpLCBkZWZzX2J1Y2tldC5idWNrZXRBcm5dLFxuICAgICAgICBwcmluY2lwYWxzOiBbbmV3IEFueVByaW5jaXBhbCgpXSxcbiAgICAgICAgY29uZGl0aW9uczoge1xuICAgICAgICAgIEJvb2w6IHtcbiAgICAgICAgICAgICdhd3M6U2VjdXJlVHJhbnNwb3J0JzogZmFsc2UsXG4gICAgICAgICAgfSxcbiAgICAgICAgfSxcbiAgICAgIH0pLFxuICAgICk7XG4gICAgZGVmc19idWNrZXQuYWRkVG9SZXNvdXJjZVBvbGljeShcbiAgICAgIG5ldyBQb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgICBlZmZlY3Q6IEVmZmVjdC5BTExPVyxcbiAgICAgICAgYWN0aW9uczogWydzMzpHZXRPYmplY3QnLCAnczM6TGlzdEJ1Y2tldCddLFxuICAgICAgICByZXNvdXJjZXM6IFtkZWZzX2J1Y2tldC5hcm5Gb3JPYmplY3RzKCcqJyksIGRlZnNfYnVja2V0LmJ1Y2tldEFybl0sXG4gICAgICAgIHByaW5jaXBhbHM6IFtuZXcgQW55UHJpbmNpcGFsKCldLFxuICAgICAgICBjb25kaXRpb25zOiB7XG4gICAgICAgICAgU3RyaW5nRXF1YWxzOiB7XG4gICAgICAgICAgICAnYXdzOlNvdXJjZVZwY2UnOiB0aGlzLl9zM0d3LnZwY0VuZHBvaW50SWQsXG4gICAgICAgICAgfSxcbiAgICAgICAgfSxcbiAgICAgIH0pLFxuICAgICk7XG4gICAgZGVmc19idWNrZXQuYWRkVG9SZXNvdXJjZVBvbGljeShcbiAgICAgIG5ldyBQb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgICBlZmZlY3Q6IEVmZmVjdC5ERU5ZLFxuICAgICAgICBhY3Rpb25zOiBbJ3MzOlB1dEJ1Y2tldFBvbGljeScsICdzMzpEZWxldGVCdWNrZXRQb2xpY3knXSxcbiAgICAgICAgcmVzb3VyY2VzOiBbZGVmc19idWNrZXQuYnVja2V0QXJuXSxcbiAgICAgICAgbm90UHJpbmNpcGFsczogW25ldyBBY2NvdW50Um9vdFByaW5jaXBhbCgpXSxcbiAgICAgIH0pLFxuICAgICk7XG4gICAgdGhpcy5fczNHdy5hZGRUb1BvbGljeShcbiAgICAgIG5ldyBQb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgICBlZmZlY3Q6IEVmZmVjdC5BTExPVyxcbiAgICAgICAgYWN0aW9uczogWydzMzpHZXRPYmplY3QnLCAnczM6TGlzdEJ1Y2tldCddLFxuICAgICAgICByZXNvdXJjZXM6IFtkZWZzX2J1Y2tldC5hcm5Gb3JPYmplY3RzKCcqJyksIGRlZnNfYnVja2V0LmJ1Y2tldEFybl0sXG4gICAgICAgIHByaW5jaXBhbHM6IFtuZXcgQW55UHJpbmNpcGFsKCldLFxuICAgICAgfSksXG4gICAgKTtcblxuICAgIHRoaXMuX3NjYW5GdW5jdGlvbiA9IG5ldyBEb2NrZXJJbWFnZUZ1bmN0aW9uKHRoaXMsICdTZXJ2ZXJsZXNzQ2xhbXNjYW4nLCB7XG4gICAgICBjb2RlOiBEb2NrZXJJbWFnZUNvZGUuZnJvbUltYWdlQXNzZXQoXG4gICAgICAgIHBhdGguam9pbihfX2Rpcm5hbWUsICcuLi9hc3NldHMvbGFtYmRhL2NvZGUvc2NhbicpLFxuICAgICAgICB7XG4gICAgICAgICAgYnVpbGRBcmdzOiB7XG4gICAgICAgICAgICAvLyBPbmx5IGZvcmNlIHVwZGF0ZSB0aGUgZG9ja2VyIGxheWVyIGNhY2hlIG9uY2UgYSBkYXlcbiAgICAgICAgICAgIENBQ0hFX0RBVEU6IG5ldyBEYXRlKCkudG9EYXRlU3RyaW5nKCksXG4gICAgICAgICAgfSxcbiAgICAgICAgICBleHRyYUhhc2g6IERhdGUubm93KCkudG9TdHJpbmcoKSxcbiAgICAgICAgfSxcbiAgICAgICksXG4gICAgICBvblN1Y2Nlc3M6IHRoaXMucmVzdWx0RGVzdCxcbiAgICAgIG9uRmFpbHVyZTogdGhpcy5lcnJvckRlc3QsXG4gICAgICBmaWxlc3lzdGVtOiBMYW1iZGFGaWxlU3lzdGVtLmZyb21FZnNBY2Nlc3NQb2ludChcbiAgICAgICAgbGFtYmRhX2FwLFxuICAgICAgICB0aGlzLl9lZnNNb3VudFBhdGgsXG4gICAgICApLFxuICAgICAgdnBjOiB2cGMsXG4gICAgICB2cGNTdWJuZXRzOiB7IHN1Ym5ldHM6IHZwYy5pc29sYXRlZFN1Ym5ldHMgfSxcbiAgICAgIGFsbG93QWxsT3V0Ym91bmQ6IGZhbHNlLFxuICAgICAgdGltZW91dDogRHVyYXRpb24ubWludXRlcygxNSksXG4gICAgICBtZW1vcnlTaXplOiAxMDI0MCxcbiAgICAgIHJlc2VydmVkQ29uY3VycmVudEV4ZWN1dGlvbnM6IHByb3BzLnJlc2VydmVkQ29uY3VycmVuY3ksXG4gICAgICBlbnZpcm9ubWVudDoge1xuICAgICAgICBFRlNfTU9VTlRfUEFUSDogdGhpcy5fZWZzTW91bnRQYXRoLFxuICAgICAgICBFRlNfREVGX1BBVEg6IHRoaXMuX2Vmc0RlZnNQYXRoLFxuICAgICAgICBERUZTX1VSTDogZGVmc19idWNrZXQudmlydHVhbEhvc3RlZFVybEZvck9iamVjdCgpLFxuICAgICAgICBQT1dFUlRPT0xTX01FVFJJQ1NfTkFNRVNQQUNFOiAnc2VydmVybGVzcy1jbGFtc2NhbicsXG4gICAgICAgIFBPV0VSVE9PTFNfU0VSVklDRV9OQU1FOiAndmlydXMtc2NhbicsXG4gICAgICB9LFxuICAgIH0pO1xuICAgIGlmICh0aGlzLl9zY2FuRnVuY3Rpb24ucm9sZSkge1xuICAgICAgTmFnU3VwcHJlc3Npb25zLmFkZFJlc291cmNlU3VwcHJlc3Npb25zKHRoaXMuX3NjYW5GdW5jdGlvbi5yb2xlLCBbXG4gICAgICAgIHtcbiAgICAgICAgICBpZDogJ0F3c1NvbHV0aW9ucy1JQU00JyxcbiAgICAgICAgICByZWFzb246XG4gICAgICAgICAgICAnVGhlIEFXU0xhbWJkYUJhc2ljRXhlY3V0aW9uUm9sZSBkb2VzIG5vdCBwcm92aWRlIHBlcm1pc3Npb25zIGJleW9uZCB1cGxvYWRpbmcgbG9ncyB0byBDbG91ZFdhdGNoLiBUaGUgQVdTTGFtYmRhVlBDQWNjZXNzRXhlY3V0aW9uUm9sZSBpcyByZXF1aXJlZCBmb3IgZnVuY3Rpb25zIHdpdGggVlBDIGFjY2VzcyB0byBtYW5hZ2UgZWxhc3RpYyBuZXR3b3JrIGludGVyZmFjZXMuJyxcbiAgICAgICAgfSxcbiAgICAgIF0pO1xuICAgICAgTmFnU3VwcHJlc3Npb25zLmFkZFJlc291cmNlU3VwcHJlc3Npb25zKFxuICAgICAgICB0aGlzLl9zY2FuRnVuY3Rpb24ucm9sZSxcbiAgICAgICAgW1xuICAgICAgICAgIHtcbiAgICAgICAgICAgIGlkOiAnQXdzU29sdXRpb25zLUlBTTUnLFxuICAgICAgICAgICAgcmVhc29uOlxuICAgICAgICAgICAgICAnVGhlIEVGUyBtb3VudCBwb2ludCBwZXJtaXNzaW9ucyBhcmUgY29udHJvbGxlZCB0aHJvdWdoIGEgY29uZGl0aW9uIHdoaWNoIGxpbWl0IHRoZSBzY29wZSBvZiB0aGUgKiByZXNvdXJjZXMuJyxcbiAgICAgICAgICB9LFxuICAgICAgICBdLFxuICAgICAgICB0cnVlLFxuICAgICAgKTtcbiAgICB9XG4gICAgdGhpcy5fc2NhbkZ1bmN0aW9uLmNvbm5lY3Rpb25zLmFsbG93VG9BbnlJcHY0KFxuICAgICAgUG9ydC50Y3AoNDQzKSxcbiAgICAgICdBbGxvdyBvdXRib3VuZCBIVFRQUyB0cmFmZmljIGZvciBTMyBhY2Nlc3MuJyxcbiAgICApO1xuICAgIGRlZnNfYnVja2V0LmdyYW50UmVhZCh0aGlzLl9zY2FuRnVuY3Rpb24pO1xuXG4gICAgY29uc3QgZG93bmxvYWRfZGVmcyA9IG5ldyBEb2NrZXJJbWFnZUZ1bmN0aW9uKHRoaXMsICdEb3dubG9hZERlZnMnLCB7XG4gICAgICBjb2RlOiBEb2NrZXJJbWFnZUNvZGUuZnJvbUltYWdlQXNzZXQoXG4gICAgICAgIHBhdGguam9pbihfX2Rpcm5hbWUsICcuLi9hc3NldHMvbGFtYmRhL2NvZGUvZG93bmxvYWRfZGVmcycpLFxuICAgICAgICB7XG4gICAgICAgICAgYnVpbGRBcmdzOiB7XG4gICAgICAgICAgICAvLyBPbmx5IGZvcmNlIHVwZGF0ZSB0aGUgZG9ja2VyIGxheWVyIGNhY2hlIG9uY2UgYSBkYXlcbiAgICAgICAgICAgIENBQ0hFX0RBVEU6IG5ldyBEYXRlKCkudG9EYXRlU3RyaW5nKCksXG4gICAgICAgICAgfSxcbiAgICAgICAgICBleHRyYUhhc2g6IERhdGUubm93KCkudG9TdHJpbmcoKSxcbiAgICAgICAgfSxcbiAgICAgICksXG4gICAgICB0aW1lb3V0OiBEdXJhdGlvbi5taW51dGVzKDUpLFxuICAgICAgbWVtb3J5U2l6ZTogMTAyNCxcbiAgICAgIGVudmlyb25tZW50OiB7XG4gICAgICAgIERFRlNfQlVDS0VUOiBkZWZzX2J1Y2tldC5idWNrZXROYW1lLFxuICAgICAgICBQT1dFUlRPT0xTX1NFUlZJQ0VfTkFNRTogJ2ZyZXNoY2xhbS11cGRhdGUnLFxuICAgICAgfSxcbiAgICB9KTtcbiAgICBjb25zdCBzdGFjayA9IFN0YWNrLm9mKHRoaXMpO1xuXG4gICAgaWYgKGRvd25sb2FkX2RlZnMucm9sZSkge1xuICAgICAgY29uc3QgZG93bmxvYWRfZGVmc19yb2xlID0gYGFybjoke3N0YWNrLnBhcnRpdGlvbn06c3RzOjoke3N0YWNrLmFjY291bnR9OmFzc3VtZWQtcm9sZS8ke2Rvd25sb2FkX2RlZnMucm9sZS5yb2xlTmFtZX0vJHtkb3dubG9hZF9kZWZzLmZ1bmN0aW9uTmFtZX1gO1xuICAgICAgY29uc3QgZG93bmxvYWRfZGVmc19hc3N1bWVkX3ByaW5jaXBhbCA9IG5ldyBBcm5QcmluY2lwYWwoXG4gICAgICAgIGRvd25sb2FkX2RlZnNfcm9sZSxcbiAgICAgICk7XG4gICAgICBkZWZzX2J1Y2tldC5hZGRUb1Jlc291cmNlUG9saWN5KFxuICAgICAgICBuZXcgUG9saWN5U3RhdGVtZW50KHtcbiAgICAgICAgICBlZmZlY3Q6IEVmZmVjdC5ERU5ZLFxuICAgICAgICAgIGFjdGlvbnM6IFsnczM6UHV0T2JqZWN0KiddLFxuICAgICAgICAgIHJlc291cmNlczogW2RlZnNfYnVja2V0LmFybkZvck9iamVjdHMoJyonKV0sXG4gICAgICAgICAgbm90UHJpbmNpcGFsczogW2Rvd25sb2FkX2RlZnMucm9sZSwgZG93bmxvYWRfZGVmc19hc3N1bWVkX3ByaW5jaXBhbF0sXG4gICAgICAgIH0pLFxuICAgICAgKTtcbiAgICAgIGRlZnNfYnVja2V0LmdyYW50UmVhZFdyaXRlKGRvd25sb2FkX2RlZnMpO1xuICAgICAgTmFnU3VwcHJlc3Npb25zLmFkZFJlc291cmNlU3VwcHJlc3Npb25zKGRvd25sb2FkX2RlZnMucm9sZSwgW1xuICAgICAgICB7XG4gICAgICAgICAgaWQ6ICdBd3NTb2x1dGlvbnMtSUFNNCcsXG4gICAgICAgICAgcmVhc29uOlxuICAgICAgICAgICAgJ1RoZSBBV1NMYW1iZGFCYXNpY0V4ZWN1dGlvblJvbGUgZG9lcyBub3QgcHJvdmlkZSBwZXJtaXNzaW9ucyBiZXlvbmQgdXBsb2FkaW5nIGxvZ3MgdG8gQ2xvdWRXYXRjaC4nLFxuICAgICAgICB9LFxuICAgICAgXSk7XG4gICAgICBOYWdTdXBwcmVzc2lvbnMuYWRkUmVzb3VyY2VTdXBwcmVzc2lvbnMoXG4gICAgICAgIGRvd25sb2FkX2RlZnMucm9sZSxcbiAgICAgICAgW1xuICAgICAgICAgIHtcbiAgICAgICAgICAgIGlkOiAnQXdzU29sdXRpb25zLUlBTTUnLFxuICAgICAgICAgICAgcmVhc29uOlxuICAgICAgICAgICAgICAnVGhlIGZ1bmN0aW9uIGlzIGFsbG93ZWQgdG8gaW52b2tlIHRoZSBkb3dubG9hZCBkZWZzIExhbWJkYSBmdW5jdGlvbi4nLFxuICAgICAgICAgIH0sXG4gICAgICAgIF0sXG4gICAgICAgIHRydWUsXG4gICAgICApO1xuICAgIH1cblxuICAgIG5ldyBSdWxlKHRoaXMsICdWaXJ1c0RlZnNVcGRhdGVSdWxlJywge1xuICAgICAgc2NoZWR1bGU6IFNjaGVkdWxlLnJhdGUoRHVyYXRpb24uaG91cnMoMTIpKSxcbiAgICAgIHRhcmdldHM6IFtuZXcgTGFtYmRhRnVuY3Rpb24oZG93bmxvYWRfZGVmcyldLFxuICAgIH0pO1xuXG4gICAgY29uc3QgaW5pdF9kZWZzX2NyID0gbmV3IEZ1bmN0aW9uKHRoaXMsICdJbml0RGVmcycsIHtcbiAgICAgIHJ1bnRpbWU6IFJ1bnRpbWUuUFlUSE9OXzNfOCxcbiAgICAgIGNvZGU6IENvZGUuZnJvbUFzc2V0KFxuICAgICAgICBwYXRoLmpvaW4oX19kaXJuYW1lLCAnLi4vYXNzZXRzL2xhbWJkYS9jb2RlL2luaXRpYWxpemVfZGVmc19jcicpLFxuICAgICAgKSxcbiAgICAgIGhhbmRsZXI6ICdsYW1iZGEubGFtYmRhX2hhbmRsZXInLFxuICAgICAgdGltZW91dDogRHVyYXRpb24ubWludXRlcyg1KSxcbiAgICB9KTtcbiAgICBkb3dubG9hZF9kZWZzLmdyYW50SW52b2tlKGluaXRfZGVmc19jcik7XG4gICAgaWYgKGluaXRfZGVmc19jci5yb2xlKSB7XG4gICAgICBOYWdTdXBwcmVzc2lvbnMuYWRkUmVzb3VyY2VTdXBwcmVzc2lvbnMoXG4gICAgICAgIGluaXRfZGVmc19jci5yb2xlLFxuICAgICAgICBbXG4gICAgICAgICAge1xuICAgICAgICAgICAgaWQ6ICdBd3NTb2x1dGlvbnMtSUFNNCcsXG4gICAgICAgICAgICByZWFzb246XG4gICAgICAgICAgICAgICdUaGUgQVdTTGFtYmRhQmFzaWNFeGVjdXRpb25Sb2xlIGRvZXMgbm90IHByb3ZpZGUgcGVybWlzc2lvbnMgYmV5b25kIHVwbG9hZGluZyBsb2dzIHRvIENsb3VkV2F0Y2guJyxcbiAgICAgICAgICB9LFxuICAgICAgICAgIHtcbiAgICAgICAgICAgIGlkOiAnQXdzU29sdXRpb25zLUlBTTUnLFxuICAgICAgICAgICAgcmVhc29uOlxuICAgICAgICAgICAgICAnVGhlIEFXU0xhbWJkYUJhc2ljRXhlY3V0aW9uUm9sZSBkb2VzIG5vdCBwcm92aWRlIHBlcm1pc3Npb25zIGJleW9uZCB1cGxvYWRpbmcgbG9ncyB0byBDbG91ZFdhdGNoLicsXG4gICAgICAgICAgfSxcbiAgICAgICAgXSxcbiAgICAgICAgdHJ1ZSxcbiAgICAgICk7XG4gICAgfVxuICAgIG5ldyBDdXN0b21SZXNvdXJjZSh0aGlzLCAnSW5pdERlZnNDcicsIHtcbiAgICAgIHNlcnZpY2VUb2tlbjogaW5pdF9kZWZzX2NyLmZ1bmN0aW9uQXJuLFxuICAgICAgcHJvcGVydGllczoge1xuICAgICAgICBGbk5hbWU6IGRvd25sb2FkX2RlZnMuZnVuY3Rpb25OYW1lLFxuICAgICAgfSxcbiAgICB9KTtcblxuICAgIGlmIChwcm9wcy5idWNrZXRzKSB7XG4gICAgICBwcm9wcy5idWNrZXRzLmZvckVhY2goKGJ1Y2tldCkgPT4ge1xuICAgICAgICB0aGlzLmFkZFNvdXJjZUJ1Y2tldChidWNrZXQpO1xuICAgICAgfSk7XG4gICAgfVxuICB9XG5cbiAgLyoqXG4gICAqIFNldHMgdGhlIHNwZWNpZmllZCBTMyBCdWNrZXQgYXMgYSBzMzpPYmplY3RDcmVhdGUqIGZvciB0aGUgQ2xhbUFWIGZ1bmN0aW9uLlxuICAgICBHcmFudHMgdGhlIENsYW1BViBmdW5jdGlvbiBwZXJtaXNzaW9ucyB0byBnZXQgYW5kIHRhZyBvYmplY3RzLlxuICAgICBBZGRzIGEgYnVja2V0IHBvbGljeSB0byBkaXNhbGxvdyBHZXRPYmplY3Qgb3BlcmF0aW9ucyBvbiBmaWxlcyB0aGF0IGFyZSB0YWdnZWQgJ0lOIFBST0dSRVNTJywgJ0lORkVDVEVEJywgb3IgJ0VSUk9SJy5cbiAgICogQHBhcmFtIGJ1Y2tldCBUaGUgYnVja2V0IHRvIGFkZCB0aGUgc2Nhbm5pbmcgYnVja2V0IHBvbGljeSBhbmQgczM6T2JqZWN0Q3JlYXRlKiB0cmlnZ2VyIHRvLlxuICAgKi9cbiAgYWRkU291cmNlQnVja2V0KGJ1Y2tldDogQnVja2V0KSB7XG4gICAgdGhpcy5fc2NhbkZ1bmN0aW9uLmFkZEV2ZW50U291cmNlKFxuICAgICAgbmV3IFMzRXZlbnRTb3VyY2UoYnVja2V0LCB7IGV2ZW50czogW0V2ZW50VHlwZS5PQkpFQ1RfQ1JFQVRFRF0gfSksXG4gICAgKTtcbiAgICBidWNrZXQuZ3JhbnRSZWFkKHRoaXMuX3NjYW5GdW5jdGlvbik7XG4gICAgdGhpcy5fc2NhbkZ1bmN0aW9uLmFkZFRvUm9sZVBvbGljeShcbiAgICAgIG5ldyBQb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgICBlZmZlY3Q6IEVmZmVjdC5BTExPVyxcbiAgICAgICAgYWN0aW9uczogWydzMzpQdXRPYmplY3RUYWdnaW5nJywgJ3MzOlB1dE9iamVjdFZlcnNpb25UYWdnaW5nJ10sXG4gICAgICAgIHJlc291cmNlczogW2J1Y2tldC5hcm5Gb3JPYmplY3RzKCcqJyldLFxuICAgICAgfSksXG4gICAgKTtcblxuICAgIGlmICh0aGlzLl9zY2FuRnVuY3Rpb24ucm9sZSkge1xuICAgICAgY29uc3Qgc3RhY2sgPSBTdGFjay5vZih0aGlzKTtcbiAgICAgIGNvbnN0IHNjYW5fYXNzdW1lZF9yb2xlID0gYGFybjoke3N0YWNrLnBhcnRpdGlvbn06c3RzOjoke3N0YWNrLmFjY291bnR9OmFzc3VtZWQtcm9sZS8ke3RoaXMuX3NjYW5GdW5jdGlvbi5yb2xlLnJvbGVOYW1lfS8ke3RoaXMuX3NjYW5GdW5jdGlvbi5mdW5jdGlvbk5hbWV9YDtcbiAgICAgIGNvbnN0IHNjYW5fYXNzdW1lZF9wcmluY2lwYWwgPSBuZXcgQXJuUHJpbmNpcGFsKHNjYW5fYXNzdW1lZF9yb2xlKTtcbiAgICAgIHRoaXMuX3MzR3cuYWRkVG9Qb2xpY3koXG4gICAgICAgIG5ldyBQb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgICAgIGVmZmVjdDogRWZmZWN0LkFMTE9XLFxuICAgICAgICAgIGFjdGlvbnM6IFsnczM6R2V0T2JqZWN0KicsICdzMzpHZXRCdWNrZXQqJywgJ3MzOkxpc3QqJ10sXG4gICAgICAgICAgcmVzb3VyY2VzOiBbYnVja2V0LmJ1Y2tldEFybiwgYnVja2V0LmFybkZvck9iamVjdHMoJyonKV0sXG4gICAgICAgICAgcHJpbmNpcGFsczogW3RoaXMuX3NjYW5GdW5jdGlvbi5yb2xlLCBzY2FuX2Fzc3VtZWRfcHJpbmNpcGFsXSxcbiAgICAgICAgfSksXG4gICAgICApO1xuICAgICAgdGhpcy5fczNHdy5hZGRUb1BvbGljeShcbiAgICAgICAgbmV3IFBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgICAgZWZmZWN0OiBFZmZlY3QuQUxMT1csXG4gICAgICAgICAgYWN0aW9uczogWydzMzpQdXRPYmplY3RUYWdnaW5nJywgJ3MzOlB1dE9iamVjdFZlcnNpb25UYWdnaW5nJ10sXG4gICAgICAgICAgcmVzb3VyY2VzOiBbYnVja2V0LmFybkZvck9iamVjdHMoJyonKV0sXG4gICAgICAgICAgcHJpbmNpcGFsczogW3RoaXMuX3NjYW5GdW5jdGlvbi5yb2xlLCBzY2FuX2Fzc3VtZWRfcHJpbmNpcGFsXSxcbiAgICAgICAgfSksXG4gICAgICApO1xuXG4gICAgICAvLyBOZWVkIHRoZSBhc3N1bWVkIHJvbGUgZm9yIHRoZSBub3QgUHJpbmNpcGFsIEFjdGlvbiB3aXRoIExhbWJkYVxuICAgICAgYnVja2V0LmFkZFRvUmVzb3VyY2VQb2xpY3koXG4gICAgICAgIG5ldyBQb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgICAgIGVmZmVjdDogRWZmZWN0LkRFTlksXG4gICAgICAgICAgYWN0aW9uczogWydzMzpHZXRPYmplY3QnXSxcbiAgICAgICAgICByZXNvdXJjZXM6IFtidWNrZXQuYXJuRm9yT2JqZWN0cygnKicpXSxcbiAgICAgICAgICBub3RQcmluY2lwYWxzOiBbdGhpcy5fc2NhbkZ1bmN0aW9uLnJvbGUsIHNjYW5fYXNzdW1lZF9wcmluY2lwYWxdLFxuICAgICAgICAgIGNvbmRpdGlvbnM6IHtcbiAgICAgICAgICAgIFN0cmluZ0VxdWFsczoge1xuICAgICAgICAgICAgICAnczM6RXhpc3RpbmdPYmplY3RUYWcvc2Nhbi1zdGF0dXMnOiBbXG4gICAgICAgICAgICAgICAgJ0lOIFBST0dSRVNTJyxcbiAgICAgICAgICAgICAgICAnSU5GRUNURUQnLFxuICAgICAgICAgICAgICAgICdFUlJPUicsXG4gICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICB9LFxuICAgICAgICAgIH0sXG4gICAgICAgIH0pLFxuICAgICAgKTtcbiAgICB9XG4gIH1cbn1cbiJdfQ==