# -*- coding: utf-8 -*-
from dcicutils.ff_utils import get_metadata
from tibanna import create_logger
from tibanna.utils import _tibanna_settings
# from tibanna_4dn.vars import TIBANNA_DEFAULT_STEP_FUNCTION_NAME
from tibanna_4dn.core import API
from tibanna_ffcommon.exceptions import TibannaStartException, FdnConnectionException
from tibanna_ffcommon.portal_utils import (
    TibannaSettings,
    FormatExtensionMap,
    parse_formatstr
)
from tibanna_4dn.vars import AWS_REGION, LAMBDA_TYPE, BUCKET_NAME


logger = create_logger(__name__)


config = {
    'function_name': 'validate_md5_s3_initiator_' + LAMBDA_TYPE,
    'function_module': 'service',
    'function_handler': 'handler',
    'handler': 'service.handler',
    'region': AWS_REGION,
    'runtime': 'python3.6',
    'role': 'sysadmin',
    'description': 'initiates md5/fastqc runs',
    'timeout': 300,
    'memory_size': 256
}


TIBANNA_DEFAULT_STEP_FUNCTION_NAME = 'tibanna_' + LAMBDA_TYPE + '_tmp_md5'


def handler(event, context):
    '''
    this is triggered on completed file upload from s3 and
    event will be set to file data.
    '''
    # get file name
    # print(event)

    print("is status uploading: %s" % event)
    upload_key = event['Records'][0]['s3']['object']['key']
    if upload_key.endswith('html'):
        return False

    uuid, object_key = upload_key.split('/')
    accession = object_key.split('.')[0]

    # guess env from bucket name
    bucket = event['Records'][0]['s3']['bucket']['name']
    env = '-'.join(bucket.split('-')[1:3])
    if 'prod' in env:
        env = 'data'

    try:
        tibanna = TibannaSettings(env=env)
    except Exception as e:
        raise TibannaStartException("%s" % e)
    try:
        meta = get_metadata(accession,
                            key=tibanna.ff_keys,
                            ff_env=env,
                            add_on='frame=object',
                            check_queue=True)
    except Exception as e:
        raise FdnConnectionException("can't get metadata for the accession %s: %s" % (accession, str(e)))

    status = get_status(meta)
    input_json = make_input(event)
    file_format, extra = get_file_format(meta, object_key, tibanna.ff_keys)
    if extra:  # the file is an extra file
        extra_status = get_status_for_extra_file(meta, file_format)
        if status != 'to be uploaded by workflow':
            if not extra_status or extra_status != 'to be uploaded by workflow':
                input_json['input_files'][0]['format_if_extra'] = file_format
                response = API().run_workflow(sfn=TIBANNA_DEFAULT_STEP_FUNCTION_NAME, input_json=input_json,
                                              open_browser=False)
            else:
                return {'info': 'status for extra file is to be uploaded by workflow'}
        else:
            return {'info': 'parent status for extra file is to be uploaded by workflow'}
    else:
        # only run if status is uploading...
        if status == 'uploading' or event.get('force_run'):
            # trigger the step function to run
            response = API().run_workflow(sfn=TIBANNA_DEFAULT_STEP_FUNCTION_NAME, input_json=input_json,
                                          open_browser=False)
        else:
            return {'info': 'status is not uploading'}

    # run fastqc as a dependent of md5
    if file_format == 'fastq':
        skip_list = ['pacbio', 'pac-bio', 'promethion', 'gridion', 'minion', 'smidgion']
        sequencer = meta.get('instrument', 'no value').lower()
        for skip in skip_list:
            if skip in sequencer:
                return serialize_startdate(response)
        md5_arn = response['_tibanna']['exec_arn']
        input_json_fastqc = make_input(event, 'fastqc', dependency=[md5_arn], run_name_prefix='fastqc')
        response_fastqc = API().run_workflow(sfn=TIBANNA_DEFAULT_STEP_FUNCTION_NAME, input_json=input_json_fastqc,
                                             open_browser=False)
        serialize_startdate(response_fastqc)
        response['fastqc'] = response_fastqc
    serialize_startdate(response)
    return response


def get_fileformats_for_accession(meta):
    file_format = parse_formatstr(meta.get('file_format'))
    extra_formats = [parse_formatstr(v.get('file_format')) for v in meta.get('extra_files', [])]
    return file_format, extra_formats


def get_file_format(meta, object_key, ff_keys):
    '''if the file extension matches the regular file format,
    returns (format, None)
    if it matches one of the format of an extra file,
    returns (format (e.g. 'pairs_px2'), 'extra')
    '''
    # guess env from bucket name
    accession = object_key.split('.')[0]
    extension = object_key.replace(accession + '.', '')

    file_format, extra_formats = get_fileformats_for_accession(meta)
    if file_format:
        fe_map = FormatExtensionMap(ff_keys)
        logger.debug("fe_map= " + str(fe_map))
        if extension == fe_map.get_extension(file_format):
            return (file_format, None)
        elif extension in fe_map.get_other_extensions(file_format):
            return (file_format, None)
        else:
            for extra_format in extra_formats:
                if extension == fe_map.get_extension(extra_format):
                    return (extra_format, 'extra')
                elif extension in fe_map.get_other_extensions(extra_format):
                    return (extra_format, 'extra')
        raise Exception("file extension not matching: %s vs %s (%s)" %
                        (extension, fe_map.get_extension(file_format), file_format))
    else:
        raise Exception("Cannot get file format from input metadata")


def get_status_for_extra_file(meta, extra_format):
    if meta and 'extra_files' in meta:
        for exf in meta['extra_files']:
            if parse_formatstr(exf['file_format']) == extra_format:
                return exf.get('status', None)
    return None


def get_status(meta):
    if meta:
        return meta.get('status', '')
    else:
        return ''


def get_outbucket_name(bucket):
    '''chop up bucket name and have a play'''
    return bucket.replace("files", "wfoutput")


def make_input(event, wf='md5', dependency=None, run_name_prefix='validate'):
    upload_key = event['Records'][0]['s3']['object']['key']

    uuid, object_key = upload_key.split('/')

    # guess env from bucket name
    bucket = event['Records'][0]['s3']['bucket']['name']
    env = '-'.join(bucket.split('-')[1:3])

    if 'prod' in env:
        env = 'data'

    run_name = run_name_prefix + "_%s" % (object_key.split('.')[0])
    if event.get('run_name'):
        run_name = event.get('run_name')  # used for testing

    return _make_input(env, wf, object_key, uuid, run_name, dependency)


_workflows = {'md5':
              {'uuid': 'c77a117b-9a58-477e-aaa5-291a109a99f6',
               'arg_name': 'input_file'
               },
              'fastqc':
              {'uuid': '49e96b51-ed6c-4418-a693-d0e9f79adfa5',
               'arg_name': 'input_fastq'
               },
              }


def _make_input(env, workflow, object_key, uuid, run_name, dependency=None):
    workflow_uuid = _workflows[workflow]['uuid']
    workflow_arg_name = _workflows[workflow]['arg_name']

    data = {"parameters": {},
            "app_name": workflow,
            "workflow_uuid": workflow_uuid,
            "input_files": [
                {"workflow_argument_name": workflow_arg_name,
                 "uuid": uuid,
                 "mount": True
                 }
             ],
            "config": {
                "instance_type": "t3.small",
                "ebs_size": 10,
                "shutdown_min": 0,
                "log_bucket": "tibanna-output",
                "key_name": "4dn-encode"
              }
            }
    if dependency:
        data["dependency"] = {"exec_arn": dependency}
    data.update(_tibanna_settings({'run_id': str(object_key),
                                   'run_name': run_name,
                                   'run_type': workflow,
                                   'env': env,
                                   }))
    return data


# fix non json-serializable datetime startDate
def serialize_startdate(response):
    tibanna_resp = response.get('_tibanna', {}).get('response')
    if tibanna_resp and tibanna_resp.get('startDate'):
        tibanna_resp['startDate'] = str(tibanna_resp['startDate'])
