import json

from pandas import DataFrame, Series

from connect.evidence import Evidence


class DataProfilerEvidence(Evidence):
    """
    Evidence for the result of pandas profiler"""

    def __init__(self, data: dict, additional_labels: dict = None, **metadata):
        self._data = data
        super().__init__("dataset_profiler", additional_labels, **metadata)

    @property
    def data(self):
        return self._data

    @property
    def base_label(self):
        return {}


class ModelProfilerEvidence(Evidence):
    """
    Evidence for the Model Profiler.

    Splits the data into info, parameters and feature names.

    Labeling contains info on where the entries come from: user vs autogenerated
    """

    def __init__(self, data: DataFrame, additional_labels: dict = None, **metadata):
        super().__init__("model_profiler", additional_labels, **metadata)
        self._data = data["results"]

    @property
    def data(self):
        parameters = self._data.loc["parameters"]
        features_names = self._data.loc["feature_names"]
        remaining_info = self._data[
            ~self._data.index.isin(["parameters", "feature_names", "model_name"])
        ]
        return {
            "info": remaining_info.to_dict(),
            "parameters": parameters,
            "features_names": features_names,
        }

    @property
    def base_label(self):
        return {"model_name": self._data.loc["model_name"]}
