"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("@aws-cdk/cdk");
const records_1 = require("./records");
const route53_generated_1 = require("./route53.generated");
const util_1 = require("./util");
class HostedZone extends cdk.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * VPCs to which this hosted zone will be added
         */
        this.vpcs = new Array();
        util_1.validateZoneName(props.zoneName);
        const hostedZone = new route53_generated_1.CfnHostedZone(this, 'Resource', {
            name: props.zoneName + '.',
            hostedZoneConfig: props.comment ? { comment: props.comment } : undefined,
            queryLoggingConfig: props.queryLogsLogGroupArn ? { cloudWatchLogsLogGroupArn: props.queryLogsLogGroupArn } : undefined,
            vpcs: new cdk.Token(() => this.vpcs.length === 0 ? undefined : this.vpcs)
        });
        this.hostedZoneId = hostedZone.ref;
        this.hostedZoneNameServers = hostedZone.hostedZoneNameServers;
        this.zoneName = props.zoneName;
        for (const vpc of props.vpcs || []) {
            this.addVpc(vpc);
        }
    }
    /**
     * Imports a hosted zone from another stack.
     */
    static import(scope, id, props) {
        return new ImportedHostedZone(scope, id, props);
    }
    export() {
        return {
            hostedZoneId: new cdk.CfnOutput(this, 'HostedZoneId', { value: this.hostedZoneId }).makeImportValue(),
            zoneName: this.zoneName,
        };
    }
    /**
     * Add another VPC to this private hosted zone.
     *
     * @param vpc the other VPC to add.
     */
    addVpc(vpc) {
        this.vpcs.push({ vpcId: vpc.vpcId, vpcRegion: vpc.vpcRegion });
    }
}
exports.HostedZone = HostedZone;
/**
 * Create a Route53 public hosted zone.
 */
class PublicHostedZone extends HostedZone {
    constructor(scope, id, props) {
        super(scope, id, props);
    }
    addVpc(_vpc) {
        throw new Error('Cannot associate public hosted zones with a VPC');
    }
    /**
     * Adds a delegation from this zone to a designated zone.
     *
     * @param delegate the zone being delegated to.
     * @param opts     options for creating the DNS record, if any.
     */
    addDelegation(delegate, opts = {}) {
        new records_1.ZoneDelegationRecord(this, `${this.zoneName} -> ${delegate.zoneName}`, {
            zone: this,
            delegatedZoneName: delegate.zoneName,
            nameServers: delegate.hostedZoneNameServers,
            comment: opts.comment,
            ttl: opts.ttl,
        });
    }
}
exports.PublicHostedZone = PublicHostedZone;
/**
 * Create a Route53 private hosted zone for use in one or more VPCs.
 *
 * Note that `enableDnsHostnames` and `enableDnsSupport` must have been enabled
 * for the VPC you're configuring for private hosted zones.
 */
class PrivateHostedZone extends HostedZone {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.addVpc(props.vpc);
    }
}
exports.PrivateHostedZone = PrivateHostedZone;
/**
 * Imported hosted zone
 */
class ImportedHostedZone extends cdk.Construct {
    constructor(scope, name, props) {
        super(scope, name);
        this.props = props;
        this.hostedZoneId = props.hostedZoneId;
        this.zoneName = props.zoneName;
    }
    export() {
        return this.props;
    }
}
//# sourceMappingURL=data:application/json;base64,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