"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.NetworkListener = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cxschema = require("@aws-cdk/cloud-assembly-schema");
const core_1 = require("@aws-cdk/core");
const base_listener_1 = require("../shared/base-listener");
const enums_1 = require("../shared/enums");
const util_1 = require("../shared/util");
const network_listener_action_1 = require("./network-listener-action");
const network_listener_certificate_1 = require("./network-listener-certificate");
const network_target_group_1 = require("./network-target-group");
/**
 * Define a Network Listener
 *
 * @resource AWS::ElasticLoadBalancingV2::Listener
 */
class NetworkListener extends base_listener_1.BaseListener {
    constructor(scope, id, props) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_elasticloadbalancingv2_NetworkListenerProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.constructor);
            }
            throw error;
        }
        const certs = props.certificates || [];
        const proto = props.protocol || (certs.length > 0 ? enums_1.Protocol.TLS : enums_1.Protocol.TCP);
        util_1.validateNetworkProtocol(proto);
        if (proto === enums_1.Protocol.TLS && certs.filter(v => v != null).length === 0) {
            throw new Error('When the protocol is set to TLS, you must specify certificates');
        }
        if (proto !== enums_1.Protocol.TLS && certs.length > 0) {
            throw new Error('Protocol must be TLS when certificates have been specified');
        }
        if (proto !== enums_1.Protocol.TLS && props.alpnPolicy) {
            throw new Error('Protocol must be TLS when alpnPolicy have been specified');
        }
        super(scope, id, {
            loadBalancerArn: props.loadBalancer.loadBalancerArn,
            protocol: proto,
            port: props.port,
            sslPolicy: props.sslPolicy,
            certificates: core_1.Lazy.any({ produce: () => this.certificateArns.map(certificateArn => ({ certificateArn })) }, { omitEmptyArray: true }),
            alpnPolicy: props.alpnPolicy ? [props.alpnPolicy] : undefined,
        });
        this.certificateArns = [];
        this.loadBalancer = props.loadBalancer;
        this.protocol = proto;
        if (certs.length > 0) {
            this.addCertificates('DefaultCertificates', certs);
        }
        if (props.defaultAction && props.defaultTargetGroups) {
            throw new Error('Specify at most one of \'defaultAction\' and \'defaultTargetGroups\'');
        }
        if (props.defaultAction) {
            this.setDefaultAction(props.defaultAction);
        }
        if (props.defaultTargetGroups) {
            this.setDefaultAction(network_listener_action_1.NetworkListenerAction.forward(props.defaultTargetGroups));
        }
    }
    /**
     * Looks up a network listener
     */
    static fromLookup(scope, id, options) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_elasticloadbalancingv2_NetworkListenerLookupOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromLookup);
            }
            throw error;
        }
        let listenerProtocol;
        if (options.listenerProtocol) {
            util_1.validateNetworkProtocol(options.listenerProtocol);
            switch (options.listenerProtocol) {
                case enums_1.Protocol.TCP:
                    listenerProtocol = cxschema.LoadBalancerListenerProtocol.TCP;
                    break;
                case enums_1.Protocol.UDP:
                    listenerProtocol = cxschema.LoadBalancerListenerProtocol.UDP;
                    break;
                case enums_1.Protocol.TCP_UDP:
                    listenerProtocol = cxschema.LoadBalancerListenerProtocol.TCP_UDP;
                    break;
                case enums_1.Protocol.TLS:
                    listenerProtocol = cxschema.LoadBalancerListenerProtocol.TLS;
                    break;
            }
        }
        const props = base_listener_1.BaseListener._queryContextProvider(scope, {
            userOptions: options,
            listenerProtocol: listenerProtocol,
            loadBalancerType: cxschema.LoadBalancerType.NETWORK,
        });
        class LookedUp extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.listenerArn = props.listenerArn;
            }
        }
        return new LookedUp(scope, id);
    }
    /**
     * Import an existing listener
     */
    static fromNetworkListenerArn(scope, id, networkListenerArn) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.listenerArn = networkListenerArn;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Add one or more certificates to this listener.
     *
     * After the first certificate, this creates NetworkListenerCertificates
     * resources since cloudformation requires the certificates array on the
     * listener resource to have a length of 1.
     */
    addCertificates(id, certificates) {
        const additionalCerts = [...certificates];
        if (this.certificateArns.length === 0 && additionalCerts.length > 0) {
            const first = additionalCerts.splice(0, 1)[0];
            this.certificateArns.push(first.certificateArn);
        }
        // Only one certificate can be specified per resource, even though
        // `certificates` is of type Array
        for (let i = 0; i < additionalCerts.length; i++) {
            new network_listener_certificate_1.NetworkListenerCertificate(this, `${id}${i + 1}`, {
                listener: this,
                certificates: [additionalCerts[i]],
            });
        }
    }
    /**
     * Load balance incoming requests to the given target groups.
     *
     * All target groups will be load balanced to with equal weight and without
     * stickiness. For a more complex configuration than that, use `addAction()`.
     */
    addTargetGroups(_id, ...targetGroups) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_elasticloadbalancingv2_INetworkTargetGroup(targetGroups);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addTargetGroups);
            }
            throw error;
        }
        this.setDefaultAction(network_listener_action_1.NetworkListenerAction.forward(targetGroups));
    }
    /**
     * Perform the given Action on incoming requests
     *
     * This allows full control of the default Action of the load balancer,
     * including weighted forwarding. See the `NetworkListenerAction` class for
     * all options.
     */
    addAction(_id, props) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_elasticloadbalancingv2_AddNetworkActionProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addAction);
            }
            throw error;
        }
        this.setDefaultAction(props.action);
    }
    /**
     * Load balance incoming requests to the given load balancing targets.
     *
     * This method implicitly creates a NetworkTargetGroup for the targets
     * involved, and a 'forward' action to route traffic to the given TargetGroup.
     *
     * If you want more control over the precise setup, create the TargetGroup
     * and use `addAction` yourself.
     *
     * It's possible to add conditions to the targets added in this way. At least
     * one set of targets must be added without conditions.
     *
     * @returns The newly created target group
     */
    addTargets(id, props) {
        var _b;
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_elasticloadbalancingv2_AddNetworkTargetsProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addTargets);
            }
            throw error;
        }
        if (!this.loadBalancer.vpc) {
            // eslint-disable-next-line max-len
            throw new Error('Can only call addTargets() when using a constructed Load Balancer or imported Load Balancer with specified VPC; construct a new TargetGroup and use addTargetGroup');
        }
        const group = new network_target_group_1.NetworkTargetGroup(this, id + 'Group', {
            deregistrationDelay: props.deregistrationDelay,
            healthCheck: props.healthCheck,
            port: props.port,
            protocol: (_b = props.protocol) !== null && _b !== void 0 ? _b : this.protocol,
            proxyProtocolV2: props.proxyProtocolV2,
            preserveClientIp: props.preserveClientIp,
            targetGroupName: props.targetGroupName,
            targets: props.targets,
            vpc: this.loadBalancer.vpc,
        });
        this.addTargetGroups(id, group);
        return group;
    }
    /**
     * Wrapper for _setDefaultAction which does a type-safe bind
     */
    setDefaultAction(action) {
        action.bind(this, this);
        this._setDefaultAction(action);
    }
}
exports.NetworkListener = NetworkListener;
_a = JSII_RTTI_SYMBOL_1;
NetworkListener[_a] = { fqn: "@aws-cdk/aws-elasticloadbalancingv2.NetworkListener", version: "1.155.0" };
//# sourceMappingURL=data:application/json;base64,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