"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.BaseLoadBalancer = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const cxschema = require("@aws-cdk/cloud-assembly-schema");
const core_1 = require("@aws-cdk/core");
const cxapi = require("@aws-cdk/cx-api");
const region_info_1 = require("@aws-cdk/region-info");
const elasticloadbalancingv2_generated_1 = require("../elasticloadbalancingv2.generated");
const util_1 = require("./util");
/**
 * Base class for both Application and Network Load Balancers
 */
class BaseLoadBalancer extends core_1.Resource {
    constructor(scope, id, baseProps, additionalProps) {
        super(scope, id, {
            physicalName: baseProps.loadBalancerName,
        });
        /**
         * Attributes set on this load balancer
         */
        this.attributes = {};
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_elasticloadbalancingv2_BaseLoadBalancerProps(baseProps);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.constructor);
            }
            throw error;
        }
        const internetFacing = util_1.ifUndefined(baseProps.internetFacing, false);
        const vpcSubnets = util_1.ifUndefined(baseProps.vpcSubnets, (internetFacing ? { subnetType: ec2.SubnetType.PUBLIC } : {}));
        const { subnetIds, internetConnectivityEstablished } = baseProps.vpc.selectSubnets(vpcSubnets);
        this.vpc = baseProps.vpc;
        const resource = new elasticloadbalancingv2_generated_1.CfnLoadBalancer(this, 'Resource', {
            name: this.physicalName,
            subnets: subnetIds,
            scheme: internetFacing ? 'internet-facing' : 'internal',
            loadBalancerAttributes: core_1.Lazy.any({ produce: () => util_1.renderAttributes(this.attributes) }, { omitEmptyArray: true }),
            ...additionalProps,
        });
        if (internetFacing) {
            resource.node.addDependency(internetConnectivityEstablished);
        }
        this.setAttribute('deletion_protection.enabled', baseProps.deletionProtection ? 'true' : 'false');
        this.loadBalancerCanonicalHostedZoneId = resource.attrCanonicalHostedZoneId;
        this.loadBalancerDnsName = resource.attrDnsName;
        this.loadBalancerFullName = resource.attrLoadBalancerFullName;
        this.loadBalancerName = resource.attrLoadBalancerName;
        this.loadBalancerArn = resource.ref;
        this.loadBalancerSecurityGroups = resource.attrSecurityGroups;
    }
    /**
     * Queries the load balancer context provider for load balancer info.
     * @internal
     */
    static _queryContextProvider(scope, options) {
        var _b;
        if (core_1.Token.isUnresolved(options.userOptions.loadBalancerArn)
            || Object.values((_b = options.userOptions.loadBalancerTags) !== null && _b !== void 0 ? _b : {}).some(core_1.Token.isUnresolved)) {
            throw new Error('All arguments to look up a load balancer must be concrete (no Tokens)');
        }
        let cxschemaTags;
        if (options.userOptions.loadBalancerTags) {
            cxschemaTags = util_1.mapTagMapToCxschema(options.userOptions.loadBalancerTags);
        }
        const props = core_1.ContextProvider.getValue(scope, {
            provider: cxschema.ContextProvider.LOAD_BALANCER_PROVIDER,
            props: {
                loadBalancerArn: options.userOptions.loadBalancerArn,
                loadBalancerTags: cxschemaTags,
                loadBalancerType: options.loadBalancerType,
            },
            dummyValue: {
                ipAddressType: cxapi.LoadBalancerIpAddressType.DUAL_STACK,
                loadBalancerArn: `arn:aws:elasticloadbalancing:us-west-2:123456789012:loadbalancer/${options.loadBalancerType}/my-load-balancer/50dc6c495c0c9188`,
                loadBalancerCanonicalHostedZoneId: 'Z3DZXE0EXAMPLE',
                loadBalancerDnsName: 'my-load-balancer-1234567890.us-west-2.elb.amazonaws.com',
                securityGroupIds: ['sg-1234'],
                vpcId: 'vpc-12345',
            },
        }).value;
        return props;
    }
    /**
     * Enable access logging for this load balancer.
     *
     * A region must be specified on the stack containing the load balancer; you cannot enable logging on
     * environment-agnostic stacks. See https://docs.aws.amazon.com/cdk/latest/guide/environments.html
     */
    logAccessLogs(bucket, prefix) {
        prefix = prefix || '';
        this.setAttribute('access_logs.s3.enabled', 'true');
        this.setAttribute('access_logs.s3.bucket', bucket.bucketName.toString());
        this.setAttribute('access_logs.s3.prefix', prefix);
        const region = core_1.Stack.of(this).region;
        if (core_1.Token.isUnresolved(region)) {
            throw new Error('Region is required to enable ELBv2 access logging');
        }
        const account = region_info_1.RegionInfo.get(region).elbv2Account;
        if (!account) {
            throw new Error(`Cannot enable access logging; don't know ELBv2 account for region ${region}`);
        }
        const logsDeliveryServicePrincipal = new aws_iam_1.ServicePrincipal('delivery.logs.amazonaws.com');
        bucket.grantPut(new iam.AccountPrincipal(account), `${(prefix ? prefix + '/' : '')}AWSLogs/${core_1.Stack.of(this).account}/*`);
        bucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
            actions: ['s3:PutObject'],
            principals: [logsDeliveryServicePrincipal],
            resources: [
                bucket.arnForObjects(`${prefix ? prefix + '/' : ''}AWSLogs/${this.stack.account}/*`),
            ],
            conditions: {
                StringEquals: { 's3:x-amz-acl': 'bucket-owner-full-control' },
            },
        }));
        bucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
            actions: ['s3:GetBucketAcl'],
            principals: [logsDeliveryServicePrincipal],
            resources: [bucket.bucketArn],
        }));
        // make sure the bucket's policy is created before the ALB (see https://github.com/aws/aws-cdk/issues/1633)
        this.node.addDependency(bucket);
    }
    /**
     * Set a non-standard attribute on the load balancer
     *
     * @see https://docs.aws.amazon.com/elasticloadbalancing/latest/application/application-load-balancers.html#load-balancer-attributes
     */
    setAttribute(key, value) {
        this.attributes[key] = value;
    }
    /**
     * Remove an attribute from the load balancer
     */
    removeAttribute(key) {
        this.setAttribute(key, undefined);
    }
    validate() {
        const ret = super.validate();
        // https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-elasticloadbalancingv2-loadbalancer.html#cfn-elasticloadbalancingv2-loadbalancer-name
        const loadBalancerName = this.physicalName;
        if (!core_1.Token.isUnresolved(loadBalancerName) && loadBalancerName !== undefined) {
            if (loadBalancerName.length > 32) {
                ret.push(`Load balancer name: "${loadBalancerName}" can have a maximum of 32 characters.`);
            }
            if (loadBalancerName.startsWith('internal-')) {
                ret.push(`Load balancer name: "${loadBalancerName}" must not begin with "internal-".`);
            }
            if (loadBalancerName.startsWith('-') || loadBalancerName.endsWith('-')) {
                ret.push(`Load balancer name: "${loadBalancerName}" must not begin or end with a hyphen.`);
            }
            if (!/^[0-9a-z-]+$/i.test(loadBalancerName)) {
                ret.push(`Load balancer name: "${loadBalancerName}" must contain only alphanumeric characters or hyphens.`);
            }
        }
        return ret;
    }
}
exports.BaseLoadBalancer = BaseLoadBalancer;
_a = JSII_RTTI_SYMBOL_1;
BaseLoadBalancer[_a] = { fqn: "@aws-cdk/aws-elasticloadbalancingv2.BaseLoadBalancer", version: "1.155.0" };
//# sourceMappingURL=data:application/json;base64,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