from sklearn.base import BaseEstimator, TransformerMixin
from nilearn._utils import CacheMixin


class BaseGLM(BaseEstimator, TransformerMixin, CacheMixin):
    def generate_report(
            self, contrasts, title=None, bg_img="MNI152TEMPLATE",
            threshold=3.09, alpha=0.001, cluster_threshold=0,
            height_control='fpr', min_distance=8., plot_type='slice',
            display_mode=None, report_dims=(1600, 800)):
        """ Returns HTMLDocument object
        for a report which shows all important aspects of a fitted GLM.
        The object can be opened in a browser, displayed in a notebook,
        or saved to disk as a standalone HTML file.

        The GLM must be fitted and have the computed design matrix(ces).

        Parameters
        ----------
            A fitted first or second level model object.

        contrasts : Dict[string, ndarray] or String or List[String] or
            ndarray or List[ndarray]

            Contrasts information for a first or second level model.

            Example:

                Dict of contrast names and coefficients,
                or list of contrast names
                or list of contrast coefficients
                or contrast name
                or contrast coefficient

                Each contrast name must be a string.
                Each contrast coefficient must be a list or
                numpy array of ints.

            Contrasts are passed to ``contrast_def`` for FirstLevelModel
            (:func:`nilearn.glm.first_level.FirstLevelModel.compute_contrast`)
            & second_level_contrast for SecondLevelModel
            (:func:`nilearn.glm.second_level.SecondLevelModel.compute_contrast`)

        title : String, optional
            If string, represents the web page's title and primary heading,
            model type is sub-heading.
            If None, page titles and headings are autogenerated
            using contrast names.

        bg_img : Niimg-like object, optional
            Default is the MNI152 template (Default='MNI152TEMPLATE')
            See http://nilearn.github.io/manipulating_images/input_output.html
            The background image for mask and stat maps to be plotted on upon.
            To turn off background image, just pass "bg_img=None".

        threshold : float, optional
            Cluster forming threshold in same scale as `stat_img` (either a
            t-scale or z-scale value). Used only if height_control is None.
            Default=3.09

        alpha : float, optional
            Number controlling the thresholding (either a p-value or q-value).
            Its actual meaning depends on the height_control parameter.
            This function translates alpha to a z-scale threshold.
            Default=0.001

        cluster_threshold : int, optional
            Cluster size threshold, in voxels.
            Default=0

        height_control : string or None, optional
            false positive control meaning of cluster forming
            threshold: 'fpr' (default) or 'fdr' or 'bonferroni' or None
            Default='fpr'.

        min_distance : float, optional
            For display purposes only.
            Minimum distance between subpeaks in mm. Default=8mm.

        plot_type : String. ['slice', 'glass'], optional
            Specifies the type of plot to be drawn for the statistical maps.
            Default='slice'.

        display_mode : string, optional
            Default is 'z' if plot_type is 'slice'; '
            ortho' if plot_type is 'glass'.

            Choose the direction of the cuts:
            'x' - sagittal, 'y' - coronal, 'z' - axial,
            'l' - sagittal left hemisphere only,
            'r' - sagittal right hemisphere only,
            'ortho' - three cuts are performed in orthogonal directions.

            Possible values are:
            'ortho', 'x', 'y', 'z', 'xz', 'yx', 'yz',
            'l', 'r', 'lr', 'lzr', 'lyr', 'lzry', 'lyrz'.

        report_dims : Sequence[int, int], optional
            Default is (1600, 800) pixels.
            Specifies width, height (in pixels) of report window
            within a notebook.
            Only applicable when inserting the report into a Jupyter notebook.
            Can be set after report creation using report.width, report.height.

        Returns
        -------
        report_text : HTMLDocument Object
            Contains the HTML code for the GLM Report.

        """
        from nilearn.reporting import make_glm_report
        return make_glm_report(
            self, contrasts, title=title, bg_img=bg_img, threshold=threshold,
            alpha=alpha, cluster_threshold=cluster_threshold,
            height_control=height_control, min_distance=min_distance,
            plot_type=plot_type, display_mode=display_mode,
            report_dims=report_dims)
