import os
import tempfile
from unittest import TestCase

import numpy as np
from tensorflow import keras
from tensorflow.keras import backend as K

from keras_adaptive_softmax import AdaptiveEmbedding


class TestEmbedding(TestCase):

    def test_sample_default(self):
        input_layer = keras.layers.Input(shape=(None,))
        embed_layer = AdaptiveEmbedding(
            input_dim=3,
            output_dim=16,
            return_embeddings=True,
            return_projections=True,
        )(input_layer)
        func = K.function([input_layer], embed_layer)
        outputs = func([np.array([[0, 1, 2]])])
        self.assertTrue(np.allclose(outputs[0], outputs[1]))

    def test_single_projection(self):
        input_layer = keras.layers.Input(shape=(None,))
        embed_layer = AdaptiveEmbedding(
            input_dim=3,
            output_dim=16,
            embed_dim=5,
            return_embeddings=True,
            return_projections=True,
        )(input_layer)
        model = keras.models.Model(input_layer, embed_layer)
        model_path = os.path.join(tempfile.gettempdir(), 'test_ada_embed_%f.h5' % np.random.random())
        model.save(model_path)
        model = keras.models.load_model(model_path, custom_objects={'AdaptiveEmbedding': AdaptiveEmbedding})
        model.summary()

    def test_sample_cutoffs(self):
        embed_0 = np.array([
            [
                0.7562694862279867, -0.7532437781410828, -0.2882295795429552, -1.6990371818805843,
                -0.09864164817566004, -0.5235034477186453, -1.600153091413999, 0.03441732751250957,
            ],
            [
                -0.3680529905261407, 1.1673600332887637, -0.6914459306809843, -0.7645030146906124,
                2.0434827620248606, -0.2743642839675437, 0.04834288951969495, -1.0368596183756285,
            ],
            [
                -0.8440324158987662, 0.05585795322288273, -0.5827731797867599, 1.502853709909658,
                -0.09311037618863122, 1.366316512453695, -0.3834091917878978, -1.2647642860801802,
            ],
            [
                1.5212768184170435, -0.7854311748221854, -0.4674213048014483, -1.0460200278367862,
                0.3705555995848165, -0.12273261562651422, 1.8138708310050653, -0.26957084415202764,
            ],
            [
                -0.15162771245260723, -0.19654664890167275, -1.77930041719533, -0.6987101769248606,
                0.32681036318004547, 0.19156716698736181, 0.8386004334587568, -1.8390076172747616,
            ],
            [
                -1.1363779747587972, -0.15233131547247872, 0.158423477487577, -0.6984487776859649,
                1.2424950830966563, -0.16130616338419873, -1.6298737099566283, 1.7229575808498785,
            ],
            [
                0.613169803410901, -1.5391239758406403, -1.2476893436624792, -0.05514513857644962,
                -0.5537408608863357, -0.9965187549427492, -0.6842234254089083, -1.2420165307152238,
            ],
            [
                -0.4086071455923046, -0.7286151488450243, 1.2938629380821804, 0.7450912596769113,
                -0.13042129128885002, -1.4269400640112133, -0.713571658756806, -0.5036154349645774,
            ],
            [
                0.7326026846217363, 0.12752591749386982, 0.7387086112901271, -1.4161019970745967,
                -0.6396944907214142, -2.0010110577965783, 0.5843029435066284, -0.4033331631189724,
            ],
            [
                1.22301664512685, -0.024541032664251092, -0.27128167541306714, 1.910258142043872,
                -0.9673069099782774, 0.6614265651081772, -1.165650716838653, -0.5085143504562967,
            ],
        ])
        embed_1 = np.array([
            [0.6593494357199338, -0.06233478795012013, 0.3394579881849406, 0.05894554241531747],
            [1.0015451559801243, 0.7487130375684998, -0.4244371286817957, -0.45182923128222996],
            [-0.41965070720383035, -0.2875756074838825, 1.8712603426351773, 2.531083895835167],
            [-0.6800689195006436, -0.39454047242128376, 0.5442439581019961, -0.21672610899025968],
            [-1.3119449289237803, 1.5645034642903253, 1.3203132828621442, 1.7673879116655695],
            [-0.8817194029613362, -0.6655645822150862, 0.2341787847442309, -0.7641095447924122],
            [-0.47497798682688624, 1.0109350638555383, -0.5514102704837403, -0.1450007600387442],
            [-0.531267085230172, 0.12862169808408846, 0.18339345878624577, 1.5279135983387981],
            [0.43338928943049837, 0.2660771849859784, 1.4227633495535283, -0.5072818940455809],
            [0.8704222505796531, 0.9361117741463981, 0.7442665348863866, 0.91392694614948],
        ])
        embed_2 = np.array([
            [1.2712292341556446, 1.009655780936284],
            [0.4420362222435132, 1.5186087787070979],
            [-0.10018465175352317, -0.09182475290216006],
            [-1.246047485363712, 1.6404603895987184],
            [1.4427767754835976, 1.2102150762070925],
        ])
        embed_3 = np.array([
            [0.8285545743394414],
            [0.7111875779008273],
            [0.35799413043562894],
            [-0.15005629449852656],
            [0.6263946579941496],
        ])
        proj_0 = np.array([
            [0.3409731658714878, 0.032745006392315756, 0.668797744010083],
            [-0.3082491589087075, -1.0028023345331745, 0.2122102239605163],
            [-0.3751562822576601, -0.5825445529201775, 0.43389258576225614],
            [0.26067868083146517, 0.8192897299406429, 0.073726048897453],
            [1.1346146882950412, -2.456072992985481, -0.054474463562940736],
            [-1.0283521269636255, -0.1983876737118115, 1.0132159972212373],
            [2.72334361610427, 0.5683724225575054, 2.403638230905517],
            [-0.2137114185905606, 0.3048293347650425, 1.510425235737199],
        ])
        proj_1 = np.array([
            [0.42186259731067743, 0.6034344571434473, 2.362015513199549],
            [-0.9313583984951119, -0.8242699945665621, 0.2596454482698166],
            [0.8871149648450185, -0.663397984939589, -1.195129355668761],
            [0.8016784490871957, 0.13830808473255815, -0.6580242457235711],
        ])
        proj_2 = np.array([
            [1.4802477891158519, 0.12638370704617574, -0.18503256737397666],
            [-0.3900434531439191, 0.14771223879593204, -0.8863321455068343],
        ])
        proj_3 = np.array([[-0.589729339138385, 2.018799784975004, -0.08431336326635828]])

        input_layer = keras.layers.Input(shape=(None,))
        embed_layer = AdaptiveEmbedding(
            input_dim=30,
            output_dim=3,
            embed_dim=8,
            cutoffs=[10, 20, 25],
            div_val=2,
            mask_zero=True,
            return_embeddings=True,
            return_projections=True,
        )
        func = K.function([input_layer], embed_layer(input_layer))
        embed_layer.set_weights([
            embed_0, proj_0,
            embed_1, proj_1,
            embed_2, proj_2,
            embed_3, proj_3,
        ])
        outputs = func([np.array([list(range(30))])])
        expected = np.array([
            [-3.783413887023926, -0.9968423843383789, -4.223631381988525],
            [2.528728485107422, -6.659335613250732, -2.194012403488159],
            [-1.9791769981384277, 0.8412808179855347, -2.137157917022705],
            [6.2075581550598145, 0.31576472520828247, 4.379002094268799],
            [3.3448808193206787, -0.268412709236145, -1.552351474761963],
            [-3.813311815261841, -3.9697980880737305, -2.3214385509490967],
            [-0.06424117088317871, 3.0353987216949463, -4.962082862854004],
            [-0.7221541404724121, 0.6183103322982788, -3.726100444793701],
            [2.573601245880127, 0.48284363746643066, -0.4642190933227539],
            [-3.8191750049591064, 3.2147698402404785, -2.0111422538757324],
            [0.6846045255661011, 0.23221178352832794, 1.0967247486114502],
            [-1.013551950454712, 0.20630428194999695, 3.3646368980407715],
            [3.7799394130706787, -0.9075126051902771, -4.967802047729492],
            [0.3896251916885376, -0.4761944115161896, -2.216604709625244],
            [0.5775725841522217, -2.712695360183716, -5.433547496795654],
            [-0.1569119393825531, -0.24449113011360168, -2.0325169563293457],
            [-1.7473266124725342, -0.7741519212722778, -0.10500013828277588],
            [1.04367196559906, -0.33694392442703247, -2.4460482597351074],
            [0.7904950380325317, -0.971816897392273, -0.2738245725631714],
            [0.8882685303688049, -0.6137074828147888, 0.8081271648406982],
            [1.487924575805664, 0.3098011910915375, -1.130109190940857],
            [0.06199967861175537, 0.2801832854747772, -1.4277828931808472],
            [-0.1124824583530426, -0.026225347071886063, 0.09992465376853943],
            [-2.484309673309326, 0.0848359763622284, -1.2234333753585815],
            [1.6636306047439575, 0.3611070513725281, -1.3396131992340088],
            [-0.4886229634284973, 1.6726857423782349, -0.06985822319984436],
            [-0.4194082021713257, 1.435745358467102, -0.059962619096040726],
            [-0.2111196517944336, 0.7227184772491455, -0.030183689668774605],
            [0.08849260210990906, -0.30293360352516174, 0.012651749886572361],
            [-0.36940330266952515, 1.264565348625183, -0.05281343683600426],
        ])
        self.assertTrue(np.allclose(expected, outputs[0][0]))
