from pathlib import Path
from typing import Callable, Optional
from lupa import LuaRuntime
import logging
import shutil
import re
pre_edited_cs = __import__("pre-edited-cs")


CSVERSION = 3

class CaverException(Exception):
    pass

def patch_files(patch_data: dict, output_dir: Path, progress_update: Callable[[str, float], None]):
    progress_update("Copying base files...", 0.0)
    ensure_base_files_exist(output_dir)

    mapcount = len(patch_data["maps"].keys())

    lua_file = Path(__file__).parent.joinpath("tsc_file.lua").read_text()
    try:
        TscFile = LuaRuntime().execute(lua_file)
    except Exception as e:
        print(lua_file)
        raise e
    
    for i, (mapname, mapdata) in enumerate(patch_data["maps"].items()):
        progress_update(f"Patching {mapname}...", i/mapcount)
        patch_map(mapname, mapdata, TscFile, output_dir)
    
    progress_update("Copying MyChar...", 1.0)
    patch_mychar(patch_data["mychar"], output_dir)

    progress_update("Copying hash...", 1.0)
    patch_hash(patch_data["hash"], output_dir)

def ensure_base_files_exist(output_dir: Path):
    internal_copy = pre_edited_cs.get_path()

    version = output_dir.joinpath("data", "Stage", "_version.txt")
    keep_existing_files = version.exists() and int(version.read_text()) >= CSVERSION

    def should_ignore(path: str, names: list[str]):
        base = ["__init__.py", "__pycache__", "ScriptSource"]
        if keep_existing_files:
            p = Path(path)
            base.extend([p.joinpath(name) for name in names if p.joinpath(name).exists() and p.joinpath(name).is_file()])
        return base
    
    try:
        shutil.copytree(internal_copy, output_dir, ignore=should_ignore, dirs_exist_ok=True)
    except shutil.Error:
        raise CaverException("Error copying base files. Ensure the directory is not read-only, and that Doukutsu.exe is closed.")
    output_dir.joinpath("data", "Plaintext").mkdir(exist_ok=True)

def patch_map(mapname: str, mapdata: dict[str, dict], TscFile, output_dir: Path):
    mappath = output_dir.joinpath("data", "Stage", f"{mapname}.tsc")
    tsc_file = TscFile.new(TscFile, mappath.read_bytes(), logging.getLogger("caver"))

    for event, script in mapdata["pickups"].items():
        TscFile.placeScriptAtEvent(tsc_file, script, event, mapname)
    
    for event, song in mapdata["music"].items():
        TscFile.placeSongAtCue(tsc_file, song["song_id"], event, song["original_id"], mapname)
    
    for event, script in mapdata["entrances"].items():
        needle = "<EVE...." # TODO: create a proper pattern
        TscFile.placeScriptAtEvent(tsc_file, script, event, mapname, needle)
    
    for event, hint in mapdata["hints"].items():
        script = create_hint_script(hint["text"], hint.get("facepic", "0000") != "0000", hint.get("ending", "<END"))
        TscFile.placeScriptAtEvent(tsc_file, script, event, mapname)

    chars = TscFile.getText(tsc_file).values()
    mappath.write_bytes(bytes(chars))
    output_dir.joinpath("data", "Plaintext", f"{mapname}.txt").write_text(TscFile.getPlaintext(tsc_file))

def patch_mychar(mychar: Optional[str], output_dir: Path):
    if mychar is None:
        return
    mychar_img = Path(mychar).read_bytes()
    output_dir.joinpath("data", "MyChar.bmp").write_bytes(mychar_img)

def patch_hash(hash: list[int], output_dir: Path):
    hash_strings = [f"{num:04d}" for num in hash]
    hash_string = ",".join(hash_strings)
    output_dir.joinpath("data", "hash.txt").write_text(hash_string)

def create_hint_script(text: str, facepic: bool, ending: str) -> str:
    """
    A desperate attempt to generate valid <MSG text. Fills one text box (up to three lines). Attempts to wrap words elegantly.
    """
    hard_limit = 35
    msgbox_limit = 26 if facepic else hard_limit
    pattern = r' [^ ]*$'
    line1, line2, line3 = "", "", ""

    split = 0
    line1 = text[split:split+msgbox_limit]

    match = next(re.finditer(pattern, line1), None)
    if match is not None and len(text) > msgbox_limit:
        line1 = line1[:match.start(0)]
        split += match.start(0)+1
        if split % hard_limit != 0:
            line2 = "\r\n"
        line2 += text[split:split+msgbox_limit]

        match = next(re.finditer(pattern, line2), None)
        if match is not None and len(text) > msgbox_limit*2:
            line2 = line2[:match.start(0)]
            if split % hard_limit != 0:
                split -= 2
            split += match.start(0)+1
            if split % hard_limit != 0:
                line3 = "\r\n"
            line3 += text[split:split+msgbox_limit]
    
    return f"<PRI<MSG<TUR{line1}{line2}{line3}<NOD{ending}"

if __name__ == "__main__":
    data = {
        'maps': {'MiBox': {'pickups': {}, 'music': {}, 'entrances': {}, 'hints': {}}, 'EggR2': {'pickups': {'0303': '<EVE0059'}, 'music': {'0090': {'song_id': '0065', 'original_id': '0008'}, '0091': {'song_id': '0050', 'original_id': '0008'}, '0092': {'song_id': '0065', 'original_id': '0008'}, '0093': {'song_id': '0050', 'original_id': '0008'}, '0094': {'song_id': '0047', 'original_id': '0008'}, '1000': {'song_id': '0053', 'original_id': '0008'}, '0304': {'song_id': '0043', 'original_id': '0004'}}, 'entrances': {}, 'hints': {}}, 'Hell3': {'pickups': {'0400': '<EVE0033'}, 'music': {'0090': {'song_id': '0055', 'original_id': '0036'}, '0091': {'song_id': '0053', 'original_id': '0036'}, '0092': {'song_id': '0059', 'original_id': '0036'}, '0093': {'song_id': '0047', 'original_id': '0036'}, '0094': {'song_id': '0064', 'original_id': '0036'}, '0300': {'song_id': '0045', 'original_id': '0007'}}, 'entrances': {}, 'hints': {}}, 'Ring2': {'pickups': {}, 'music': {'0090': {'song_id': '0050', 'original_id': '0030'}, '0091': {'song_id': '0048', 'original_id': '0030'}, '0092': {'song_id': '0053', 'original_id': '0030'}, '0093': {'song_id': '0043', 'original_id': '0030'}, '0094': {'song_id': '0059', 'original_id': '0030'}, '0098': {'song_id': '0045', 'original_id': '0030'}, '0420': {'song_id': '0051', 'original_id': '0030'}, '0502': {'song_id': '0055', 'original_id': '0007'}, '0410': {'song_id': '0053', 'original_id': '0031'}, '0099': {'song_id': '0059', 'original_id': '0018'}}, 'entrances': {}, 'hints': {}}, '0': {'pickups': {}, 'music': {'0100': {'song_id': '0045', 'original_id': '0033'}, '0200': {'song_id': '0047', 'original_id': '0001'}}, 'entrances': {}, 'hints': {}}, 'Jenka1': {'pickups': {}, 'music': {'0090': {'song_id': '0065', 'original_id': '0019'}, '0091': {'song_id': '0065', 'original_id': '0019'}, '0092': {'song_id': '0044', 'original_id': '0019'}, '0093': {'song_id': '0050', 'original_id': '0019'}, '0094': {'song_id': '0054', 'original_id': '0019'}}, 'entrances': {}, 'hints': {'0201': {'text': "perhaps I'll give you a 3HP Life Capsule in return...", 'facepic': '0000', 'ending': '<END'}, '0202': {'text': "perhaps I'll give you the Ma Pignon in return...", 'facepic': '0000', 'ending': '<END'}}}, 'River': {'pickups': {}, 'music': {'0090': {'song_id': '0045', 'original_id': '0038'}, '0091': {'song_id': '0055', 'original_id': '0038'}, '0092': {'song_id': '0059', 'original_id': '0038'}, '0093': {'song_id': '0047', 'original_id': '0038'}, '0094': {'song_id': '0056', 'original_id': '0038'}, '0095': {'song_id': '0052', 'original_id': '0038'}}, 'entrances': {}, 'hints': {}}, 'Mimi': {'pickups': {'0202': '<EVE0089'}, 'music': {'0090': {'song_id': '0043', 'original_id': '0009'}, '0091': {'song_id': '0060', 'original_id': '0009'}, '0092': {'song_id': '0051', 'original_id': '0009'}, '0093': {'song_id': '0053', 'original_id': '0009'}, '0094': {'song_id': '0044', 'original_id': '0009'}, '0302': {'song_id': '0057', 'original_id': '0009'}, '0095': {'song_id': '0065', 'original_id': '0028'}, '0096': {'song_id': '0042', 'original_id': '0028'}, '0097': {'song_id': '0052', 'original_id': '0028'}, '0098': {'song_id': '0055', 'original_id': '0028'}, '0099': {'song_id': '0056', 'original_id': '0028'}}, 'entrances': {}, 'hints': {}}, 'Jail1': {'pickups': {'0301': '<EVE0009'}, 'music': {'0090': {'song_id': '0045', 'original_id': '0008'}, '0091': {'song_id': '0045', 'original_id': '0008'}, '0092': {'song_id': '0048', 'original_id': '0008'}, '0093': {'song_id': '0054', 'original_id': '0008'}, '0094': {'song_id': '0060', 'original_id': '0008'}, '0220': {'song_id': '0047', 'original_id': '0008'}}, 'entrances': {}, 'hints': {}}, 'Statue': {'pickups': {}, 'music': {'0100': {'song_id': '0065', 'original_id': '0024'}}, 'entrances': {}, 'hints': {'0301': {'text': 'The Little Man.', 'facepic': 
        '0000', 'ending': '<END'}, '0302': {'text': 'A Progressive Missile Launcher.', 'facepic': '0000', 'ending': '<END'}}}, 'WeedS': {'pickups': {}, 'music': {}, 'entrances': {}, 'hints': {}}, 'WeedB': {'pickups': {'0301': '<EVE0065'}, 'music': {'0302': {'song_id': '0050', 'original_id': '0004'}}, 'entrances': {}, 'hints': {}}, 'Priso2': {'pickups': {'0300': '<EVE0057'}, 'music': {'0090': {'song_id': '0052', 'original_id': '0029'}, '0091': {'song_id': '0048', 'original_id': '0029'}, '0092': {'song_id': '0060', 'original_id': '0029'}, '0093': {'song_id': '0055', 'original_id': '0029'}, '0094': {'song_id': '0045', 'original_id': '0029'}, '0250': {'song_id': '0055', 'original_id': '0029'}, '0241': {'song_id': '0045', 'original_id': '0004'}}, 'entrances': {}, 'hints': {}}, 'Drain': {'pickups': {}, 'music': {'0090': {'song_id': '0054', 'original_id': '0008'}, '0091': {'song_id': '0053', 'original_id': '0008'}, '0092': {'song_id': '0059', 'original_id': '0008'}, '0093': {'song_id': '0045', 'original_id': '0008'}, '0094': {'song_id': '0047', 'original_id': '0008'}, '0150': {'song_id': '0065', 'original_id': '0008'}, '0095': {'song_id': '0044', 'original_id': '0023'}, '0096': {'song_id': '0052', 'original_id': '0023'}, '0097': {'song_id': '0052', 'original_id': '0023'}}, 'entrances': {}, 'hints': {}}, 'Pixel': {'pickups': {}, 'music': {'0090': {'song_id': '0043', 'original_id': '0014'}, '0091': {'song_id': '0047', 'original_id': '0014'}, '0092': {'song_id': '0050', 'original_id': '0014'}, '0093': {'song_id': '0048', 'original_id': '0014'}, '0094': {'song_id': '0060', 'original_id': '0014'}, '0253': {'song_id': '0047', 'original_id': '0014'}}, 'entrances': {}, 'hints': {}}, 'Mapi': {'pickups': {'0501': '<EVE0030', '0202': '<EVE0086'}, 'music': {'0420': {'song_id': '0051', 'original_id': '0004'}}, 'entrances': {}, 'hints': {}}, 'MazeW': {'pickups': {}, 'music': {'0090': {'song_id': '0054', 'original_id': '0037'}, '0091': {'song_id': '0060', 'original_id': '0037'}, '0092': {'song_id': '0060', 'original_id': '0037'}, '0093': {'song_id': '0055', 'original_id': '0037'}, '0094': {'song_id': '0046', 'original_id': '0037'}, '1000': {'song_id': '0057', 'original_id': '0037'}, '0302': {'song_id': '0059', 'original_id': '0007'}}, 'entrances': {}, 'hints': {}}, 'Stream': {'pickups': {}, 'music': {'0090': {'song_id': '0054', 'original_id': '0018'}, '0091': {'song_id': '0051', 'original_id': '0018'}, '0092': {'song_id': '0056', 'original_id': '0018'}, '0093': {'song_id': '0045', 'original_id': '0018'}, '0094': {'song_id': '0042', 'original_id': '0018'}}, 'entrances': {}, 'hints': {}}, 'Blcny2': {'pickups': {}, 'music': {'0090': {'song_id': '0057', 'original_id': '0018'}, '0091': {'song_id': '0043', 'original_id': '0018'}, '0092': {'song_id': '0046', 'original_id': '0018'}, '0093': {'song_id': '0046', 'original_id': '0018'}, '0094': {'song_id': '0047', 'original_id': '0018'}, '0310': {'song_id': '0053', 'original_id': '0018'}, '0400': {'song_id': '0045', 'original_id': '0035'}}, 'entrances': {}, 'hints': {}}, 'EgEnd2': {'pickups': {}, 'music': {}, 'entrances': {}, 'hints': {}}, 'Santa': {'pickups': {'0302': '<EVE0010', '0501': '<EVE0058'}, 'music': {'0090': {'song_id': '0064', 'original_id': '0002'}, '0091': {'song_id': '0059', 'original_id': '0002'}, '0092': {'song_id': '0057', 'original_id': '0002'}, '0093': {'song_id': '0045', 'original_id': '0002'}, '0094': {'song_id': '0053', 'original_id': '0002'}, '0099': {'song_id': '0055', 'original_id': '0028'}}, 'entrances': {}, 'hints': {}}, 'Ostep': {'pickups': {}, 'music': {}, 'entrances': {}, 'hints': {}}, 'CurlyS': {'pickups': {'0421': '<EVE0064', '0401': '<EVE0064'}, 'music': {}, 'entrances': {}, 'hints': {}}, 'Cthu': {'pickups': {}, 'music': {'0090': {'song_id': '0045', 'original_id': '0008'}, '0091': {'song_id': '0050', 'original_id': '0008'}, '0092': {'song_id': '0057', 'original_id': '0008'}, '0093': {'song_id': '0042', 'original_id': '0008'}, '0094': {'song_id': '0045', 'original_id': '0008'}}, 'entrances': {}, 'hints': {'0200': {'text': 'Did you know that Bigfoot and Santa Claus exist in the Metroid Prime canon?', 'facepic': '0000', 'ending': '<END'}}}, 'Sand': {'pickups': {'0503': '<EVE0064', '0502': '<EVE0072', '0423': '<EVE0064', '0422': '<EVE0064', '0421': '<EVE0070'}, 'music': {'0090': {'song_id': '0054', 'original_id': '0006'}, '0091': {'song_id': '0042', 'original_id': '0006'}, '0092': {'song_id': '0051', 
        'original_id': '0006'}, '0093': {'song_id': '0048', 'original_id': '0006'}, '0094': {'song_id': '0042', 'original_id': '0006'}, '0099': {'song_id': '0044', 'original_id': '0006'}, '0210': {'song_id': '0050', 'original_id': '0006'}, '0601': {'song_id': '0065', 'original_id': '0006'}, '0202': {'song_id': '0044', 'original_id': '0007'}}, 'entrances': {}, 'hints': {}}, 'Eggs2': {'pickups': {'0321': '<EVE0077'}, 'music': {'0090': {'song_id': '0057', 'original_id': '0027'}, '0091': {'song_id': '0043', 'original_id': '0027'}, '0092': {'song_id': '0053', 'original_id': '0027'}, '0093': {'song_id': '0052', 'original_id': '0027'}, '0094': {'song_id': '0057', 'original_id': '0027'}, '0099': {'song_id': '0062', 'original_id': '0027'}}, 'entrances': {}, 'hints': {'0210': {'text': 'They say the Sprinkler can be found in Grasstown.', 'facepic': '0000', 'ending': '<END'}}}, 'Cemet': {'pickups': {'0301': '<EVE0030', '0202': '<EVE0035'}, 'music': {'0090': {'song_id': '0047', 'original_id': '0012'}, '0091': {'song_id': '0042', 'original_id': '0012'}, '0092': {'song_id': '0060', 'original_id': '0012'}, '0093': {'song_id': '0060', 'original_id': '0012'}, '0094': {'song_id': '0054', 'original_id': '0012'}}, 'entrances': {}, 'hints': {}}, 'lounge': {'pickups': {'0204': '<EVE0076'}, 'music': {}, 'entrances': {}, 'hints': {}}, 'Chako': {'pickups': {'0211': '<EVE0002'}, 'music': {'0090': {'song_id': '0065', 'original_id': '0002'}, '0091': {'song_id': '0050', 'original_id': '0002'}, '0092': {'song_id': '0050', 'original_id': '0002'}, '0093': {'song_id': '0053', 'original_id': '0002'}, '0094': {'song_id': '0043', 'original_id': '0002'}, '0099': {'song_id': '0046', 'original_id': '0028'}}, 'entrances': {}, 'hints': {}}, 'Hell2': {'pickups': {}, 'music': {'0090': {'song_id': '0046', 'original_id': '0036'}, '0091': {'song_id': '0053', 'original_id': '0036'}, '0092': {'song_id': '0050', 'original_id': '0036'}, '0093': {'song_id': '0064', 'original_id': '0036'}, '0094': {'song_id': '0055', 'original_id': '0036'}}, 'entrances': {}, 'hints': {}}, 'Malco': {'pickups': {'0350': '<EVE0005'}, 'music': {'0090': {'song_id': '0042', 'original_id': '0008'}, '0091': {'song_id': '0046', 'original_id': '0008'}, '0092': {'song_id': '0046', 'original_id': '0008'}, '0093': {'song_id': '0043', 'original_id': '0008'}, '0094': {'song_id': '0065', 'original_id': '0008'}, '0203': {'song_id': '0055', 'original_id': '0008'}, '0200': {'song_id': '0062', 'original_id': '0011'}, '0204': {'song_id': '0057', 'original_id': '0004'}}, 'entrances': {}, 'hints': {'0306': {'text': 'BUT ALL I KNOW HOW TO DO IS MAKE THE SNAKE...', 'facepic': '0000', 'ending': '<END'}}}, 'Priso1': {'pickups': {}, 'music': {'0090': {'song_id': '0053', 'original_id': '0029'}, '0091': {'song_id': '0053', 'original_id': '0029'}, '0092': {'song_id': '0043', 'original_id': '0029'}, '0093': {'song_id': '0065', 'original_id': '0029'}, '0094': {'song_id': '0053', 'original_id': '0029'}}, 'entrances': {}, 'hints': {}}, 'CentW': {'pickups': {}, 'music': {}, 'entrances': {}, 'hints': {}}, 'EggR': {'pickups': {'0301': '<EVE0071'}, 'music': {'0090': {'song_id': '0057', 'original_id': '0008'}, '0091': {'song_id': '0054', 'original_id': '0008'}, '0092': {'song_id': '0057', 'original_id': '0008'}, '0093': {'song_id': '0047', 'original_id': '0008'}, '0094': {'song_id': '0051', 'original_id': '0008'}}, 'entrances': {}, 'hints': {}}, 'MazeB': {'pickups': {'0502': '<EVE0053'}, 'music': {'0090': {'song_id': '0059', 'original_id': '0008'}, '0091': {'song_id': '0055', 'original_id': '0008'}, '0092': {'song_id': '0045', 'original_id': '0008'}, '0093': {'song_id': '0057', 'original_id': '0008'}, '0094': {'song_id': '0056', 'original_id': '0008'}, '0099': {'song_id': '0062', 'original_id': '0008'}}, 'entrances': {}, 'hints': {}}, 'Eggs': {'pickups': {'0403': '<EVE0014', '0404': '<EVE0067'}, 'music': {'0090': {'song_id': '0052', 'original_id': '0001'}, '0091': {'song_id': '0055', 'original_id': '0001'}, '0092': {'song_id': '0045', 'original_id': '0001'}, '0093': {'song_id': '0064', 'original_id': '0001'}, '0094': {'song_id': '0042', 'original_id': '0001'}, '0095': {'song_id': '0055', 'original_id': '0001'}, '0099': {'song_id': '0062', 'original_id': '0001'}, '0503': {'song_id': '0059', 'original_id': '0001'}, '0600': {'song_id': '0057', 'original_id': '0004'}}, 'entrances': {}, 'hints': {'0200': {'text': 'The Fireball can be found exactly 3 rooms away from the Clinic Key.', 'facepic': '0000', 'ending': '<END'}}}, 'WeedD': {'pickups': {'0305': '<EVE0008'}, 'music': {}, 'entrances': {}, 'hints': {}}, 'Jenka2': {'pickups': {'0221': '<EVE0012'}, 'music': {'0090': {'song_id': '0055', 'original_id': '0019'}, '0091': {'song_id': '0065', 'original_id': '0019'}, '0092': {'song_id': '0059', 'original_id': '0019'}, '0093': {'song_id': '0057', 'original_id': '0019'}, '0095': {'song_id': '0062', 'original_id': '0019'}, '0200': {'song_id': '0042', 'original_id': '0019'}, '0094': {'song_id': '0045', 'original_id': '0011'}}, 'entrances': {}, 'hints': {}}, 'Cent': {'pickups': {'0417': '<EVE0078', '0324': '<EVE0087', '0268': '<EVE0004', '0501': '<EVE0054', '0452': '<EVE0083'}, 'music': {'0090': {'song_id': '0064', 'original_id': '0024'}, '0091': {'song_id': 
        '0052', 'original_id': '0024'}, '0092': {'song_id': '0046', 'original_id': '0024'}, '0093': {'song_id': '0060', 'original_id': '0024'}, '0094': {'song_id': '0064', 'original_id': '0024'}}, 'entrances': {}, 'hints': {'0310': {'text': 'A Progressive Polar Star can be found up to 3 rooms away from Sand Zone - Sand Zone.', 'facepic': '0000', 'ending': '<END'}}}, 'Gard': {'pickups': {'0602': '<EVE0084'}, 'music': {'0502': {'song_id': '0043', 'original_id': '0004'}}, 'entrances': {}, 'hints': {}}, 'Little': {'pickups': {'0204': '<EVE0080'}, 'music': {'0090': {'song_id': '0045', 'original_id': '0002'}, 
        '0091': {'song_id': '0064', 'original_id': '0002'}, '0092': {'song_id': '0051', 'original_id': '0002'}, '0093': {'song_id': '0048', 'original_id': '0002'}, '0094': {'song_id': '0060', 'original_id': '0002'}}, 'entrances': {}, 'hints': {'0212': {'text': 'He was exploring the island with the Tow Rope...', 'facepic': '0000', 'ending': '<END'}}}, 'Ring1': {'pickups': {}, 'music': {'0090': {'song_id': '0065', 'original_id': '0030'}, '0091': {'song_id': '0055', 'original_id': '0030'}, '0092': {'song_id': '0054', 'original_id': '0030'}, '0093': {'song_id': '0062', 'original_id': '0030'}, '0094': {'song_id': '0055', 'original_id': '0030'}, '0097': {'song_id': '0059', 'original_id': '0030'}, '0098': {'song_id': '0046', 'original_id': '0030'}, '0402': {'song_id': '0042', 'original_id': '0030'}, '0502': 
        {'song_id': '0042', 'original_id': '0007'}, '0503': {'song_id': '0060', 'original_id': '0007'}, '0099': {'song_id': '0046', 'original_id': '0018'}}, 'entrances': {}, 'hints': {}}, 'MazeO': {'pickups': 
        {'0401': '<EVE0014', '0305': '<EVE0074'}, 'music': {'0090': {'song_id': '0057', 'original_id': '0002'}, '0091': {'song_id': '0054', 'original_id': '0002'}, '0092': {'song_id': '0050', 'original_id': '0002'}, '0093': {'song_id': '0043', 'original_id': '0002'}, '0094': {'song_id': '0045', 'original_id': '0002'}}, 'entrances': {}, 'hints': {}}, 'Egg6': {'pickups': {'0201': '<EVE0081'}, 'music': {}, 'entrances': {}, 'hints': {}}, 'MazeI': {'pickups': {'0301': '<EVE0085'}, 'music': {'0090': {'song_id': '0044', 'original_id': '0019'}, '0091': {'song_id': '0048', 'original_id': '0019'}, '0092': {'song_id': '0048', 'original_id': '0019'}, '0093': {'song_id': '0064', 'original_id': '0019'}, '0094': {'song_id': '0051', 'original_id': '0019'}, '0400': {'song_id': '0059', 'original_id': '0019'}, '0601': {'song_id': '0048', 'original_id': '0019'}}, 'entrances': {}, 'hints': {'0500': {'text': 'The Broken Sprinkler can be found exactly 4 rooms away from Egg Corridor - Egg No. 00.', 'facepic': '0000', 'ending': '<END'}, '0502': {'text': 'They say a major item can be found in Clinic Ruins.', 'facepic': '0000', 'ending': '<END'}}}, 'Hell1': {'pickups': {'0401': '<EVE0082'}, 'music': {'0090': {'song_id': '0044', 'original_id': '0036'}, '0091': {'song_id': '0047', 'original_id': '0036'}, '0092': {'song_id': '0054', 'original_id': '0036'}, '0094': {'song_id': '0054', 'original_id': '0036'}, '0096': {'song_id': '0053', 'original_id': '0036'}}, 'entrances': {}, 'hints': {}}, 'Itoh': {'pickups': {'0405': '<EVE0063'}, 'music': {'0090': {'song_id': '0052', 'original_id': '0008'}, '0091': {'song_id': '0054', 'original_id': '0008'}, '0092': {'song_id': '0051', 'original_id': '0008'}, '0093': {'song_id': '0060', 'original_id': '0008'}, '0094': {'song_id': '0053', 'original_id': '0008'}, '0095': {'song_id': 
        '0057', 'original_id': '0008'}}, 'entrances': {}, 'hints': {}}, 'Comu': {'pickups': {'0303': '<EVE0014'}, 'music': {'0090': {'song_id': '0065', 'original_id': '0009'}, '0091': {'song_id': '0047', 'original_id': '0009'}, '0092': {'song_id': '0060', 'original_id': '0009'}, '0093': {'song_id': '0062', 'original_id': '0009'}, '0094': {'song_id': '0051', 'original_id': '0009'}, '0095': {'song_id': '0044', 'original_id': '0028'}, '0096': {'song_id': '0054', 'original_id': '0028'}, '0097': {'song_id': '0065', 'original_id': '0028'}, '0098': {'song_id': '0056', 'original_id': '0028'}, '0099': {'song_id': '0047', 'original_id': '0028'}}, 'entrances': {}, 'hints': {}}, 'Prefa1': {'pickups': {}, 'music': {'0090': {'song_id': '0051', 'original_id': '0030'}, '0091': {'song_id': '0065', 'original_id': '0030'}, '0092': {'song_id': '0065', 'original_id': '0030'}, '0093': {'song_id': '0056', 'original_id': '0030'}, '0094': {'song_id': '0053', 'original_id': '0030'}}, 'entrances': {}, 'hints': {}}, 'Ballo1': {'pickups': {}, 'music': {'0095': {'song_id': '0059', 'original_id': '0039'}, '0500': {'song_id': '0050', 'original_id': '0004'}, '0900': {'song_id': '0046', 'original_id': '0007'}, '1000': {'song_id': '0055', 'original_id': '0032'}}, 'entrances': {}, 'hints': {}}, 'Blcny1': {'pickups': {}, 'music': {'0090': {'song_id': '0051', 'original_id': '0030'}, '0091': {'song_id': '0059', 'original_id': '0030'}, '0092': {'song_id': '0057', 'original_id': '0030'}, '0093': {'song_id': '0046', 'original_id': '0030'}, '0094': {'song_id': '0047', 'original_id': '0030'}}, 'entrances': {}, 'hints': {}}, 'MazeS': {'pickups': {'0202': '<EVE0052'}, 'music': {'0090': {'song_id': '0051', 'original_id': '0008'}, '0091': {'song_id': '0044', 'original_id': '0008'}, '0092': {'song_id': '0046', 'original_id': '0008'}, 
        '0093': {'song_id': '0054', 'original_id': '0008'}, '0094': {'song_id': '0064', 'original_id': '0008'}, '0310': {'song_id': '0055', 'original_id': '0008'}, '0600': {'song_id': '0065', 'original_id': '0008'}, '0321': {'song_id': '0051', 'original_id': '0004'}}, 'entrances': {}, 'hints': {}}, 'Shelt': {'pickups': {}, 'music': {'0090': {'song_id': '0054', 'original_id': '0008'}, '0091': {'song_id': '0056', 'original_id': '0008'}, '0092': {'song_id': '0065', 'original_id': '0008'}, '0093': {'song_id': '0064', 'original_id': '0008'}, '0094': {'song_id': '0055', 'original_id': '0008'}, '0099': {'song_id': '0043', 'original_id': '0008'}}, 'entrances': {}, 'hints': {}}, 'Pool': {'pickups': {'0412': '<EVE0013', '0301': '<EVE0013'}, 'music': {'0090': {'song_id': '0050', 'original_id': '0009'}, '0091': {'song_id': '0046', 'original_id': '0009'}, '0092': {'song_id': '0053', 'original_id': '0009'}, '0093': {'song_id': '0060', 'original_id': '0009'}, '0094': {'song_id': '0047', 'original_id': '0009'}, '0095': {'song_id': '0052', 'original_id': '0028'}, '0096': {'song_id': '0052', 'original_id': '0028'}, '0097': {'song_id': '0062', 'original_id': '0028'}, '0098': {'song_id': '0056', 'original_id': '0028'}, '0099': {'song_id': '0064', 'original_id': '0028'}, '0410': {'song_id': '0064', 'original_id': '0028'}}, 'entrances': {}, 'hints': {}}, 'Start': {'pickups': {'0201': '<FL+6000<FL+4011<FL+4012<MP+0040<MP+0043<MP+0032<MP+0033<MP+0036<FL+0301<FL+0302<FL+1641<FL+1642<FL+0320<FL+0321<FL+6201<TRA0001:0094:0008:0004'}, 'music': {'0090': {'song_id': '0054', 'original_id': '0008'}, '0091': {'song_id': '0062', 'original_id': '0008'}, '0092': {'song_id': '0056', 'original_id': '0008'}, '0093': {'song_id': '0064', 'original_id': '0008'}, '0094': {'song_id': '0050', 'original_id': '0008'}}, 'entrances': {}, 'hints': {}}, 'Oside': {'pickups': {}, 'music': {'0090': {'song_id': '0045', 'original_id': '0025'}, '0091': {'song_id': '0062', 'original_id': '0025'}, '0092': {'song_id': '0052', 'original_id': '0025'}, '0093': {'song_id': '0053', 'original_id': '0025'}, '0094': {'song_id': '0065', 'original_id': '0025'}, '0402': {'song_id': '0054', 'original_id': '0026'}}, 'entrances': {}, 'hints': {}}, 'Clock': {'pickups': {'0300': '<EVE0069'}, 'music': {}, 'entrances': {}, 'hints': {}}, 'MazeH': {'pickups': {}, 'music': {'0090': {'song_id': '0051', 'original_id': '0019'}, '0091': {'song_id': '0054', 'original_id': '0019'}, '0092': {'song_id': '0057', 'original_id': 
        '0019'}, '0093': {'song_id': '0055', 'original_id': '0019'}, '0094': {'song_id': '0065', 'original_id': '0019'}}, 'entrances': {}, 'hints': {}}, 'Curly': {'pickups': {'0518': '<EVE0061'}, 'music': {'0090': {'song_id': '0044', 'original_id': '0002'}, '0091': {'song_id': '0052', 'original_id': '0002'}, '0092': {'song_id': '0056', 'original_id': '0002'}, '0093': {'song_id': '0051', 'original_id': '0002'}, '0094': {'song_id': '0055', 'original_id': '0002'}, '0300': {'song_id': '0045', 'original_id': '0004'}}, 'entrances': {}, 'hints': {}}, 'Frog': {'pickups': {'0300': '<EVE0068'}, 'music': {'0090': {'song_id': '0052', 'original_id': '0008'}, '0091': {'song_id': '0048', 'original_id': '0008'}, '0092': {'song_id': '0046', 'original_id': '0008'}, '0093': {'song_id': '0052', 'original_id': '0008'}, '0094': {'song_id': '0060', 'original_id': '0008'}, '1000': {'song_id': '0062', 'original_id': '0008'}, '0202': {'song_id': '0053', 'original_id': '0007'}}, 'entrances': {}, 'hints': {}}, 'SandE': {'pickups': {}, 'music': {'0090': {'song_id': '0062', 'original_id': '0006'}, '0091': {'song_id': '0047', 'original_id': '0006'}, '0092': {'song_id': '0045', 'original_id': '0006'}, '0093': {'song_id': '0053', 'original_id': '0006'}, '0094': {'song_id': '0050', 'original_id': '0006'}, '0600': {'song_id': '0047', 'original_id': '0011'}}, 'entrances': {}, 'hints': {}}, 'Barr': {'pickups': {}, 'music': {'0090': {'song_id': '0059', 'original_id': '0008'}, '0091': {'song_id': '0053', 'original_id': '0008'}, '0092': {'song_id': '0043', 'original_id': '0008'}, '0093': {'song_id': '0046', 'original_id': '0008'}, '0094': {'song_id': '0050', 'original_id': '0008'}, '0402': {'song_id': '0053', 'original_id': '0008'}, '1001': {'song_id': '0056', 'original_id': '0008'}, '1000': {'song_id': '0042', 'original_id': '0004'}}, 'entrances': {}, 'hints': {}}, 'Egg1': {'pickups': {}, 'music': {}, 'entrances': {}, 'hints': {}}, 'Pens1': {'pickups': {'0652': '<EVE0002'}, 'music': {'0090': {'song_id': '0054', 'original_id': '0002'}, '0091': {'song_id': '0059', 'original_id': '0002'}, '0092': {'song_id': '0059', 'original_id': '0002'}, '0093': {'song_id': '0045', 'original_id': '0002'}, '0094': {'song_id': '0060', 'original_id': '0002'}, '0099': {'song_id': '0047', 'original_id': '0002'}, '0095': {'song_id': '0062', 'original_id': '0014'}, '0098': {'song_id': '0051', 'original_id': '0014'}}, 'entrances': {}, 'hints': {}}, 'EgEnd1': {'pickups': {}, 'music': {}, 'entrances': {}, 'hints': {}}, 'Weed': {'pickups': {'0303': '<EVE0014', '0801': '<EVE0079', '0800': '<EVE0030', '0702': '<EVE0066', '0701': '<EVE0075', '0700': '<EVE0007'}, 'music': {'0090': {'song_id': '0052', 'original_id': '0005'}, '0091': {'song_id': '0045', 'original_id': '0005'}, '0092': {'song_id': '0044', 'original_id': '0005'}, '0093': {'song_id': '0057', 'original_id': '0005'}, '0094': {'song_id': '0051', 'original_id': '0005'}, '0098': {'song_id': '0062', 'original_id': '0005'}, '0099': {'song_id': '0047', 'original_id': '0005'}, '0600': {'song_id': '0054', 'original_id': '0005'}}, 'entrances': {}, 'hints': {'0205': {'text': 'By this point in your run, you should have consumed at least 200 mL of water to maintain optimum hydration.', 'facepic': '0000', 'ending': '<END'}, '0201': {'text': 'Rumour has it, a Progressive Booster can be found in Sand Zone.', 'facepic': '0000', 'ending': '<END'}}}, 'Ring3': {'pickups': {}, 'music': {'0502': {'song_id': '0044', 'original_id': '0032'}, '0600': {'song_id': '0053', 'original_id': '0018'}}, 'entrances': {}, 'hints': {}}, 'MazeM': {'pickups': {}, 'music': {'0090': {'song_id': '0055', 'original_id': '0020'}, '0091': {'song_id': '0064', 'original_id': '0020'}, '0092': {'song_id': '0047', 'original_id': '0020'}, '0093': {'song_id': '0056', 'original_id': '0020'}, '0094': {'song_id': '0056', 'original_id': '0020'}, '0301': {'song_id': '0050', 'original_id': '0020'}}, 'entrances': {}, 'hints': {}}, 'Almond': {'pickups': {'1111': '<EVE0062', '0243': '<EVE0014'}, 'music': {'0090': {'song_id': 
        '0055', 'original_id': '0023'}, '0091': {'song_id': '0042', 'original_id': '0023'}, '0092': {'song_id': '0053', 'original_id': '0023'}, '0093': {'song_id': '0051', 'original_id': '0023'}, '0094': {'song_id': '0064', 'original_id': '0023'}, '0361': {'song_id': '0046', 'original_id': '0023'}, '0452': {'song_id': '0056', 'original_id': '0022'}, '0500': {'song_id': '0046', 'original_id': '0022'}}, 'entrances': {}, 'hints': {}}, 'Jail2': {'pickups': {}, 'music': {'0090': {'song_id': '0043', 'original_id': '0008'}, '0091': {'song_id': '0057', 'original_id': '0008'}, '0092': {'song_id': '0060', 'original_id': '0008'}, '0093': {'song_id': '0044', 'original_id': '0008'}, '0094': {'song_id': '0052', 'original_id': '0008'}, '0099': {'song_id': '0065', 'original_id': '0008'}}, 'entrances': {}, 'hints': {}}, 'Lounge': {'pickups': {}, 'music': {'0090': {'song_id': '0046', 'original_id': '0002'}, '0091': {'song_id': '0064', 'original_id': '0002'}, '0092': {'song_id': '0046', 'original_id': '0002'}, '0093': {'song_id': '0051', 'original_id': '0002'}, '0094': {'song_id': '0052', 'original_id': '0002'}}, 'entrances': {}, 'hints': {}}, 'Momo': {'pickups': {'0201': '<EVE0051'}, 'music': {'0090': {'song_id': '0056', 'original_id': '0002'}, '0091': {'song_id': '0059', 'original_id': '0002'}, '0092': {'song_id': '0055', 'original_id': '0002'}, '0093': {'song_id': '0050', 'original_id': '0002'}, '0094': {'song_id': '0064', 'original_id': '0002'}, '0280': {'song_id': '0064', 'original_id': '0002'}, '0281': {'song_id': '0065', 'original_id': '0002'}}, 'entrances': {}, 'hints': {}}, 'MazeA': {'pickups': {'0512': '<EVE0030', '0502': '<EVE0012', '0522': '<EVE0014'}, 'music': {'0090': {'song_id': '0062', 'original_id': '0008'}, '0091': {'song_id': '0042', 'original_id': '0008'}, '0092': {'song_id': '0044', 'original_id': '0008'}, '0093': {'song_id': '0053', 'original_id': '0008'}, '0094': {'song_id': '0054', 'original_id': '0008'}, '0301': {'song_id': '0043', 'original_id': '0008'}}, 'entrances': {}, 'hints': {}}, 'Dark': {'pickups': {'0401': '<EVE0068'}, 'music': {}, 'entrances': {}, 'hints': {}}, 'Cave': {'pickups': {'0401': '<EVE0033'}, 'music': {'0090': {'song_id': '0046', 'original_id': '0008'}, '0091': {'song_id': '0043', 'original_id': '0008'}, '0092': {'song_id': '0046', 'original_id': '0008'}, '0093': {'song_id': '0045', 'original_id': '0008'}, '0094': {'song_id': '0054', 'original_id': '0008'}}, 'entrances': {}, 'hints': {}}, 'Prefa2': {'pickups': {}, 'music': {}, 'entrances': {}, 'hints': {}}, 'Plant': {'pickups': {'0401': '<EVE0088'}, 'music': {'0090': {'song_id': '0045', 'original_id': '0013'}, '0091': {'song_id': '0044', 'original_id': '0013'}, '0092': {'song_id': '0064', 'original_id': '0013'}, '0093': {'song_id': '0052', 'original_id': '0013'}, '0094': {'song_id': '0046', 'original_id': '0013'}, '0095': {'song_id': '0052', 'original_id': '0028'}, '0096': {'song_id': '0062', 'original_id': '0028'}, '0097': {'song_id': '0057', 'original_id': '0028'}, '0098': {'song_id': '0046', 'original_id': '0028'}, '0099': {'song_id': '0064', 'original_id': '0028'}}, 'entrances': {}, 'hints': {}}, 'Ballo2': {'pickups': {}, 'music': {'0500': {'song_id': '0062', 'original_id': '0034'}}, 'entrances': {}, 'hints': {}}, 'MazeD': {'pickups': {'0201': '<EVE0060'}, 'music': {'0400': {'song_id': '0051', 'original_id': '0004'}}, 'entrances': {}, 'hints': {}}, 'EggX': {'pickups': {}, 'music': {'0095': {'song_id': '0050', 'original_id': '0014'}}, 'entrances': {}, 'hints': {}}, 'Cthu2': {'pickups': {}, 'music': {'0090': {'song_id': '0051', 'original_id': '0008'}, '0091': {'song_id': '0045', 'original_id': '0008'}, '0092': {'song_id': '0060', 'original_id': '0008'}, '0093': {'song_id': '0044', 'original_id': '0008'}, '0094': {'song_id': '0053', 'original_id': '0008'}}, 'entrances': {}, 'hints': {}}, 'Hell4': {'pickups': {}, 'music': {}, 'entrances': {}, 'hints': {}}, 'EggX2': {'pickups': {}, 'music': {'0090': {'song_id': '0051', 'original_id': '0014'}, '0091': {'song_id': '0047', 'original_id': '0014'}, '0092': {'song_id': '0046', 'original_id': '0014'}, '0093': {'song_id': '0055', 'original_id': '0014'}, '0095': {'song_id': '0053', 'original_id': '0014'}}, 'entrances': {}, 'hints': {}}, 'Pole': {'pickups': {'0202': '<EVE0012', '0303': '<EVE0014'}, 'music': {'0090': {'song_id': '0060', 'original_id': '0008'}, '0091': {'song_id': '0055', 'original_id': '0008'}, '0092': {'song_id': '0048', 'original_id': '0008'}, '0093': {'song_id': '0051', 'original_id': '0008'}, '0094': {'song_id': '0056', 'original_id': '0008'}, '0095': {'song_id': '0062', 'original_id': '0008'}}, 'entrances': {}, 'hints': {}}}, 'mychar': 'C:\\Users\\dunca\\Documents\\git\\randovania\\randovania\\games\\cave_story\\assets\\mychar\\Sue.bmp', 'hash': [25, 26, 4, 31, 1]
    }
    patch_files(data, Path("C:/Users/dunca/Documents/csrandotest"), lambda x, y: None)
