# Allen Institute Software License - This software license is the 2-clause BSD
# license plus a third clause that prohibits redistribution for commercial
# purposes without further permission.
#
# Copyright 2017. Allen Institute. All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# 1. Redistributions of source code must retain the above copyright notice,
# this list of conditions and the following disclaimer.
#
# 2. Redistributions in binary form must reproduce the above copyright notice,
# this list of conditions and the following disclaimer in the documentation
# and/or other materials provided with the distribution.
#
# 3. Redistributions for commercial purposes are not permitted without the
# Allen Institute's written permission.
# For purposes of this license, commercial purposes is the incorporation of the
# Allen Institute's software into anything for which you will charge fees or
# other compensation. Contact terms@alleninstitute.org for commercial licensing
# opportunities.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
# LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
# CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
# SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
# INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
# CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
# ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
# POSSIBILITY OF SUCH DAMAGE.
#
from .rma_template import RmaTemplate
from ..cache import cacheable

from allensdk.core.structure_tree import StructureTree


class OntologiesApi(RmaTemplate):
    '''
    See: `Atlas Drawings and Ontologies
    <http://help.brain-map.org/display/api/Atlas+Drawings+and+Ontologies>`_
    '''

    rma_templates = \
        {"ontology_queries": [
            {'name': 'structures_by_graph_ids',
             'description': 'see name',
             'model': 'Structure',
             'criteria': '[graph_id$in{{ graph_ids }}]',
             'order': ['structures.graph_order'],
             'num_rows': 'all',
             'count': False,
             'criteria_params': ['graph_ids']
             },
            {'name': 'structures_by_graph_names',
             'description': 'see name',
             'model': 'Structure',
             'criteria': 'graph[structure_graphs.name$in{{ graph_names }}]',
             'order': ['structures.graph_order'],
             'num_rows': 'all',
             'count': False,
             'criteria_params': ['graph_names']
             },
            {'name': 'structures_by_set_ids',
             'description': 'see name',
             'model': 'Structure',
             'criteria': '[structure_set_id$in{{ set_ids }}]',
             'order': ['structures.graph_order'],
             'num_rows': 'all',
             'count': False,
             'criteria_params': ['set_ids']
             },
            {'name': 'structures_by_set_names',
             'description': 'see name',
             'model': 'Structure',
             'criteria': 'structure_sets[name$in{{ set_names }}]',
             'order': ['structures.graph_order'],
             'num_rows': 'all',
             'count': False,
             'criteria_params': ['set_names']
             },
            {'name': 'structure_graphs_list',
             'description': 'see name',
             'model': 'StructureGraph',
             'num_rows': 'all',
             'count': False
             },
            {'name': 'structure_sets_list',
             'description': 'see name',
             'model': 'StructureSet',
             'num_rows': 'all',
             'count': False
             },
            {'name': 'atlases_list',
             'description': 'see name',
             'model': 'Atlas',
             'num_rows': 'all',
             'count': False
             },
            {'name': 'atlases_table',
             'description': 'see name',
             'model': 'Atlas',
             'criteria': '{% if atlas_ids is defined %}[id$in{{ atlas_ids }}],{%endif%}structure_graph(ontology),graphic_group_labels',
             'include': 'structure_graph(ontology),graphic_group_labels',
             'only': ['atlases.id',
                      'atlases.name',
                      'atlases.image_type',
                      'ontologies.id',
                      'ontologies.name',
                      'structure_graphs.id',
                      'structure_graphs.name',
                      'graphic_group_labels.id',
                      'graphic_group_labels.name'],
             'num_rows': 'all',
             'count': False,
             'criteria_params': ['atlas_ids']
             }, 
             {'name': 'structures_with_sets', 
             'description': 'see name',
             'model': 'Structure',
             'include': 'structure_sets', 
             'criteria': '[graph_id$in{{ graph_ids }}]',
             'order': ['structures.graph_order'], 
             'num_rows': 'all', 
             'count': False, 
             'criteria_params': ['graph_ids']
             },
             {'name': 'structure_sets_by_id', 
              'description': 'see name', 
              'model': 'StructureSet',
              'criteria': '[id$in{{ set_ids }}]', 
              'num_rows': 'all',
              'count': False, 
              'criteria_params': ['set_ids']
             }
        ]}

    def __init__(self, base_uri=None):
        super(OntologiesApi, self).__init__(base_uri,
                                            query_manifest=OntologiesApi.rma_templates)

    @cacheable()
    def get_structures(self,
                       structure_graph_ids=None,
                       structure_graph_names=None,
                       structure_set_ids=None,
                       structure_set_names=None,
                       order=['structures.graph_order'],
                       num_rows='all',
                       count=False,
                       **kwargs):
        '''Retrieve data about anatomical structures.

        Parameters
        ----------
        structure_graph_ids : int or list of ints, optional
            database keys to get all structures in particular graphs
        structure_graph_names : string or list of strings, optional
            list of graph names to narrow the query
        structure_set_ids : int or list of ints, optional
            database keys to get all structures in a particular set
        structure_set_names : string or list of strings, optional
            list of set names to narrow the query.
        order : list of strings
            list of RMA order clauses for sorting
        num_rows : int
            how many records to retrieve

        Returns
        -------
        dict
            the parsed json response containing data from the API

        Notes
        -----
        Only one of the methods of limiting the query should be used at a time.
        '''
        if structure_graph_ids is not None:
            data = self.template_query('ontology_queries',
                                       'structures_by_graph_ids',
                                       graph_ids=structure_graph_ids,
                                       order=order,
                                       num_rows=num_rows,
                                       count=count)
        elif structure_graph_names is not None:
            data = self.template_query('ontology_queries',
                                       'structures_by_graph_names',
                                       graph_names=structure_graph_names,
                                       order=order,
                                       num_rows=num_rows,
                                       count=count)
        elif structure_set_ids is not None:
            data = self.template_query('ontology_queries',
                                       'structures_by_set_ids',
                                       set_ids=structure_set_ids,
                                       order=order,
                                       num_rows=num_rows,
                                       count=count)
        elif structure_set_names is not None:
            data = self.template_query('ontology_queries',
                                       'structures_by_set_names',
                                       set_names=structure_set_names,
                                       order=order,
                                       num_rows=num_rows,
                                       count=count)

        return data 
        
        
    @cacheable()
    def get_structures_with_sets(self, structure_graph_ids, order=['structures.graph_order'], 
                                 num_rows='all', count=False, **kwargs):
        '''Download structures along with the sets to which they belong.

        Parameters
        ----------
        structure_graph_ids : int or list of int
            Only fetch structure records from these graphs.
        order : list of strings
            list of RMA order clauses for sorting
        num_rows : int
            how many records to retrieve
 
        Returns
        -------
        dict
            the parsed json response containing data from the API

        '''
    
        return self.template_query('ontology_queries', 'structures_with_sets', 
                                   graph_ids=structure_graph_ids, 
                                   order=order, num_rows=num_rows, 
                                   count=count)
    

    def unpack_structure_set_ancestors(self, structure_dataframe):
        '''Convert a slash-separated structure_id_path field to a list.

        Parameters
        ----------
        structure_dataframe : DataFrame
            structure data from the API

        Returns
        -------
        None
            A new column is added to the dataframe containing the ancestor list.
        '''
        ancestors = structure_dataframe['structure_id_path'].apply(
            lambda e: [int(a) for a in e.split('/')[1:-1]])
        structure_ancestors = [
            [n for n in ancestors_n] for ancestors_n in ancestors
        ]
        structure_dataframe['structure_set_ancestor'] = structure_ancestors

    @cacheable()
    def get_atlases_table(self, atlas_ids=None, brief=True):
        '''List Atlases available through the API
        with associated ontologies and structure graphs.

        Parameters
        ----------
        atlas_ids : integer or list of integers, optional
            only select specific atlases
        brief : boolean, optional
            True (default) requests only name and id fields.

        Returns
        -------
        dict : atlas metadata

        Notes
        -----
        This query is based on the
        `table of available Atlases <http://help.brain-map.org/display/api/Atlas+Drawings+and+Ontologies>`_.
        See also: `Class: Atlas <http://api.brain-map.org/doc/Atlas.html>`_
        '''
        if brief is True:
            data = self.template_query('ontology_queries',
                                       'atlases_table',
                                       atlas_ids=atlas_ids)
        else:
            data = self.template_query('ontology_queries',
                                       'atlases_table',
                                       atlas_ids=atlas_ids,
                                       only=None)

        return data

    @cacheable()
    def get_atlases(self):
        return self.template_query('ontology_queries',
                                   'atlases_list')

    @cacheable()
    def get_structure_graphs(self):
        return self.template_query('ontology_queries',
                                   'structure_graphs_list')

    @cacheable()
    def get_structure_sets(self, structure_set_ids=None):

        if structure_set_ids is None:
            return self.template_query('ontology_queries',
                                       'structure_sets_list')
        else:
            return self.template_query('ontology_queries', 
                                       'structure_sets_by_id', 
                                       set_ids=list(structure_set_ids))
