# Allen Institute Software License - This software license is the 2-clause BSD
# license plus a third clause that prohibits redistribution for commercial
# purposes without further permission.
#
# Copyright 2017. Allen Institute. All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# 1. Redistributions of source code must retain the above copyright notice,
# this list of conditions and the following disclaimer.
#
# 2. Redistributions in binary form must reproduce the above copyright notice,
# this list of conditions and the following disclaimer in the documentation
# and/or other materials provided with the distribution.
#
# 3. Redistributions for commercial purposes are not permitted without the
# Allen Institute's written permission.
# For purposes of this license, commercial purposes is the incorporation of the
# Allen Institute's software into anything for which you will charge fees or
# other compensation. Contact terms@alleninstitute.org for commercial licensing
# opportunities.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
# LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
# CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
# SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
# INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
# CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
# ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
# POSSIBILITY OF SUCH DAMAGE.
#
from .fitgaussian2D import fitgaussian2D, GaussianFitError, gaussian2D
import numpy as np
import pandas as pd
import collections
import sys
import warnings

def add_to_fit_parameters_dict_single(fit_parameters_dict, p):

    fit_parameters_dict['height'].append(p[0])
    fit_parameters_dict['center_y'].append(p[1])
    fit_parameters_dict['center_x'].append(p[2])
    fit_parameters_dict['width_y'].append(p[3])
    fit_parameters_dict['width_x'].append(p[4])
    fit_parameters_dict['rotation'].append(p[5])
    if (p[3] is None) or (p[4] is None):
        fit_parameters_dict['area'].append(None)
    else:
        fit_parameters_dict['area'].append(np.pi * (3./2) ** 2 * np.abs(p[3]) * np.abs(p[4]))

def get_gaussian_fit_single_channel(rf, fit_parameters_dict):

    try:
        p_fit = fitgaussian2D(rf)
        add_to_fit_parameters_dict_single(fit_parameters_dict, p_fit)
        data_fitted_on = gaussian2D(*p_fit)(*np.indices(rf.shape))
        fit_parameters_dict['data'].append(data_fitted_on)
    except GaussianFitError:
        warnings.warn('GaussianFitError (on subfield) caught')
        add_to_fit_parameters_dict_single(fit_parameters_dict, [None]*6)
        fit_parameters_dict['data'].append(np.zeros_like(rf))

def compute_distance(center_on, center_off):

    center_x_on, center_y_on = center_on
    center_x_off, center_y_off = center_off

    if (center_x_on is None) or (center_y_on is None) or (center_x_off is None) or (center_y_off is None):
        return None
    else:
        return np.sqrt((center_x_off-center_x_on)**2+(center_y_off-center_y_on)**2)

def compute_overlap(data_fitted_on, data_fitted_off):

    on_bin = np.where(data_fitted_on > 0.001, 1, 0)
    off_bin = np.where(data_fitted_off > 0.001, 1, 0)

    return float((np.multiply(on_bin, off_bin)).sum()) / (np.sqrt(on_bin.sum()) * np.sqrt(off_bin.sum()))



