# Allen Institute Software License - This software license is the 2-clause BSD
# license plus a third clause that prohibits redistribution for commercial
# purposes without further permission.
#
# Copyright 2017. Allen Institute. All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# 1. Redistributions of source code must retain the above copyright notice,
# this list of conditions and the following disclaimer.
#
# 2. Redistributions in binary form must reproduce the above copyright notice,
# this list of conditions and the following disclaimer in the documentation
# and/or other materials provided with the distribution.
#
# 3. Redistributions for commercial purposes are not permitted without the
# Allen Institute's written permission.
# For purposes of this license, commercial purposes is the incorporation of the
# Allen Institute's software into anything for which you will charge fees or
# other compensation. Contact terms@alleninstitute.org for commercial licensing
# opportunities.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
# LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
# CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
# SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
# INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
# CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
# ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
# POSSIBILITY OF SUCH DAMAGE.
#
import allensdk.core.json_utilities as ju
import logging
from allensdk.config.manifest import Manifest
import pandas as pd
import six


class ManifestBuilder(object):
    df_columns = ['key', 'parent_key', 'spec', 'type', 'format']

    def __init__(self):
        self._log = logging.getLogger(__name__)
        self.path_info = []
        self.sections = {}

    def set_version(self, value):
        self.path_info.append({'type': Manifest.VERSION, 'value': value})

    def add_path(self, key, spec,
                 typename='dir',
                 parent_key=None,
                 format=None):
        entry = {
            'key': key,
            'type': typename,
            'spec': spec}

        if format is not None:
            entry['format'] = format

        if parent_key is not None:
            entry['parent_key'] = parent_key

        self.path_info.append(entry)

    def add_section(self, name, contents):
        self.sections[name] = contents

    def write_json_file(self, path, overwrite=False):
        mode = 'wb'

        if overwrite is True:
            mode = 'wb+'

        json_string = self.write_json_string()

        with open(path, mode) as f:
            try:
                f.write(json_string)   # Python 2.7
            except TypeError:
                f.write(bytes(json_string, 'utf-8'))  # Python 3

    def get_config(self):
        wrapper = {"manifest": self.path_info}
        for section in self.sections.values():
            wrapper.update(section)

        return wrapper

    def get_manifest(self):
        return Manifest(self.path_info)

    def write_json_string(self):
        config = self.get_config()
        return ju.write_string(config)

    def as_dataframe(self):
        return pd.DataFrame(self.path_info,
                            columns=ManifestBuilder.df_columns)

    def from_dataframe(self, df):
        self.path_info = {}

        for _, k, p, s, t, f in six.iteritems(df.loc[:, ManifestBuilder.df_columns]):
            self.add_path(k, s, typename=t, parent=p, format=f)
