# Allen Institute Software License - This software license is the 2-clause BSD
# license plus a third clause that prohibits redistribution for commercial
# purposes without further permission.
#
# Copyright 2014-2016. Allen Institute. All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# 1. Redistributions of source code must retain the above copyright notice,
# this list of conditions and the following disclaimer.
#
# 2. Redistributions in binary form must reproduce the above copyright notice,
# this list of conditions and the following disclaimer in the documentation
# and/or other materials provided with the distribution.
#
# 3. Redistributions for commercial purposes are not permitted without the
# Allen Institute's written permission.
# For purposes of this license, commercial purposes is the incorporation of the
# Allen Institute's software into anything for which you will charge fees or
# other compensation. Contact terms@alleninstitute.org for commercial licensing
# opportunities.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
# LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
# CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
# SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
# INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
# CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
# ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
# POSSIBILITY OF SUCH DAMAGE.
#
import logging
from allensdk.config.model.description import Description


class DescriptionParser(object):
    log = logging.getLogger(__name__)

    def __init__(self):
        pass

    def read(self, file_path, description=None, section=None, **kwargs):
        '''Parse data needed for a simulation.

        Parameters
        ----------
        description : dict
            Configuration from parsing previous files.
        section : string, optional
            What configuration section to read it into if the file does not specify.
        '''
        if description is None:
            description = Description()

        self.reader = self.parser_for_extension(file_path)
        self.reader.read(file_path, description, section, **kwargs)

        return description

    def read_string(self, data_string, description=None, section=None, header=None):
        '''Parse data needed for a simulation from a string.'''
        raise Exception("Not implemented, use a sub class")

    def write(self, filename, description):
        """Save the configuration.

        Parameters
        ----------
        filename : string
            Name of the file to write.
        """
        writer = self.parser_for_extension(filename)

        writer.write(filename, description)

    def parser_for_extension(self, filename):
        '''Choose a subclass that can read the format.

        Parameters
        ----------
        filename : string
           For the extension.

        Returns
        -------
        DescriptionParser
            Appropriate subclass.
        '''
        # Circular imports
        from allensdk.config.model.formats.json_description_parser import JsonDescriptionParser
        from allensdk.config.model.formats.pycfg_description_parser import PycfgDescriptionParser

        parser = None

        if filename.endswith('.json'):
            parser = JsonDescriptionParser()
        elif filename.endswith('.pycfg'):
            parser = PycfgDescriptionParser()
        else:
            raise Exception('could not determine file format')

        return parser
