# Allen Institute Software License - This software license is the 2-clause BSD
# license plus a third clause that prohibits redistribution for commercial
# purposes without further permission.
#
# Copyright 2017. Allen Institute. All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# 1. Redistributions of source code must retain the above copyright notice,
# this list of conditions and the following disclaimer.
#
# 2. Redistributions in binary form must reproduce the above copyright notice,
# this list of conditions and the following disclaimer in the documentation
# and/or other materials provided with the distribution.
#
# 3. Redistributions for commercial purposes are not permitted without the
# Allen Institute's written permission.
# For purposes of this license, commercial purposes is the incorporation of the
# Allen Institute's software into anything for which you will charge fees or
# other compensation. Contact terms@alleninstitute.org for commercial licensing
# opportunities.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
# LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
# CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
# SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
# INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
# CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
# ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
# POSSIBILITY OF SUCH DAMAGE.
#
import logging
from json import dump, dumps
from allensdk.config.model.description_parser import DescriptionParser
from allensdk.config.model.description import Description
from allensdk.core.json_utilities import JsonComments


class JsonDescriptionParser(DescriptionParser):
    log = logging.getLogger(__name__)

    def __init__(self):
        super(JsonDescriptionParser, self).__init__()

    def read(self, file_path, description=None, section=None, **kwargs):
        '''Parse a complete or partial configuration.

        Parameters
        ----------
        json_string : string
            Input to parse.
        description : Description, optional
            Where to put the parsed configuration.  If None a new one is created.
        section : string, optional
            Where to put the parsed configuration within the description.

        Returns
        -------
        Description
            The input description with parsed configuration added.

        Section is only specified for "bare" objects that are to be added to a section array.
        '''
        if description is None:
            description = Description()

        data = JsonComments.read_file(file_path)
        description.unpack(data, section)

        return description

    def read_string(self, json_string, description=None, section=None, **kwargs):
        '''Parse a complete or partial configuration.

        Parameters
        ----------
        json_string : string
            Input to parse.
        description : Description, optional
            Where to put the parsed configuration.  If None a new one is created.
        section : string, optional
            Where to put the parsed configuration within the description.

        Returns
        -------
        Description
            The input description with parsed configuration added.

        Section is only specified for "bare" objects that are to be added to a section array.
        '''
        if description is None:
            description = Description()

        data = JsonComments.read_string(json_string)

        description.unpack(data, section)

        return description

    def write(self, filename, description):
        '''Write the description to a JSON file.

        Parameters
        ----------
        description : Description
            Object to write.
        '''
        try:
            with open(filename, 'w') as f:
                dump(description.data, f, indent=2)

        except Exception:
            self.log.warn(
                "Couldn't write allensdk json description: %s" % filename)
            raise

        return

    def write_string(self, description):
        '''Write the description to a JSON string.

        Parameters
        ----------
        description : Description
            Object to write.

        Returns
        -------
        string
           JSON serialization of the input.
        '''
        try:
            json_string = dumps(description.data,
                                indent=2)
            return json_string
        except Exception:
            self.log.warn("Couldn't write allensdk json description: %s")
            raise
