# Allen Institute Software License - This software license is the 2-clause BSD
# license plus a third clause that prohibits redistribution for commercial
# purposes without further permission.
#
# Copyright 2017. Allen Institute. All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# 1. Redistributions of source code must retain the above copyright notice,
# this list of conditions and the following disclaimer.
#
# 2. Redistributions in binary form must reproduce the above copyright notice,
# this list of conditions and the following disclaimer in the documentation
# and/or other materials provided with the distribution.
#
# 3. Redistributions for commercial purposes are not permitted without the
# Allen Institute's written permission.
# For purposes of this license, commercial purposes is the incorporation of the
# Allen Institute's software into anything for which you will charge fees or
# other compensation. Contact terms@alleninstitute.org for commercial licensing
# opportunities.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
# LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
# CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
# SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
# INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
# CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
# ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
# POSSIBILITY OF SUCH DAMAGE.
#
import logging
from pprint import pprint, pformat
from allensdk.config.model.description import Description
from allensdk.config.model.description_parser import DescriptionParser


class PycfgDescriptionParser(DescriptionParser):
    log = logging.getLogger(__name__)

    def __init__(self):
        super(PycfgDescriptionParser, self).__init__()

    def read(self, pycfg_file_path, description=None, section=None, **kwargs):
        '''Read a serialized description from a Python (.pycfg) file.

        Parameters
        ----------
        filename : string
            Name of the .pycfg file.

        Returns
        -------
        Description
            Configuration object.
        '''
        header = kwargs.get('prefix', '')

        with open(pycfg_file_path, 'r') as f:
            return self.read_string(f.read(), description, section, header=header)

    def read_string(self, python_string, description=None, section=None, **kwargs):
        '''Read a serialized description from a Python (.pycfg) string.

        Parameters
        ----------
        python_string : string
            Python string with a serialized description.

        Returns
        -------
        Description
            Configuration object.
        '''

        if description is None:
            description = Description()

        header = kwargs.get('header', '')

        python_string = "%s\n\nallensdk_description = %s" % (
            header, python_string)

        ns = {}
        code = compile(python_string, 'string', 'exec')
        exec(code, ns)
        data = ns['allensdk_description']
        description.unpack(data, section)

        return description

    def write(self, filename, description):
        '''Write the description to a Python (.pycfg) file.

        Parameters
        ----------
        filename : string
            Name of the file to write.
        '''
        try:
            with open(filename, 'w') as f:
                pprint(description.data, f, indent=2)

        except Exception:
            self.log.warn(
                "Couldn't write allensdk python description: %s" % filename)
            raise

        return

    def write_string(self, description):
        '''Write the description to a pretty-printed Python string.

        Parameters
        ----------
        description : Description
            Configuration object to write.
        '''
        pycfg_string = pformat(description.data, indent=2)

        return pycfg_string
