# Allen Institute Software License - This software license is the 2-clause BSD
# license plus a third clause that prohibits redistribution for commercial
# purposes without further permission.
#
# Copyright 2015-2017. Allen Institute. All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# 1. Redistributions of source code must retain the above copyright notice,
# this list of conditions and the following disclaimer.
#
# 2. Redistributions in binary form must reproduce the above copyright notice,
# this list of conditions and the following disclaimer in the documentation
# and/or other materials provided with the distribution.
#
# 3. Redistributions for commercial purposes are not permitted without the
# Allen Institute's written permission.
# For purposes of this license, commercial purposes is the incorporation of the
# Allen Institute's software into anything for which you will charge fees or
# other compensation. Contact terms@alleninstitute.org for commercial licensing
# opportunities.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
# LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
# CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
# SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
# INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
# CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
# ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
# POSSIBILITY OF SUCH DAMAGE.
#


import numpy as np


def read_obj(path):
    with open(path, 'r') as obj_file:
        lines = obj_file.read().split('\n')
        output = parse_obj(lines)
    return output


def parse_obj(lines):
    '''Parse a wavefront obj file into a triplet of vertices, normals, and faces. 
    This parser is specific to obj files generated from our annotation volumes

    Parameters
    ----------
    lines : list of str
        Lines of input obj file

    Returns
    -------
    vertices : np.ndarray
        Dimensions are (nSamples, nCoordinates=3). Locations in the reference space
        of vertices
    vertex_normals : np.ndarray
        Dimensions are (nSample, nElements=3). Vectors normal to vertices.
    face_vertices : np.ndarray
        Dimensions are (sample, nVertices=3). References are given in indices 
        (0-indexed here, but 1-indexed in the file) of vertices that make up each face.
    face_normals : np.ndarray
        Dimensions are (sample, nNormals=3). References are given in indices 
        (0-indexed here, but 1-indexed in the file) of vertex normals that make up each face.

    Notes
    -----
    This parser is specialized to the obj files that the Allen Institute for Brain Science 
    generates from our own structure annotations.
    '''

    vertices = []
    vertex_normals = []
    face_vertices = []
    face_normals = []

    for line in lines:
        
        if line[:2] == 'v ':
            vertices.append( line.split()[1:] )

        elif line[:3] == 'vn ':
            vertex_normals.append( line.split()[1:] )

        elif line[:2] == 'f ':
            line = line.replace('//', ' ').split()[1:]

            face_vertices.append( line[::2] )
            face_normals.append( line[1::2] )
            
    vertices = np.array(vertices).astype(float)
    vertex_normals = np.array(vertex_normals).astype(float)
    face_vertices = np.array(face_vertices).astype(int) - 1
    face_normals = np.array(face_normals).astype(int) - 1

    return vertices, vertex_normals, face_vertices, face_normals
