# Allen Institute Software License - This software license is the 2-clause BSD
# license plus a third clause that prohibits redistribution for commercial
# purposes without further permission.
#
# Copyright 2017. Allen Institute. All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# 1. Redistributions of source code must retain the above copyright notice,
# this list of conditions and the following disclaimer.
#
# 2. Redistributions in binary form must reproduce the above copyright notice,
# this list of conditions and the following disclaimer in the documentation
# and/or other materials provided with the distribution.
#
# 3. Redistributions for commercial purposes are not permitted without the
# Allen Institute's written permission.
# For purposes of this license, commercial purposes is the incorporation of the
# Allen Institute's software into anything for which you will charge fees or
# other compensation. Contact terms@alleninstitute.org for commercial licensing
# opportunities.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
# LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
# CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
# SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
# INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
# CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
# ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
# POSSIBILITY OF SUCH DAMAGE.
#
import re
import logging
from pkg_resources import resource_filename  # @UnresolvedImport
from allensdk.config.app.application_config import ApplicationConfig
from allensdk.config.model.description_parser import DescriptionParser
from allensdk.config.model.description import Description


class Config(ApplicationConfig):
    _log = logging.getLogger(__name__)

    _DEFAULT_LOG_CONFIG = resource_filename(__name__, 'logging.conf')

    #: A structure that defines the available configuration parameters.
    #: The default value and help strings may be seen by viewing the source.
    _DEFAULTS = {
        'workdir': {'default': 'workdir',
                    'help': 'writable directory where intermediate and output files are written.'},
        'data_dir': {'default': '',
                     'help': 'writable directory where intermediate and output files are written.'},
        'model_file': {'default': 'config.json',
                       'help': 'file where the model parameters are set.'},
        'main': {'default': 'simulation#run',
                 'help': 'module#function that runs the actual simulation'}
    }

    def __init__(self):
        super(Config, self).__init__(Config._DEFAULTS,
                                     name='biophys',
                                     halp='tools for biophysically detailed modeling at the Allen Institute.',
                                     default_log_config=Config._DEFAULT_LOG_CONFIG)

    def load(self, config_path,
             disable_existing_logs=False):
        '''Parse the application configuration then immediately load
        the model configuration files.

        Parameters
        ----------
        disable_existing_logs : boolean, optional
            If false (default) leave existing logs after configuration.
        '''
        super(Config, self).load([config_path], disable_existing_logs)
        description = self.read_model_description()

        return description

    def read_model_description(self):
        '''parse the model_file field of the application configuration
        and read the files.

        The model_file field of the application configuration is
        first split at commas, since it may list more than one file.

        The files may be uris of the form :samp:`file:filename?section=name`,
        in which case a bare configuration object is read from filename
        into the configuration section with key 'name'.

        A simple filename without a section option
        is treated as a standard multi-section configuration file.

        Returns
        -------
        description : Description
            Configuration object.
        '''
        reader = DescriptionParser()
        description = Description()

        Config._log.info("model file: %s" % self.model_file)

        # TODO: make space aware w/ regex
        for model_file in self.model_file.split(','):
            if not model_file.startswith("file:"):
                model_file = 'file:' + model_file

            file_regex = re.compile(r"^file:([^?]*)(\?(.*)?)?")
            m = file_regex.match(model_file)
            model_file = m.group(1)
            file_url_params = {}
            if m.group(3):
                file_url_params.update(((x[0], x[1])
                                        for x in (y.split('=')
                                                  for y in m.group(3).split('&'))))
            if 'section' in file_url_params:
                section = file_url_params['section']
            else:
                section = None
            Config._log.info("reading model file %s" % (model_file))
            reader.read(model_file, description, section)

        return description
