# Allen Institute Software License - This software license is the 2-clause BSD
# license plus a third clause that prohibits redistribution for commercial
# purposes without further permission.
#
# Copyright 2015-2016. Allen Institute. All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# 1. Redistributions of source code must retain the above copyright notice,
# this list of conditions and the following disclaimer.
#
# 2. Redistributions in binary form must reproduce the above copyright notice,
# this list of conditions and the following disclaimer in the documentation
# and/or other materials provided with the distribution.
#
# 3. Redistributions for commercial purposes are not permitted without the
# Allen Institute's written permission.
# For purposes of this license, commercial purposes is the incorporation of the
# Allen Institute's software into anything for which you will charge fees or
# other compensation. Contact terms@alleninstitute.org for commercial licensing
# opportunities.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
# LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
# CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
# SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
# INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
# CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
# ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
# POSSIBILITY OF SUCH DAMAGE.
#
import logging


class HocUtils(object):
    '''A helper class for containing references to NEUORN.

    Attributes
    ----------
    h : object
        The NEURON hoc object.
    nrn : object
        The NEURON python object.
    neuron : module
        The NEURON module.
    '''
    _log = logging.getLogger(__name__)
    h = None
    nrn = None
    neuron = None

    def __init__(self, description):
        import neuron
        import nrn

        self.h = neuron.h
        HocUtils.neuron = neuron
        HocUtils.nrn = nrn
        HocUtils.h = self.h

        self.description = description
        self.manifest = description.manifest

        self.hoc_files = description.data['neuron'][0]['hoc']

        self.initialize_hoc()

    def initialize_hoc(self):
        '''Basic setup for NEURON.'''
        h = self.h
        params = self.description.data['conditions'][0]

        for hoc_file in self.hoc_files:
            HocUtils._log.info("loading hoc file %s" % (hoc_file))
            HocUtils.h.load_file(str(hoc_file))

        h('starttime = startsw()')

        if 'celsius' in params:
            h.celsius = params['celsius']

        if 'v_init' in params:
            h.v_init = params['v_init']

        if 'dt' in params:
            h.dt = params['dt']
            h.steps_per_ms = 1.0 / h.dt

        if 'tstop' in params:
            h.tstop = params['tstop']
            h.runStopAt = h.tstop
