"""Multithreaded and parallel Solidity event reading helpers."""
from typing import Any, Optional, List, Iterable, Counter

from requests.adapters import HTTPAdapter
from web3.contract.contract import ContractEvent

from eth_defi.event_reader.filter import Filter
from eth_defi.event_reader.logresult import LogResult
from eth_defi.event_reader.reader import Web3EventReader, read_events_concurrent, ReaderConnection, ProgressUpdate
from eth_defi.event_reader.reorganisation_monitor import ReorganisationMonitor
from eth_defi.event_reader.web3factory import TunedWeb3Factory
from eth_defi.event_reader.web3worker import create_thread_pool_executor


class MultithreadEventReader(Web3EventReader):
    """Multithreaded and parallel Solidity event reading.

    - A high performance event reader for JSON-RPC APIs

    - Uses parallel requests, but the consumer will always get the events in the order
      they have happened on-chain

    - This class wraps around lower level event reading functions to
      simpler to use and well-documented example.

    - It is designed to be used standalone (you read the events)

    - It is designed to be passed to the functions that expect :py:class:`Web3EventReader` protocol

    See :ref:`multithread-reader` for full tutorial.

    Example how to read events:

    .. code-block:: python

        # Get one of the contracts prepackaged ABIs from eth_defi package
        value_interpreter_contract = get_contract(web3, "enzyme/ValueInterpreter.json")

        # Read events only for this contract
        # See https://docs.enzyme.finance/developers/contracts/polygon
        target_contract_address = "0x66De7e286Aae66f7f3Daf693c22d16EEa48a0f45"

        # Create eth_getLogs event filtering
        filter = Filter.create_filter(
            target_contract_address,
            [value_interpreter_contract.events.PrimitiveAdded],
        )

        # Set up multithreaded Polygon event reader.
        # Print progress to the console how many blocks there are left to read.
        reader = MultithreadEventReader(
            json_rpc_url,
            max_threads=16,
            notify=PrintProgressUpdate(),
            max_blocks_once=10_000)

        # Loop over the events as the multihreaded reader pool is feeding them to us.
        # Events will always arrive in the order they happened on chain.
        decoded_events = []
        start = datetime.datetime.utcnow()
        for event in reader(
            web3,
            start_block,
            end_block,
            filter=filter,
        ):
            # Decode the solidity event
            #
            # Indexed event parameters go to EVM topics, the second element is the first parameter
            # Non-indexed event parameters go to EVM arguments, first element is the first parameter
            arguments = decode_data(event["data"])
            topics = event["topics"]

            # event PrimitiveAdded(
            #     address indexed primitive,
            #     address aggregator,
            #     RateAsset rateAsset,
            #     uint256 unit
            # );
            primitive = convert_uint256_bytes_to_address(HexBytes(topics[1]))
            aggregator = convert_uint256_bytes_to_address(arguments[0])
            rate_asset = convert_int256_bytes_to_int(arguments[1])
            unit = convert_int256_bytes_to_int(arguments[2])

            # Primitive is a ERC-20 token, resolve its name and symbol while we are decoded the events
            token = fetch_erc20_details(web3, primitive)

            decoded = {
                "primitive": primitive,
                "aggregator": aggregator,
                "rate_asset": rate_asset,
                "unit": unit,
                "token": token,
            }

    Example how to pass the multithread reader to a function consuming events:

    .. code-block:: python

        provider = cast(HTTPProvider, web3.provider)
        json_rpc_url = provider.endpoint_uri
        reader = MultithreadEventReader(json_rpc_url, max_threads=16)

        start_block = 1
        end_block = web3.eth.block_number

        # Iterate over all price feed added events
        # over the whole blockchain range
        feed_iter = fetch_price_feeds(
            deployment,
            start_block,
            end_block,
            reader,
        )
        feeds = list(feed_iter)
        reader.close()
        assert len(feeds) == 2

    """

    def __init__(
        self,
        json_rpc_url: str,
        max_threads=10,
        reader_context: Any = None,
        api_counter=True,
        max_blocks_once=50_000,
        reorg_mon: Optional[ReorganisationMonitor] = None,
        notify: Optional[ProgressUpdate] = None,
    ):
        """Creates a multithreaded JSON-RPC reader pool.

        Can be passed to any function that expects :py:class:`Web3EventReader` protocol.

        - Sets up `requests` session pool for `max_threads` threads

        - Sets up `futureproof` thread pool executors for `max_threads` threads

        :param json_rpc_url:
            Your node URL

        :param max_threads:
            How many threads to allocate

        :param reader_context:
            Passed to the reader callback

        :param api_counter:
            Enable cross-thread API counter

        :param notify:
            Notify interface for progress reports.

        :param max_blocks_once:
            How many blocks your node's eth_getLogs call can serve.

            Crappy node providers set this value very low, around 1000,
            slowing down the reading.

        :param reorg_mon:
           Chain reorganisation monitor.

           The policy class for dealing with chain tip changes during the read or between event reads.

        """
        self.http_adapter = HTTPAdapter(pool_connections=max_threads, pool_maxsize=max_threads)
        self.web3_factory = TunedWeb3Factory(json_rpc_url, self.http_adapter, thread_local_cache=True, api_counter=api_counter)
        self.executor = create_thread_pool_executor(self.web3_factory, reader_context, max_workers=max_threads)
        self.max_blocks_once = max_blocks_once
        self.reader_context = reader_context
        self.reorg_mon = reorg_mon
        self.notify = notify

        # Set up the timestamp reading method

    def close(self):
        """Release the allocated resources."""
        self.executor.join()
        self.http_adapter.close()

    def __call__(
        self,
        web3: ReaderConnection,
        start_block: int,
        end_block: int,
        events: Optional[List[ContractEvent]] = None,
        filter: Optional[Filter] = None,
    ) -> Iterable[LogResult]:
        """Wrap the underlying low-level function.

        Wrap :py:func:`eth_defi.reader.reader.read_events_concurrent` using worker pools we have set up.

        .. note ::

            Currently timestamp reading not supported

        :return:
            Iterator for the events in the order they were written in the chain

        """
        yield from read_events_concurrent(
            self.executor,
            start_block,
            end_block,
            events=events,
            filter=filter,
            reorg_mon=self.reorg_mon,
            notify=self.notify,
            extract_timestamps=None,
        )

    def get_total_api_call_counts(self) -> Counter:
        """Sum API call counts across all threads.

        See :py:func:`eth_defi.chain.install_api_call_counter_middleware` for details.
        """
        return self.web3_factory.get_total_api_call_counts()
