# baggianalysis

Baggianalysis is a library aimed at simplifying the analysis of particle-based simulations. It makes it easy to parse, convert and analyse trajectories generated by simulation codes in an agnostic way. It is written in C++ and provides Python bindings. It is modular and can be extended from C++ and Python.

The documentation for the Python bindings can be found [here](https://lorenzo-rovigatti.github.io/baggianalysis/).

### Install the `Python` package

If all the requirements are met, the `baggianalysis` Python package can be compiled & installed by cloning the repo and using `pip` as follows:

```bash
$ git clone https://github.com/lorenzo-rovigatti/baggianalysis.git
$ PIP_COMMAND install ./baggianalysis
```

where `PIP_COMMAND` should be `pip`, `pip3`, `python -m pip` or `python3 -m pip`, depending on your local configuration. Use `-v` for verbose output.

### Requirements

* CMake >= 3.1 (>= 3.12 if installed through `pip`)
* make
* A c++14-compliant compiler (tested with GCC >= 5.4)
* If Python bindings are enabled, the `pip`, `setuptools` and `setuptools-scm` packages, as well as Python 3's header file are required (`python3-dev`)
* The pore size observable requires the [NLopt](https://nlopt.readthedocs.io/en/latest/) library (`libnlopt-dev`)
* The bond-order parameter observable requires the [GNU Scientific Library](https://www.gnu.org/software/gsl/) and [Boost's math](https://github.com/boostorg/math) libraries (`libgsl-dev` and `libboost-math-dev`)
* [Sphinx](https://www.sphinx-doc.org/en/master), [sphinx_rtd_theme](https://github.com/readthedocs/sphinx_rtd_theme) and [recommonmark](https://recommonmark.readthedocs.io/en/latest/) are required to generate the Python bindings' documentation. Those can all be installed by using `pip` (for instance with the command `pip3 install --user sphinx sphinx_rtd_theme recommonmark`)

### Compilation

The code can be compiled with the following commands:

	cd baggianalysis   # enter the baggianalysis folder
	mkdir build        # create a new build folder. It is good practice to compile out-of-source
	cd build
	cmake ..           # here you can specify additional options, see next section
	make -j4           # compile baggianalysis. The -jX make option makes it compile the code in parallel by using X threads.

At the end of the compilation a test executable used for development (`test_feature`) will be placed in the build/bin folder. If Python bindings are enabled, `make install` will install the package through `pip` to make the library accessible by the user.

### cmake options

Here is a list of options that can be passed to cmake during the pre-compilation stage:

* `-DPython=On|Off` enables|disables Python bindings (defaults to `On`)
* `-DDebug=On|Off` compiles with|without debug symbols and without|with optimisation flags (defaults to `Off`)
* `-DG=On|Off` compiles with|without debug symbols and optimisation flags (defaults to `Off`)

If `cmake` cannot find some of the libraries required, add their paths to the `CMAKE_FIND_ROOT_PATH` set command which can be found close to the top of the `CMakeLists.txt` file. 

### Test suite

baggianalysis contains a very simple testing suite that checks whether the Python bindings and core functionalities work. The test suite is run by using the `make test_quick` command. 

## Features

* Support many configuration types
* Filters: after being parsed, configurations can be modified by the so-called *filters*. Some available filters are:
	* filter particles according to a custom lambda function (`FilterByFunction`)
	* select particles of certain types only (`FilterByType`)
	* remove the centre-of-mass position and velocity (`SubtractCOM`)
	* reduce the configuration so that it contains only a single particle with position and velocity given by the centre-of-mass position and velocity (`FilterByReducingToCOM`)

## Notes

* By default, the core library is compiled dynamically. However, if Python bindings are enabled, the core library is compiled statically.
* The timestep associated to a configuration **must** be an integer number. If your preferred format stores it as a floating-precision number, your parser will have to find a way of converting that to an integer. This is *by design*, as the time of a configuration is used as a key in several maps around the code, and floating-point numbers are not good at that. Moreover, integer numbers can be stored without losing any precision, in contrast to floats.
* Normal trajectories need not load all the frames at once. Trajectories that do are called "full trajectories". Many observables, in general, do not require to have access to all frames at once, which means that frames can parsed (and hence loaded) one by one when needed (lazy loading). This allows to work on big trajectories without consuming up too much memory.
* Lists of 3D vectors are copied when accessed from the Python's side. This means that their c++ counterparts (which are `std::vector`s) are not modified when `append` or similar Python methods are used.
* Simple Python parsers can be used to either parse single `System`s or to initialise trajectories from file lists and folders only. In order to do so, parsers should inherit from `BaseParser` and override the `parse_file` method, which takes a string as its only argument.
* Molecules built by the `Topology` class are named `mol_XXX`, where `XXX` is an index that runs from zero to the number of molecules minus one.

## Acknowledgements

* The [glm 0.9.9](https://glm.g-truc.net/0.9.9/index.html), [pybind 2.4.3](https://github.com/pybind/pybind11) and [spdlog 1.8.2](https://github.com/gabime/spdlog) libraries are included in the source tree.
* Natural sorting of files is carried out with the [NaturalSort](https://github.com/scopeInfinity/NaturalSort) library.
* [akuukka's quickhull](https://github.com/akuukka/quickhull) library is included in the source tree.
