/*
 * LAMMPSDataFileExporter.cpp
 *
 *  Created on: Mar 1, 2020
 *      Author: lorenzo
 */

#include "LAMMPSDataFileExporter.h"

#include "../System.h"
#include "../topology/Topology.h"
#include "../utils/strings.h"

#include <fstream>

namespace ba {

LAMMPSDataFileExporter::LAMMPSDataFileExporter(std::string atom_style) :
				_atom_style(atom_style) {

}

LAMMPSDataFileExporter::~LAMMPSDataFileExporter() {

}

void LAMMPSDataFileExporter::_write_system_to_stream(std::shared_ptr<System> system, std::ostream &output) {
	auto types = system->types();
	// sorting and removal is faster than other options (see https://stackoverflow.com/a/56034014/5140209)
	// TODO run some performance tests
	std::sort(types.begin(), types.end());
	types.erase(std::unique(types.begin(), types.end()), types.end());

	Topology topology(system);
	topology.apply(system);
	uint N_bonds = topology.bonds().size();

	output << "Generated by baggianalysis" << std::endl << std::endl;

	output << fmt::format("{} atoms", system->N()) << std::endl;
	output << fmt::format("{} bonds", N_bonds) << std::endl;
	output << fmt::format("{} atom types", types.size()) << std::endl;
	output << fmt::format("1 bond types") << std::endl;

	output << std::endl;

	// print the box
	auto box_low = -system->box / 2.;
	auto box_high = box_low + system->box;
	output << fmt::format("{} {} xlo xhi", box_low[0], box_high[0]) << std::endl;
	output << fmt::format("{} {} ylo yhi", box_low[1], box_high[1]) << std::endl;
	output << fmt::format("{} {} zlo zhi", box_low[2], box_high[2]) << std::endl;

	output << std::endl;
	output << "Masses" << std::endl << std::endl;

	for(auto type : types) {
		output << fmt::format("{} 1 mass", type) << std::endl;
	}

	output << std::endl;
	output << "Atoms" << std::endl << std::endl;

	for(auto p : system->particles()) {
		output << _particle_line(p) << std::endl;
	}

	output << std::endl;
	output << "Bonds" << std::endl << std::endl;

	// LAMMPS's ids start from 1
	uint i = 1;
	for(auto b : topology.bonds()) {
		output << fmt::format("{} {} {} {}", i, 1, b[0], b[1]) << std::endl;
		i++;
	}
}

std::string LAMMPSDataFileExporter::_particle_line(std::shared_ptr<Particle> p) {
	int p_id = p->index();
	particle_type p_type = p->type();
	if(_atom_style == "bond") {
		int mol_id = ba::utils::lexical_cast<int>(ba::utils::split(p->molecule()->name(), "_")[1]) + 1;
		return fmt::format("{} {} {} {} {} {}", p_id, mol_id, p_type, p->position()[0], p->position()[1], p->position()[2]);
	}
	else if(_atom_style == "atomic") {
		return fmt::format("{} {} {} {} {}", p_id, p_type, p->position()[0], p->position()[1], p->position()[2]);
	}
	else {
		std::string error = fmt::format("Exporting LAMMPS atom_style '{}' is not supported", _atom_style);
		throw std::runtime_error(error);
	}
}

#ifdef PYTHON_BINDINGS

void export_LAMMPSDataFileExporter(py::module &m) {
	py::class_<LAMMPSDataFileExporter, BaseExporter, std::shared_ptr<LAMMPSDataFileExporter>> exporter(m, "LAMMPSDataFileExporter", R"pbdoc(
		 Export configurations to the LAMMPS data file format. 
	)pbdoc");

	exporter.def(py::init<std::string>(), R"pbdoc(
		The constructor takes one mandatory argument.

        Parameters
        ----------
            atom_style : str
                The atom style to be used in printing the data file (see `here <https://lammps.sandia.gov/doc/atom_style.html>`_ for details). As of now, only the "atomic" and "bond" styles are supported.
	)pbdoc");
}

#endif

} /* namespace ba */
