import { ILogGroup } from '@aws-cdk/aws-logs';
import { IStage } from './stage';
/**
 * Access log destination for a RestApi Stage.
 *
 * @stability stable
 */
export interface IAccessLogDestination {
    /**
     * Binds this destination to the RestApi Stage.
     *
     * @stability stable
     */
    bind(stage: IStage): AccessLogDestinationConfig;
}
/**
 * Options when binding a log destination to a RestApi Stage.
 *
 * @stability stable
 */
export interface AccessLogDestinationConfig {
    /**
     * The Amazon Resource Name (ARN) of the destination resource.
     *
     * @stability stable
     */
    readonly destinationArn: string;
}
/**
 * Use CloudWatch Logs as a custom access log destination for API Gateway.
 *
 * @stability stable
 */
export declare class LogGroupLogDestination implements IAccessLogDestination {
    private readonly logGroup;
    /**
     * @stability stable
     */
    constructor(logGroup: ILogGroup);
    /**
     * Binds this destination to the CloudWatch Logs.
     *
     * @stability stable
     */
    bind(_stage: IStage): AccessLogDestinationConfig;
}
/**
 * $context variables that can be used to customize access log pattern.
 *
 * @stability stable
 */
export declare class AccessLogField {
    /**
     * The API owner's AWS account ID.
     *
     * @stability stable
     */
    static contextAccountId(): string;
    /**
     * The identifier API Gateway assigns to your API.
     *
     * @stability stable
     */
    static contextApiId(): string;
    /**
     * A property of the claims returned from the Amazon Cognito user pool after the method caller is successfully authenticated.
     *
     * @param property A property key of the claims.
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-integrate-with-cognito.html
     * @stability stable
     */
    static contextAuthorizerClaims(property: string): string;
    /**
     * The principal user identification associated with the token sent by the client and returned from an API Gateway Lambda authorizer (formerly known as a custom authorizer).
     *
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-use-lambda-authorizer.html
     * @stability stable
     */
    static contextAuthorizerPrincipalId(): string;
    /**
     * The stringified value of the specified key-value pair of the `context` map returned from an API Gateway Lambda authorizer function.
     *
     * @param property key of the context map.
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-use-lambda-authorizer.html
     * @stability stable
     */
    static contextAuthorizer(property: string): string;
    /**
     * The AWS endpoint's request ID.
     *
     * @stability stable
     */
    static contextAwsEndpointRequestId(): string;
    /**
     * The full domain name used to invoke the API.
     *
     * This should be the same as the incoming `Host` header.
     *
     * @stability stable
     */
    static contextDomainName(): string;
    /**
     * The first label of the `$context.domainName`. This is often used as a caller/customer identifier.
     *
     * @stability stable
     */
    static contextDomainPrefix(): string;
    /**
     * A string containing an API Gateway error message.
     *
     * @stability stable
     */
    static contextErrorMessage(): string;
    /**
     * The quoted value of $context.error.message, namely "$context.error.message".
     *
     * @stability stable
     */
    static contextErrorMessageString(): string;
    /**
     * A type of GatewayResponse.
     *
     * This variable can only be used for simple variable substitution in a GatewayResponse body-mapping template,
     * which is not processed by the Velocity Template Language engine, and in access logging.
     *
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/customize-gateway-responses.html
     * @stability stable
     */
    static contextErrorResponseType(): string;
    /**
     * A string containing a detailed validation error message.
     *
     * @stability stable
     */
    static contextErrorValidationErrorString(): string;
    /**
     * The extended ID that API Gateway assigns to the API request, which contains more useful information for debugging/troubleshooting.
     *
     * @stability stable
     */
    static contextExtendedRequestId(): string;
    /**
     * The HTTP method used.
     *
     * Valid values include: `DELETE`, `GET`, `HEAD`, `OPTIONS`, `PATCH`, `POST`, and `PUT`.
     *
     * @stability stable
     */
    static contextHttpMethod(): string;
    /**
     * The AWS account ID associated with the request.
     *
     * @stability stable
     */
    static contextIdentityAccountId(): string;
    /**
     * For API methods that require an API key, this variable is the API key associated with the method request.
     *
     * For methods that don't require an API key, this variable is
     *
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-api-usage-plans.html
     * @stability stable
     */
    static contextIdentityApiKey(): string;
    /**
     * The API key ID associated with an API request that requires an API key.
     *
     * @stability stable
     */
    static contextIdentityApiKeyId(): string;
    /**
     * The principal identifier of the caller making the request.
     *
     * @stability stable
     */
    static contextIdentityCaller(): string;
    /**
     * The Amazon Cognito authentication provider used by the caller making the request.
     *
     * Available only if the request was signed with Amazon Cognito credentials.
     *
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-identity.html
     * @stability stable
     */
    static contextIdentityCognitoAuthenticationProvider(): string;
    /**
     * The Amazon Cognito authentication type of the caller making the request.
     *
     * Available only if the request was signed with Amazon Cognito credentials.
     *
     * @stability stable
     */
    static contextIdentityCognitoAuthenticationType(): string;
    /**
     * The Amazon Cognito identity ID of the caller making the request.
     *
     * Available only if the request was signed with Amazon Cognito credentials.
     *
     * @stability stable
     */
    static contextIdentityCognitoIdentityId(): string;
    /**
     * The Amazon Cognito identity pool ID of the caller making the request.
     *
     * Available only if the request was signed with Amazon Cognito credentials.
     *
     * @stability stable
     */
    static contextIdentityCognitoIdentityPoolId(): string;
    /**
     * The AWS organization ID.
     *
     * @stability stable
     */
    static contextIdentityPrincipalOrgId(): string;
    /**
     * The source IP address of the TCP connection making the request to API Gateway.
     *
     * Warning: You should not trust this value if there is any chance that the `X-Forwarded-For` header could be forged.
     *
     * @stability stable
     */
    static contextIdentitySourceIp(): string;
    /**
     * The principal identifier of the user making the request.
     *
     * Used in Lambda authorizers.
     *
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-lambda-authorizer-output.html
     * @stability stable
     */
    static contextIdentityUser(): string;
    /**
     * The User-Agent header of the API caller.
     *
     * @stability stable
     */
    static contextIdentityUserAgent(): string;
    /**
     * The Amazon Resource Name (ARN) of the effective user identified after authentication.
     *
     * @see https://docs.aws.amazon.com/IAM/latest/UserGuide/id_users.html
     * @stability stable
     */
    static contextIdentityUserArn(): string;
    /**
     * The request path.
     *
     * For example, for a non-proxy request URL of https://{rest-api-id.execute-api.{region}.amazonaws.com/{stage}/root/child,
     * this value is /{stage}/root/child.
     *
     * @stability stable
     */
    static contextPath(): string;
    /**
     * The request protocol, for example, HTTP/1.1.
     *
     * @stability stable
     */
    static contextProtocol(): string;
    /**
     * The ID that API Gateway assigns to the API request.
     *
     * @stability stable
     */
    static contextRequestId(): string;
    /**
     * The request header override.
     *
     * If this parameter is defined, it contains the headers to be used instead of the HTTP Headers that are defined in the Integration Request pane.
     *
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-override-request-response-parameters.html
     * @stability stable
     */
    static contextRequestOverrideHeader(headerName: string): string;
    /**
     * The request path override.
     *
     * If this parameter is defined,
     * it contains the request path to be used instead of the URL Path Parameters that are defined in the Integration Request pane.
     *
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-override-request-response-parameters.html
     * @stability stable
     */
    static contextRequestOverridePath(pathName: string): string;
    /**
     * The request query string override.
     *
     * If this parameter is defined, it contains the request query strings to be used instead
     * of the URL Query String Parameters that are defined in the Integration Request pane.
     *
     * @stability stable
     */
    static contextRequestOverrideQuerystring(querystringName: string): string;
    /**
     * The response header override.
     *
     * If this parameter is defined, it contains the header to be returned instead of the Response header
     * that is defined as the Default mapping in the Integration Response pane.
     *
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-override-request-response-parameters.html
     * @stability stable
     */
    static contextResponseOverrideHeader(headerName: string): string;
    /**
     * The response status code override.
     *
     * If this parameter is defined, it contains the status code to be returned instead of the Method response status
     * that is defined as the Default mapping in the Integration Response pane.
     *
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-override-request-response-parameters.html
     * @stability stable
     */
    static contextResponseOverrideStatus(): string;
    /**
     * The CLF-formatted request time (dd/MMM/yyyy:HH:mm:ss +-hhmm).
     *
     * @stability stable
     */
    static contextRequestTime(): string;
    /**
     * The Epoch-formatted request time.
     *
     * @stability stable
     */
    static contextRequestTimeEpoch(): string;
    /**
     * The identifier that API Gateway assigns to your resource.
     *
     * @stability stable
     */
    static contextResourceId(): string;
    /**
     * The path to your resource.
     *
     * For example, for the non-proxy request URI of `https://{rest-api-id.execute-api.{region}.amazonaws.com/{stage}/root/child`,
     * The $context.resourcePath value is `/root/child`.
     *
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-create-api-step-by-step.html
     * @stability stable
     */
    static contextResourcePath(): string;
    /**
     * The deployment stage of the API request (for example, `Beta` or `Prod`).
     *
     * @stability stable
     */
    static contextStage(): string;
    /**
     * The response received from AWS WAF: `WAF_ALLOW` or `WAF_BLOCK`.
     *
     * Will not be set if the stage is not associated with a web ACL.
     *
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-control-access-aws-waf.html
     * @stability stable
     */
    static contextWafResponseCode(): string;
    /**
     * The complete ARN of the web ACL that is used to decide whether to allow or block the request.
     *
     * Will not be set if the stage is not associated with a web ACL.
     *
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-control-access-aws-waf.html
     * @stability stable
     */
    static contextWebaclArn(): string;
    /**
     * The trace ID for the X-Ray trace.
     *
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-enabling-xray.html
     * @stability stable
     */
    static contextXrayTraceId(): string;
    /**
     * The authorizer latency in ms.
     *
     * @stability stable
     */
    static contextAuthorizerIntegrationLatency(): string;
    /**
     * The integration latency in ms.
     *
     * @stability stable
     */
    static contextIntegrationLatency(): string;
    /**
     * For Lambda proxy integration, this parameter represents the status code returned from AWS Lambda, not from the backend Lambda function.
     *
     * @stability stable
     */
    static contextIntegrationStatus(): string;
    /**
     * The response latency in ms.
     *
     * @stability stable
     */
    static contextResponseLatency(): string;
    /**
     * The response payload length.
     *
     * @stability stable
     */
    static contextResponseLength(): string;
    /**
     * The method response status.
     *
     * @stability stable
     */
    static contextStatus(): string;
}
/**
 * Properties for controlling items output in JSON standard format.
 *
 * @stability stable
 */
export interface JsonWithStandardFieldProps {
    /**
     * If this flag is enabled, the source IP of request will be output to the log.
     *
     * @stability stable
     */
    readonly ip: boolean;
    /**
     * If this flag is enabled, the principal identifier of the caller will be output to the log.
     *
     * @stability stable
     */
    readonly caller: boolean;
    /**
     * If this flag is enabled, the principal identifier of the user will be output to the log.
     *
     * @stability stable
     */
    readonly user: boolean;
    /**
     * If this flag is enabled, the CLF-formatted request time((dd/MMM/yyyy:HH:mm:ss +-hhmm) will be output to the log.
     *
     * @stability stable
     */
    readonly requestTime: boolean;
    /**
     * If this flag is enabled, the http method will be output to the log.
     *
     * @stability stable
     */
    readonly httpMethod: boolean;
    /**
     * If this flag is enabled, the path to your resource will be output to the log.
     *
     * @stability stable
     */
    readonly resourcePath: boolean;
    /**
     * If this flag is enabled, the method response status will be output to the log.
     *
     * @stability stable
     */
    readonly status: boolean;
    /**
     * If this flag is enabled, the request protocol will be output to the log.
     *
     * @stability stable
     */
    readonly protocol: boolean;
    /**
     * If this flag is enabled, the response payload length will be output to the log.
     *
     * @stability stable
     */
    readonly responseLength: boolean;
}
/**
 * factory methods for access log format.
 *
 * @stability stable
 */
export declare class AccessLogFormat {
    /**
     * Custom log format.
     *
     * You can create any log format string. You can easily get the $ context variable by using the methods of AccessLogField.
     *
     * @stability stable
     * @example
     *
     *  apigateway.AccessLogFormat.custom(JSON.stringify({
     *      requestId: apigateway.AccessLogField.contextRequestId(),
     *      sourceIp: apigateway.AccessLogField.contextIdentitySourceIp(),
     *      method: apigateway.AccessLogField.contextHttpMethod(),
     *      userContext: {
     *        sub: apigateway.AccessLogField.contextAuthorizerClaims('sub'),
     *        email: apigateway.AccessLogField.contextAuthorizerClaims('email')
     *      }
     *   }))
     */
    static custom(format: string): AccessLogFormat;
    /**
     * Generate Common Log Format.
     *
     * @stability stable
     */
    static clf(): AccessLogFormat;
    /**
     * Access log will be produced in the JSON format with a set of fields most useful in the access log.
     *
     * All fields are turned on by default with the
     * option to turn off specific fields.
     *
     * @stability stable
     */
    static jsonWithStandardFields(fields?: JsonWithStandardFieldProps): AccessLogFormat;
    /**
     * A API Gateway custom access log format
     */
    private readonly format;
    private constructor();
    /**
     * Output a format string to be used with CloudFormation.
     *
     * @stability stable
     */
    toString(): string;
}
