"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.JsonSchemaMapper = exports.validateInteger = exports.parseAwsApiCall = exports.parseMethodOptionsPath = exports.validateHttpMethod = exports.ALL_METHODS = void 0;
const url_1 = require("url");
const jsonSchema = require("./json-schema");
exports.ALL_METHODS = ['OPTIONS', 'GET', 'PUT', 'POST', 'DELETE', 'PATCH', 'HEAD'];
const ALLOWED_METHODS = ['ANY', ...exports.ALL_METHODS];
function validateHttpMethod(method, messagePrefix = '') {
    if (!ALLOWED_METHODS.includes(method)) {
        throw new Error(`${messagePrefix}Invalid HTTP method "${method}". Allowed methods: ${ALLOWED_METHODS.join(',')}`);
    }
}
exports.validateHttpMethod = validateHttpMethod;
function parseMethodOptionsPath(originalPath) {
    if (!originalPath.startsWith('/')) {
        throw new Error(`Method options path must start with '/': ${originalPath}`);
    }
    const path = originalPath.substr(1); // trim trailing '/'
    const components = path.split('/');
    if (components.length < 2) {
        throw new Error(`Method options path must include at least two components: /{resource}/{method} (i.e. /foo/bar/GET): ${path}`);
    }
    const httpMethod = components.pop().toUpperCase(); // last component is an HTTP method
    if (httpMethod !== '*') {
        validateHttpMethod(httpMethod, `${originalPath}: `);
    }
    let resourcePath = '/~1' + components.join('~1');
    if (components.length === 1 && components[0] === '*') {
        resourcePath = '/*';
    }
    else if (components.length === 1 && components[0] === '') {
        resourcePath = '/';
    }
    return {
        httpMethod,
        resourcePath,
    };
}
exports.parseMethodOptionsPath = parseMethodOptionsPath;
function parseAwsApiCall(path, action, actionParams) {
    if (actionParams && !action) {
        throw new Error('"actionParams" requires that "action" will be set');
    }
    if (path && action) {
        throw new Error(`"path" and "action" are mutually exclusive (path="${path}", action="${action}")`);
    }
    if (path) {
        return {
            apiType: 'path',
            apiValue: path,
        };
    }
    if (action) {
        if (actionParams) {
            action += '&' + url_1.format({ query: actionParams }).substr(1);
        }
        return {
            apiType: 'action',
            apiValue: action,
        };
    }
    throw new Error('Either "path" or "action" are required');
}
exports.parseAwsApiCall = parseAwsApiCall;
function validateInteger(property, messagePrefix) {
    if (property && !Number.isInteger(property)) {
        throw new Error(`${messagePrefix} should be an integer`);
    }
}
exports.validateInteger = validateInteger;
class JsonSchemaMapper {
    /**
     * Transforms naming of some properties to prefix with a $, where needed
     * according to the JSON schema spec
     * @param schema The JsonSchema object to transform for CloudFormation output
     */
    static toCfnJsonSchema(schema) {
        const result = JsonSchemaMapper._toCfnJsonSchema(schema);
        if (!('$schema' in result)) {
            result.$schema = jsonSchema.JsonSchemaVersion.DRAFT4;
        }
        return result;
    }
    static _toCfnJsonSchema(schema, preserveKeys = false) {
        if (schema == null || typeof schema !== 'object') {
            return schema;
        }
        if (Array.isArray(schema)) {
            return schema.map(entry => JsonSchemaMapper._toCfnJsonSchema(entry));
        }
        return Object.assign({}, ...Object.entries(schema).map(([key, value]) => {
            const mapKey = !preserveKeys && (key in JsonSchemaMapper.SchemaPropsWithPrefix);
            const newKey = mapKey ? JsonSchemaMapper.SchemaPropsWithPrefix[key] : key;
            // If keys were preserved, don't consider SchemaPropsWithUserDefinedChildren for those keys (they are user-defined!)
            const newValue = JsonSchemaMapper._toCfnJsonSchema(value, !preserveKeys && JsonSchemaMapper.SchemaPropsWithUserDefinedChildren[key]);
            return { [newKey]: newValue };
        }));
    }
}
exports.JsonSchemaMapper = JsonSchemaMapper;
JsonSchemaMapper.SchemaPropsWithPrefix = {
    schema: '$schema',
    ref: '$ref',
    id: '$id',
};
// The value indicates whether direct children should be key-mapped.
JsonSchemaMapper.SchemaPropsWithUserDefinedChildren = {
    definitions: true,
    properties: true,
    patternProperties: true,
    dependencies: true,
};
//# sourceMappingURL=data:application/json;base64,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