# AUTOGENERATED! DO NOT EDIT! File to edit: nbs/06_layers.ipynb (unless otherwise specified).

__all__ = ['LinearBlock', 'ValueHead', 'Conv', 'Conv1d', 'Conv2d', 'Conv3d', 'SphericalDistribution', 'Prior',
           'NormalPrior', 'SphericalPrior', 'SequenceDropout', 'Conditional_LSTM', 'LSTM', 'Conditional_LSTM_Block',
           'LSTM_Block', 'Encoder', 'LSTM_Encoder', 'MLP_Encoder', 'Conv_Encoder', 'MLP']

# Cell
from .imports import *
from .torch_imports import *
from .torch_core import *

# Cell

class LinearBlock(nn.Module):
    '''
    LinearBlock - Combined linear, batchnorm, ReLU and dropout.
    Layers are executed in the order linear, batchnorm, ReLU, dropout.
    Batchnorm, activation and dropout layers are optional

    Inputs:

    - `d_in int`: number of input dimensions

    - `d_out int`: number of output dimensions

    - `act bool`: if True, applies a ReLU activation

    - `bn bool`: if True, applies 1d batchnorm

    - `dropout float`: dropout percentage

    - `**lin_kwargs dict`: keyword args passed to nn.Linear
    '''

    def __init__(self, d_in, d_out, act=True, bn=False, dropout=0., **lin_kwargs):
        super().__init__()

        layers = [nn.Linear(d_in, d_out, **lin_kwargs)]

        if bn:
            layers.append(nn.BatchNorm1d(d_out))

        if act:
            layers.append(nn.ReLU())

        if dropout>0.:
            layers.append(nn.Dropout(p=dropout))

        self.layers = nn.Sequential(*layers)

    def forward(self, x):
        return self.layers(x)

# Cell

class ValueHead(nn.Module):
    '''
    ValueHead - used in RL algorithms to predict state values

    Inputs:

    - `d_in int`: number of input dimensions

    '''
    def __init__(self, d_in, dropout=0.):
        super().__init__()
        self.drop = nn.Dropout(dropout)
        self.layer = nn.Linear(d_in, 1)

    def forward(self, x):
        x = self.layer(self.drop(x)).squeeze(-1)
        return x

# Cell

class Conv(nn.Module):
    '''
    Conv - base module for convolutions

    Inputs:

    - `d_in int`: number of input dimensions

    - `d_out int`: number of output dimensions

    - `ks int`: kernel size

    - `stride int`: stride

    - `padding [int, None]`: padding. If None, derived from kernel size

    - `ndim int`: conv dimension (1D conv, 2D conv, 3D conv)

    - `act bool`: if True, applies a ReLU activation

    - `bn bool`: if True, applies batchnorm consistent with `ndim`

    - `dropout float`: dropout percentage

    - `**conv_kwargs dict`: keyword args passed to nn.Conv
    '''
    def __init__(self, d_in, d_out, ks=3, stride=1, padding=None, ndim=2,
                 act=True, bn=False, dropout=0., **conv_kwargs):
        super().__init__()

        if padding is None:
            padding = (ks-1)//2

        if ndim==1:
            conv_func = nn.Conv1d
            bn_func = nn.BatchNorm1d
        elif ndim==2:
            conv_func = nn.Conv2d
            bn_func = nn.BatchNorm2d
        else:
            conv_func = nn.Conv3d
            bn_func = nn.BatchNorm3d

        layers = [conv_func(d_in, d_out, ks, stride, padding=padding, **conv_kwargs)]

        if bn:
            layers.append(bn_func(d_out))

        if act:
            layers.append(nn.ReLU())

        if dropout>0.:
            layers.append(nn.Dropout(p=dropout))

        self.layers = nn.Sequential(*layers)

    def forward(self, x):
        return self.layers(x)

class Conv1d(Conv):
    '''
    Conv1d - 1D convolution

    Inputs:

    - `d_in int`: number of input dimensions

    - `d_out int`: number of output dimensions

    - `ks int`: kernel size

    - `stride int`: stride

    - `padding [int, None]`: padding. If None, derived from kernel size

    - `act bool`: if True, applies a ReLU activation

    - `bn bool`: if True, applies batchnorm consistent with `ndim`

    - `dropout float`: dropout percentage

    - `**conv_kwargs dict`: keyword args passed to nn.Conv1D
    '''
    def __init__(self, d_in, d_out, ks=3, stride=1, padding=None,
                 act=True, bn=False, dropout=0., **conv_kwargs):
        super().__init__(d_in, d_out, ks, stride, padding, ndim=1,
                 act=act, bn=bn, dropout=dropout, **conv_kwargs)

class Conv2d(Conv):
    '''
    Conv2d - 2D convolution

    Inputs:

    - `d_in int`: number of input dimensions

    - `d_out int`: number of output dimensions

    - `ks int`: kernel size

    - `stride int`: stride

    - `padding [int, None]`: padding. If None, derived from kernel size

    - `act bool`: if True, applies a ReLU activation

    - `bn bool`: if True, applies batchnorm consistent with `ndim`

    - `dropout float`: dropout percentage

    - `**conv_kwargs dict`: keyword args passed to nn.Conv2D
    '''
    def __init__(self, d_in, d_out, ks=3, stride=1, padding=None,
                 act=True, bn=False, dropout=0., **conv_kwargs):
        super().__init__(d_in, d_out, ks, stride, padding, ndim=2,
                 act=act, bn=bn, dropout=dropout, **conv_kwargs)

class Conv3d(Conv):
    '''
    Conv3d - 3D convolution

    Inputs:

    - `d_in int`: number of input dimensions

    - `d_out int`: number of output dimensions

    - `ks int`: kernel size

    - `stride int`: stride

    - `padding [int, None]`: padding. If None, derived from kernel size

    - `act bool`: if True, applies a ReLU activation

    - `bn bool`: if True, applies batchnorm consistent with `ndim`

    - `dropout float`: dropout percentage

    - `**conv_kwargs dict`: keyword args passed to nn.Conv3D
    '''
    def __init__(self, d_in, d_out, ks=3, stride=1, padding=None,
                 act=True, bn=False, dropout=0., **conv_kwargs):
        super().__init__(d_in, d_out, ks, stride, padding, ndim=3,
                 act=act, bn=bn, dropout=dropout, **conv_kwargs)

# Cell

class SphericalDistribution(torch.distributions.Distribution):
    '''
    SphericalDistribution - samples from points on the surface of a sphere

    Inputs:

    - `loc torch.Tensor`: vector of means

    - `scale torch.Tensor`: vector of variances
    '''
    def __init__(self, loc, scale, validate_args=False):
        super().__init__(loc.shape[0], validate_args=validate_args)
        self.dim = loc.shape[0]
        self.loc = loc
        self.scale = scale
        self.dist = Normal(self.loc, self.scale)

    def sample(self, n):
        s = self.dist.sample(n)
        s = F.normalize(s, p=2, dim=-1)
        return s

    def rsample(self, n):
        s = self.dist.rsample(n)
        s = F.normalize(s, p=2, dim=-1)
        return s

    def __repr__(self):
        return f'Spherical(loc: {self.loc.size()}, scale: {self.scale.size()})'

# Cell

class Prior(nn.Module):
    '''
    Prior - base class for trainable priors
    '''
    def __init__(self):
        super().__init__()

    def get_dist(self):
        raise NotImplementedError

    def log_prob(self, x):
        raise NotImplementedError

    def sample(self, n):
        if type(n)==int:
            n = [n]
        return self.get_dist().sample(n)

    def rsample(self, n):
        if type(n)==int:
            n = [n]
        return self.get_dist().rsample(n)

# Cell

class NormalPrior(Prior):
    '''
    NormalPrior - normal distribution prior

    Inputs:

    - `loc torch.Tensor`: vector of means

    - `log_scale torch.Tensor`: vector of log-variances

    - `trainable bool`: if True, `loc` and `scale` are trainable

    Note that log-variances are used for stability. Optimizing
    the variance directly can cause issues with gradient descent
    sending variance values negative
    '''
    def __init__(self, loc, log_scale, trainable=True):
        super().__init__()
        if trainable:
            loc = nn.Parameter(loc)
            log_scale = nn.Parameter(log_scale)
            self.loc = loc
            self.log_scale = log_scale
        else:
            self.register_buffer('loc', loc)
            self.register_buffer('log_scale', log_scale)

        self.trainable = trainable

    def get_dist(self):
        return Normal(self.loc, self.log_scale.exp())

    def log_prob(self, x):
        var = self.log_scale.exp().pow(2)
        return -((x - self.loc) ** 2) / (2 * var) - self.log_scale - math.log(math.sqrt(2 * math.pi))

class SphericalPrior(NormalPrior):
    '''
    SphericalPrior - spherical distribution prior

    Inputs:

    - `loc torch.Tensor`: vector of means

    - `log_scale torch.Tensor`: vector of log-variances

    - `trainable bool`: if True, `loc` and `scale` are trainable

    Note that log-variances are used for stability. Optimizing
    the variance directly can cause issues with gradient descent
    sending variance values negative
    '''
    def __init__(self, loc, log_scale, trainable=True):
        super().__init__(loc, log_scale, trainable)

    def get_dist(self):
        return SphericalDistribution(self.loc, self.log_scale.exp())

# Cell

class SequenceDropout(nn.Module):
    '''
    SequenceDropout - dropout along the sequence dimension

    Inputs:

    - `p float`: dropout probability

    - `batch_first bool`: if batch dimension is first in input tensors

    Samples a dropout mask that is constant in the sequence dimension

    '''
    def __init__(self, p, batch_first=True):
        super().__init__()
        self.p = p
        self.batch_first = batch_first

    def forward(self, x):
        if self.training and self.p>0:

            if self.batch_first:
                sizes = (x.shape[0], 1, *x.shape[2:])
            else:
                sizes = (1, x.shape[0], *x.shape[2:])

            mask = x.new_empty(*sizes).bernoulli_(1-self.p).bool()
            x = x*mask

        return x


# Cell

class Conditional_LSTM(nn.Module):
    '''
    Conditional_LSTM - Conditional LSTM module

    Inputs:

    - `d_embedding int`: embedding dimension

    - `d_hidden int`: hidden dimension

    - `d_output int`: output dimension

    - `d_latent int`: latent vector dimension

    - `n_layers int`: number of layers

    - `condition_hidden bool`: if True, latent vector is used to initialize the
    hidden state

    - `condition_output bool`: if True, latent vector is concatenated to inputs

    - `bidir bool`: if the LSTM should be bidirectional

    - `input_dropout float`: dropout percentage on inputs

    - `lstm_dropout float`: dropout on LSTM layers

    - `batch_first bool`: if batch dimension is first on input tensors
    '''
    def __init__(self, d_embedding, d_hidden, d_output, d_latent, n_layers,
                 condition_hidden=True, condition_output=True,
                 bidir=False, input_dropout=0., lstm_dropout=0., batch_first=True):
        super().__init__()

        self.d_embedding = d_embedding
        self.d_hidden = d_hidden
        self.d_output = d_output
        self.n_layers = n_layers
        self.bidir = bidir
        self.n_dir = 1 if not bidir else 2
        self.batch_first = batch_first
        self.condition_hidden = condition_hidden
        self.condition_output = condition_output

        self.input_drop = SequenceDropout(input_dropout)
        self.lstm_drop = SequenceDropout(lstm_dropout)

        self.lstms = []
        self.hidden_sizes = []

        for l in range(n_layers):
            if l==0:
                input_size = d_embedding if not self.condition_output else d_embedding+d_latent
            else:
                input_size = d_hidden

            output_size = d_output if l==n_layers-1 else d_hidden
            output_size = output_size // self.n_dir

            hidden_size = (self.n_dir, 1, output_size)
            self.hidden_sizes.append(hidden_size)

            lstm = nn.LSTM(input_size, output_size, 1, batch_first=batch_first,
                           bidirectional=bidir)
            self.lstms.append(lstm)

        self.lstms = nn.ModuleList(self.lstms)

        if self.condition_hidden:
            to_hidden = []
            for size in self.hidden_sizes:
                ndir, _, dim = size
                to_hidden.append(nn.Linear(d_latent, ndir*dim*2))

            self.to_hidden = nn.ModuleList(to_hidden)

    def forward(self, x, z, hiddens=None):
        '''
        Inputs:

        - `x torch.Tensor[(bs, sl, d_embedding) or (sl, bs, d_embedding)]`: input tensor

        - `z torch.Tensor[bs, d_latent]`: latent vector

        - `hiddens Optional[list[(hidden_state, cell_state)]]`: hidden state. If None,
        a new hidden state is initialized

        '''

        x = self.prepare_x(x, z)
        x = self.input_drop(x)
        bs = x.shape[0] if self.batch_first else x.shape[1]

        hiddens = self.get_hidden(z, hiddens, bs)
        hiddens = to_device(hiddens, x.device)

        new_hiddens = []
        for i, lstm in enumerate(self.lstms):
            x, (h,c) = lstm(x, hiddens[i])
            new_hiddens.append((h.detach(), c.detach()))
            if i<self.n_layers:
                x = self.lstm_drop(x)

        return x, new_hiddens

    def prepare_x(self, x, z):
        '''
        Concatenate x to latent vector if `self.condition_output`
        '''
        if self.condition_output:
            if self.batch_first:
                sl = x.shape[1]
                z_ = z.unsqueeze(1).repeat(1,sl,1)
            else:
                sl = x.shape[0]
                z_ = z.unsqueeze(0).repeat(sl,1,1)

            x = torch.cat([x, z_], -1)

        return x

    def get_hidden(self, z, hiddens, bs):
        '''
        Initializes hidden state.
        '''
        if hiddens is None:
            if self.condition_hidden:
                hiddens = self.latent_to_hidden(z)

            else:
                hiddens = self.get_new_hidden(bs)

        return hiddens

    def latent_to_hidden(self, z):
        '''
        converts latent vector `z` into new hidden state
        '''
        hiddens = []
        for layer in self.to_hidden:
            h = layer(z)
            h,c = torch.chunk(h, 2, dim=-1)
            bs, _ = h.shape
            h = h.contiguous().reshape(bs, self.n_dir, -1).permute(1,0,2)
            c = c.contiguous().reshape(bs, self.n_dir, -1).permute(1,0,2)
            hiddens.append((h,c))

        return hiddens

    def get_new_hidden(self, bs):
        '''
        initializes new zeroed hidden states
        '''
        hiddens = []
        for hs in self.hidden_sizes:
            h = torch.zeros(hs).repeat(1,bs,1)
            c = torch.zeros(hs).repeat(1,bs,1)
            hiddens.append((h,c))

        return hiddens

    def mixup_hiddens(self, hiddens):
        '''
        shuffles hidden states between layers for style mixing
        '''
        new_hiddens = []
        for item in hiddens:
            h,c = item
            shuffle = to_device(torch.randperm(h.shape[1]))
            h = h[:,shuffle]
            c = c[:,shuffle]
            new_hiddens.append((h,c))
        return new_hiddens

# Cell

class LSTM(Conditional_LSTM):
    '''
    LSTM - LSTM module

    Inputs:

    - `d_embedding int`: embedding dimension

    - `d_hidden int`: hidden dimension

    - `d_output int`: output dimension

    - `n_layers int`: number of layers

    - `bidir bool`: if the LSTM should be bidirectional

    - `input_dropout float`: dropout percentage on inputs

    - `lstm_dropout float`: dropout on LSTM layers

    - `batch_first bool`: if batch dimension is first on input tensors
    '''
    def __init__(self, d_embedding, d_hidden, d_output, n_layers,
                 bidir=False, input_dropout=0., lstm_dropout=0.,
                 batch_first=True):

        super().__init__(d_embedding, d_hidden, d_output, 0, n_layers,
                 condition_hidden=False, condition_output=False,
                 bidir=bidir, input_dropout=input_dropout,
                 lstm_dropout=lstm_dropout, batch_first=batch_first)

    def forward(self, x, hiddens=None):
        '''
        Inputs:

        - `x torch.Tensor[(bs, sl, d_embedding) or (sl, bs, d_embedding)]:` input tensor

        - `hiddens Optional[list[(hidden_state, cell_state)]]`: hidden state. If None,
        a new hidden state is initialized

        '''

        x, new_hiddens = super().forward(x, None, hiddens)

        return x, new_hiddens

# Cell

class Conditional_LSTM_Block(nn.Module):
    '''
    Conditional_LSTM_Block - combines Embedding, Conditional LSTM,
    and output layer

    Inputs:

    - `d_vocab int`: vocab size

    - `d_embedding int`: embedding dimension

    - `d_hidden int`: hidden dimension

    - `d_output int`: output dimension

    - `d_latent int`: latent vector dimension

    - `n_layers int`: number of layers

    - `input_dropout float`: dropout percentage on inputs

    - `lstm_dropout float`: dropout on LSTM layers

    - `bidir bool`: if the LSTM should be bidirectional

    - `condition_hidden bool`: if True, latent vector is used to initialize the
    hidden state

    - `condition_output bool`: if True, latent vector is concatenated to inputs

    - `forward_rollout bool`: if model should generate outputs through rollout
    with teacher forcing

    - `p_force float`: teacher forcing frequency

    - `p_force_decay float`: teacher forcing decay rate
    '''
    def __init__(self, d_vocab, d_embedding, d_hidden, d_output, d_latent, n_layers,
                 input_dropout=0., lstm_dropout=0., bidir=False,
                 condition_hidden=True, condition_output=False,
                 forward_rollout=False, p_force=1., p_force_decay=1.):
        super().__init__()

        self.embedding = nn.Embedding(d_vocab, d_embedding)
        self.lstm = Conditional_LSTM(d_embedding, d_hidden, d_output, d_latent, n_layers,
                                    condition_hidden=condition_hidden, condition_output=condition_output,
                                     bidir=bidir, input_dropout=input_dropout, lstm_dropout=lstm_dropout)

        self.head = nn.Linear(d_output, d_vocab)
        self.forward_rollout = forward_rollout
        self.p_force = p_force
        self.p_force_decay = p_force_decay

    def forward(self, x, z, hiddens=None):
        '''
        Inputs:

        - `x torch.Tensor[(bs, sl, d_embedding) or (sl, bs, d_embedding)]`: input tensor

        - `z torch.Tensor[bs, d_latent]`: latent vector

        - `hiddens Optional[list[(hidden_state, cell_state)]]`: hidden state. If None,
        a new hidden state is initialized
        '''
        if self.forward_rollout:
            output, hiddens, encoded = self._forward_rollout(x,z,hiddens)
        else:
            output, hiddens, encoded = self._forward(x,z,hiddens)

        return output, hiddens, encoded


    def _forward(self, x, z, hiddens=None):
        x = self.embedding(x)
        encoded, hiddens = self.lstm(x, z, hiddens)
        output = self.head(encoded)

        return output, hiddens, encoded

    def _forward_rollout(self, x, z, hiddens=None):
        '''
        _forward_rollout - forward with self-rollout

        At each time step, the model generates a new output. With `p_force`, the
        corret next step from the input is sent to the model, or with `1-p_force`
        the model's last prediction is sent as the next step
        '''
        bs = x.shape[0]
        sl = x.shape[1]

        idxs = x[:,0].unsqueeze(-1)

        output = []
        encoded = []

        for i in range(1, sl):
            output_iter, hiddens, encoded_iter = self._forward(idxs, z, hiddens=hiddens)
            output.append(output_iter)
            encoded.append(encoded_iter)

            with torch.no_grad():
                idxs = F.softmax(output_iter, -1).argmax(-1)

            if np.random.random()<self.p_force:
                idxs = x[:,i].unsqueeze(-1)

        output = torch.cat(output, 1)
        encoded = torch.cat(encoded, 1)
        self.p_force = self.p_force * self.force_decay

        return output, hiddens, encoded

# Cell

class LSTM_Block(nn.Module):
    '''
    LSTM_Block - combines Embedding, LSTM, and output layer

    Inputs:

    - `d_vocab int`: vocab size

    - `d_embedding int`: embedding dimension

    - `d_hidden int`: hidden dimension

    - `d_output int`: output dimension

    - `n_layers int`: number of LSTM layers

    - `input_dropout float`: dropout percentage on inputs

    - `lstm_dropout float`: dropout on LSTM layers

    - `bidir bool`: if the LSTM should be bidirectional
    '''
    def __init__(self, d_vocab, d_embedding, d_hidden, d_output, n_layers,
                 input_dropout=0., lstm_dropout=0., bidir=False):
        super().__init__()

        self.embedding = nn.Embedding(d_vocab, d_embedding)
        self.lstm = LSTM(d_embedding, d_hidden, d_output, n_layers,
                                     bidir=bidir, input_dropout=input_dropout, lstm_dropout=lstm_dropout)

        self.head = nn.Linear(d_output, d_vocab)

    def forward(self, x, hiddens=None):
        '''
        Inputs:

        - `x torch.Tensor[(bs, sl, d_embedding) or (sl, bs, d_embedding)]`: input tensor

        - `hiddens Optional[list[(hidden_state, cell_state)]]`: hidden state. If None,
        a new hidden state is initialized

        '''
        x = self.embedding(x)
        encoded, hiddens = self.lstm(x, hiddens)
        output = self.head(encoded)

        return output, hiddens, encoded

# Cell
class Encoder(nn.Module):
    '''
    Base encoder module. All encoders have a
    d_latent attribute which is referenced by other modules
    '''
    def __init__(self, d_latent):
        super().__init__()
        self.d_latent = d_latent

class LSTM_Encoder(Encoder):
    '''
    LSTM_Encoder - LSTM-based encoder

    Inputs:


    - `d_vocab int`: vocab size

    - `d_embedding int`: embedding dimension

    - `d_hidden int`: hidden dimension

    - `n_layers int`: number of LSTM layers

    - `d_latent int`: latent space dimension

    - `input_dropout float`: dropout percentage on inputs

    - `lstm_dropout float`: dropout on LSTM layers

    Generates latent vector from hidden states from the last LSTM layer
    '''
    def __init__(self, d_vocab, d_embedding, d_hidden, n_layers, d_latent,
                 input_dropout=0., lstm_dropout=0.):
        super().__init__(d_latent)
        self.embedding = nn.Embedding(d_vocab, d_embedding)
        self.lstm = LSTM(d_embedding, d_hidden, d_hidden, n_layers,
                         bidir=True, batch_first=True,
                         input_dropout=input_dropout, lstm_dropout=lstm_dropout)
        self.head = nn.Linear(d_hidden*2, d_latent)

    def forward(self, x):
        x = self.embedding(x)
        x, hiddens = self.lstm(x)
        hidden = torch.cat(list(torch.cat(hiddens[-1], -1)), -1) # concatenate hidden/cell states of last layer
        latent = self.head(hidden)
        return latent

# Cell

class MLP_Encoder(Encoder):
    '''
    MLP_Encoder - MLP-based encoder

    Inputs:

    - `d_in int`: number of input dimensions

    - `dims list[int]`: list of layer sizes ie `[1024, 512, 256]`

    - `dropouts list[float]`: list of dropout pobabilities ie `[0.2, 0.2, 0.3]`
    '''
    def __init__(self, d_in, dims, d_latent, dropouts):
        super().__init__(d_latent)

        dims = [d_in]+dims

        acts = [True]*(len(dims)-1)
        bns = [True]*(len(dims)-1)
        layers = [LinearBlock(d_in, d_out, act=a, bn=b, dropout=p)
                 for d_in, d_out, a, b, p in zip(dims[:-1], dims[1:], acts, bns, dropouts)]
        layers.append(nn.Linear(dims[-1], d_latent))

        self.layers = nn.Sequential(*layers)

    def forward(self, x):
        x = self.layers(x)
        return x

# Cell

class Conv_Encoder(Encoder):
    '''
    Conv_Encoder - 1D conv encoder

    Inputs:

    - `d_vocab int`: vocab size

    - `d_embedding int`: embedding dimension

    - `d_latent int`: latent vector dimension

    - `filters list[int]`: filter sizes for conv layers ie `[64, 128, 256]`

    - `kernel_sizes list[int]`: kernel sizes for conv layers ie `[5, 5, 5]`

    - `strides list[int]`: strides for conv layers ie `[2, 2, 2]`

    - `dropouts list[float]`: list of dropout pobabilities ie `[0.2, 0.2, 0.3]`
    '''
    def __init__(self, d_vocab, d_embedding, d_latent, filters, kernel_sizes, strides, dropouts):
        super().__init__(d_latent)

        self.embedding = nn.Embedding(d_vocab, d_embedding)

        filters = [d_embedding] + filters

        convs = [Conv1d(filters[i], filters[i+1], ks=kernel_sizes[i],
                        stride=strides[i], act=True, bn=True, dropout=dropouts[i])
                    for i in range(len(filters)-1)]

        self.convs = nn.Sequential(*convs)
        self.pool = nn.AdaptiveAvgPool1d(1)
        self.head = nn.Linear(filters[-1], d_latent)

    def forward(self, x):
        x = self.embedding(x)
        x = x.permute(0,2,1)
        x = self.convs(x)
        x = self.pool(x).squeeze(-1)
        x = self.head(x)
        return x

# Cell

class MLP(nn.Module):
    '''
    MLP - multi-layer perceptron

    Inputs:

    - `d_in int`: number of input dimensions

    - `dims list[int]`: list of layer sizes ie `[1024, 512, 256]`

    - `d_out int`: number of output dimensions

    - `drops list[float]`: list of dropout pobabilities ie `[0.2, 0.2, 0.3]`

    - `outrange Optional[list[float]]`: squashes the output to be between `outrange[0]`
    and `outrange[1]`
    '''
    def __init__(self, d_in, dims, d_out, drops, outrange=None, bn=True):
        super().__init__()

        dims = [d_in]+dims

        acts = [True]*(len(dims)-1)
        bns = [True]*(len(dims)-1) if bn else [False]*(len(dims)-1)
        layers = [LinearBlock(d_in, d_out, act=a, bn=b, dropout=p)
                 for d_in, d_out, a, b, p in zip(dims[:-1], dims[1:], acts, bns, drops)]
        layers.append(nn.Linear(dims[-1], d_out))

        self.layers = nn.Sequential(*layers)
        self.outrange = outrange

    def forward(self, x):
        x = self.layers(x)

        if self.outrange is not None:
            x = torch.sigmoid(x) * (self.outrange[1]-self.outrange[0]) + self.outrange[0]

        return x