from collections import namedtuple, Sequence as AbstractSequence
from typing import Iterable, List, Optional, Union

import pandas as pd

from .column_values import ColumnValues
from .constants import TIME_STRING_FORMAT
from .frames import Frames
from .session import Session
from .urls import URLS
from .util import extract_primary_camera_name, parse_query_url, URLString

SegmentTuple = namedtuple("SegmentTuple", "drive_id, start_timestamp, end_timestamp")


def listify(p: Union[None, str, Iterable]):
    if p is None:
        return []
    elif isinstance(p, str):
        return [p]
    elif isinstance(p, Iterable):
        return list(p)
    else:
        raise ValueError(f"Unsupported type for listify {type(p)}")


class Sequence:
    """
    A sequence defined by a :code:`drive_id`, :code:`start_timestamp` and an :code:`end_timestamps`. Which corresponds
    to one row from a :class:`.Sequences` object generated by a query.
    """

    def __init__(self, session: Session, drive_id: str, start_timestamp: pd.Timestamp, end_timestamp: pd.Timestamp):
        self.drive_id = drive_id

        if not end_timestamp >= start_timestamp:
            raise ValueError(
                f"Expected end_timestamp to be later than start_timestamp. Got {end_timestamp} and {start_timestamp}"
            )
        self.start_timestamp = start_timestamp
        self.end_timestamp = end_timestamp

        self._session = session
        self._frames = {}  # camera_name => list of frames

    @property
    def duration(self):
        return self.end_timestamp - self.start_timestamp

    def __repr__(self):
        content = f"drive_id='{self.drive_id}', start_timestamp='{self.start_timestamp}', duration='{self.duration}'"

        return f"<{self.__class__.__name__}({content})>"

    def sensors(self) -> dict:
        """Get available sensor names for the particular Sequence

        Examples:
            >>> sequence = sia.query("").all()[0]
            >>> sequence.sensors()
             {'cameras': [
                {'name': 'camera_00', 'is_primary': True},
                {'name': 'camera_back', 'is_primary': False}
             ]}

        Returns:
            The sensor names of this sequence, organized by modality as a dictionary

        """
        url_filled = URLS.sensors.replace("<drive_id>", self.drive_id)
        return self._session.get(url_filled, format="json")

    def column_values(self, names: Union[str, Iterable[str]]) -> ColumnValues:
        """Get the data in columns for this particular Sequence

        Args:
            names: Names of the columns to fetch

        Examples:
            >>> sequence = sia.query("").all()[0]
            >>> values = sequence.column_values(["forward_velocity", "curved_trajectory"])
            >>> values.to_pandas()
                                        drive_id  forward_velocity curved_trajectory
            timestamp
            2018-07-18 11:12:10.500    drive_0001         4.168600         [NO_BEND]
            ...

        Returns:
            The data from the columns wrapped in `ColumnValues` object
        """
        names = listify(names)
        # TODO(robert): Check here for valid columns

        url_filled = URLS.data.replace("<drive_id>", self.drive_id)
        df = self._session.get(
            url_filled,
            params={
                "start_timestamp": self.start_timestamp,
                "end_timestamp": self.end_timestamp,
                "metrics": str(names),
            },
            format="parquet",
        )
        col_values = ColumnValues(df)
        return col_values

    def to_pandas(self) -> pd.Series:
        """Returns a pandas representation of the data"""
        return pd.Series(
            [self.drive_id, self.start_timestamp, self.end_timestamp],
            index=["drive_id", "start_timestamp", "end_timestamp"],
        )

    def to_tuple(self) -> SegmentTuple:
        return SegmentTuple(
            drive_id=self.drive_id, start_timestamp=self.start_timestamp, end_timestamp=self.end_timestamp
        )

    def frames(self, camera_name: Optional[str] = None):
        """Get a Frames representation of the sequences

        Args:
            camera_name: Fetch the frames for a particular camera. If no argument is given, the primary camera
                        is selected
        """
        camera_name = camera_name or extract_primary_camera_name(self.sensors()["cameras"])
        if camera_name not in self._frames:
            frames = Frames.from_sequences(Sequences(self._session, [self]), camera_name)
            self._frames[camera_name] = frames
        return self._frames[camera_name]

    def view(self) -> URLString:
        """Get the URL corresponding to the segment in the frontend"""
        query_url = (
            f"start_timestamp={self.start_timestamp.strftime(TIME_STRING_FORMAT)}&"
            f"end_timestamp={self.end_timestamp.strftime(TIME_STRING_FORMAT)}"
        )

        query_url = parse_query_url(query_url)
        return URLString(f"{self._session.viewer_server}/drives/{self.drive_id}?{query_url}")


class Sequences(AbstractSequence):
    """
    A collection of Sequence objects. Usually the results of a :class:`.Query`. Can be treated like a
    `collections.Sequence` i.e. slicing, getting items etc.
    """

    REQUIRED_COLS = ["drive_id", "start_timestamp", "end_timestamp"]

    def __init__(self, session: Session, segments_list: List[Sequence], query: Optional[str] = None):
        self._session = session
        if not isinstance(segments_list, list):
            raise TypeError(f"Expected input to be a list, got {type(segments_list)}")
        self._check_all_segment(segments_list)
        self._segments = segments_list
        self._frames = {}  # camera_name => list of frames
        self._query = query

    @classmethod
    def from_pandas(cls, session: Session, df: pd.DataFrame, query: Optional[str] = None):
        if not all(c in df for c in cls.REQUIRED_COLS):
            raise ValueError("Expected")
        segments = [Sequence(session, row.drive_id, row.start_timestamp, row.end_timestamp) for row in df.itertuples()]
        return cls(session, segments, query)

    def to_pandas(self) -> pd.DataFrame:
        """Returns a pandas representation of the data"""
        return pd.DataFrame(seg.to_tuple() for seg in self._segments)

    def view(self) -> Optional[URLString]:
        """If a query is available, return a link to the frontend to view the query that generated the Sequences"""
        if self._query:
            query = parse_query_url(self._query)
            return URLString(f"{self._session.viewer_server}/search?query={query}&submit=true")
        return None

    def sensors(self):
        """Get available sensors names for the first Sequence. This makes the assumption that all other sequences
        share the same family of sensors.

        Examples:
            >>> sequences = sia.query("").all()
            >>> sequences.sensors()
             {'cameras': [
                {'name': 'camera_00', 'is_primary': True},
                {'name': 'camera_back', 'is_primary': False}
             ]}

        Returns:
            The sensor names of this sequence, organized by modality as a dictionary

        """
        first_sequence_drive_id = self[0].drive_id
        url_filled = URLS.sensors.replace("<drive_id>", first_sequence_drive_id)
        return self._session.get(url_filled, format="json")

    def frames(self, camera_name: Optional[str] = None):
        """Get a Frames representation of the sequences from a desired camera source"""
        camera_name = camera_name or extract_primary_camera_name(self.sensors()["cameras"])
        if camera_name not in self._frames:
            frames = Frames.from_sequences(self, camera_name)
            self._frames[camera_name] = frames
        return self._frames[camera_name]

    @staticmethod
    def _check_all_segment(data) -> None:
        if isinstance(data, Sequence):
            return
        elif isinstance(data, list):
            if not all(isinstance(d, Sequence) for d in data):
                raise TypeError(f"Expected all elements in the input list to be of type {Sequence.__class__.__name__}")
        else:
            raise TypeError(f"Unsupported type {type(data)}, expected {Sequence.__class__.__name__}")

    def __repr__(self):
        return f"<{self.__class__.__name__} class with {len(self._segments)} segments>,\n{self.to_pandas().__repr__()}"

    def __len__(self) -> int:
        return len(self._segments)

    def __getitem__(self, i: Union[int, slice]) -> Union["Sequences", Sequence]:
        if isinstance(i, int):
            return self._segments[i]
        elif isinstance(i, slice):
            return Sequences(self._session, self._segments[i])
        else:
            raise TypeError(f"{self.__class__.__name__} indices must be integers or slices, not {type(i)}")

    def __add__(self, other: "Sequences"):
        return Sequences(self._session, self._segments + other._segments)
