"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LakeStrategyFactory = exports.LakeImplStrategy = void 0;
const aws_glue_alpha_1 = require("@aws-cdk/aws-glue-alpha");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_events_1 = require("aws-cdk-lib/aws-events");
const aws_events_targets_1 = require("aws-cdk-lib/aws-events-targets");
const aws_lakeformation_1 = require("aws-cdk-lib/aws-lakeformation");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const aws_s3_1 = require("aws-cdk-lib/aws-s3");
const data_lake_bucket_1 = require("./data-lake-bucket");
const kinesis_ops_1 = require("./data-streams/kinesis-ops");
const kinesis_stream_1 = require("./data-streams/kinesis-stream");
const s3_delivery_stream_1 = require("./data-streams/s3-delivery-stream");
const glue_crawler_1 = require("./etl/glue-crawler");
const glue_job_1 = require("./etl/glue-job");
const glue_job_ops_1 = require("./etl/glue-job-ops");
const glue_table_1 = require("./etl/glue-table");
const enums_1 = require("./global/enums");
const utils_1 = require("./utils");
class LakeImplStrategy {
    constructor() {
        this.locationRegistry = [];
        this.stageName = enums_1.Stage.ALPHA;
        this.downloadLocations = {}; //used for the Custom Resource to allow downloading of existing datasets into datalake
        this.dataStreams = {};
    }
    getDataSetBucketName(pipe, dataTier) {
        return dataTier == enums_1.DataTier.RAW ? this.downloadLocations[pipe.name].rawBucketName :
            dataTier == enums_1.DataTier.REFINED ? this.downloadLocations[pipe.name].refinedBucketName :
                dataTier == enums_1.DataTier.TRUSTED ? this.downloadLocations[pipe.name].trustedBucketName : undefined;
    }
    createDataProduct(props) {
        const pipelineStack = new aws_cdk_lib_1.NestedStack(props.stack, `${props.pipe.name}-dataset-stack`); // props.product.accountId == Aws.ACCOUNT_ID ? new NestedStack(props.stack, `${props.pipe.name}-dataset-stack`) : props.stack;
        this.logBucket = props.logBucket;
        this.stageName = props.stage;
        this.securityGroup = props.securityGroup;
        this.vpc = props.vpc;
        this.datalakeAdminRoleArn = props.datalakeAdminRoleArn;
        this.datalakeDbCreatorRoleArn = props.datalakeDbCreatorRoleArn;
        // if data to download into a tier create the download locations
        if (props.pipe.dataSetDropTier) {
            this.downloadLocations[props.pipe.name] = {
                destinationPrefix: props.pipe.destinationPrefix,
                sourceBucketName: props.pipe.s3Properties ? props.pipe.s3Properties.sourceBucketName : undefined,
                sourceKeys: props.pipe.s3Properties ? props.pipe.s3Properties.sourceKeys : undefined,
                rawBucketName: utils_1.buildS3BucketName({
                    name: props.pipe.name,
                    accountId: props.product.accountId,
                    resourceUse: 'raw',
                    stage: this.stageName,
                }),
                refinedBucketName: utils_1.buildS3BucketName({
                    name: props.pipe.name,
                    accountId: props.product.accountId,
                    resourceUse: 'refined',
                    stage: this.stageName,
                }),
                trustedBucketName: utils_1.buildS3BucketName({
                    name: props.pipe.name,
                    accountId: props.product.accountId,
                    resourceUse: 'trusted',
                    stage: this.stageName,
                }),
                destinationBucketName: utils_1.buildS3BucketName({
                    name: props.pipe.name,
                    accountId: props.product.accountId,
                    resourceUse: props.pipe.dataSetDropTier == enums_1.DataTier.RAW ? 'raw' : props.pipe.dataSetDropTier == enums_1.DataTier.REFINED ? 'refined' : 'trusted',
                    stage: this.stageName,
                }),
            };
        }
        this.createBuckets(pipelineStack, props.pipe, props.product);
        const bucketName = this.getDataSetBucketName(props.pipe, props.pipe.dataSetDropTier);
        this.addPipeline(pipelineStack, props.pipe, props.product, bucketName);
        // find the correct metadata catalog account
        if (this.lakeKind() === enums_1.LakeKind.CENTRAL_CATALOG || this.lakeKind() === enums_1.LakeKind.DATA_PRODUCT_AND_CATALOG) {
            this.createCrawler(pipelineStack, props.pipe, props.database, bucketName);
        }
    }
    createCrawler(stack, pipe, database, bucketName) {
        if (pipe.table)
            return;
        // only create a crawler for the drop location of the data in the data product of the pipeline
        const crawler = new glue_crawler_1.GlueCrawler(stack, `data-lake-crawler-${pipe.name}`, {
            name: utils_1.buildGlueCrawlerName({
                stage: this.stageName,
                resourceUse: 'crawler',
                name: pipe.name,
            }),
            databaseName: database.databaseName,
            bucketName: bucketName,
            bucketPrefix: pipe.destinationPrefix,
            roleName: utils_1.buildRoleName({
                stage: this.stageName,
                resourceUse: 'crawler-role',
                name: pipe.name,
            }),
        });
        this.locationRegistry.forEach(r => {
            crawler.node.addDependency(r);
        });
    }
    createGlueTable(stack, pipeline, product, bucketName) {
        if (!pipeline.table)
            return;
        const table = new glue_table_1.GlueTable(stack, `${pipeline.name}-table`, {
            catalogId: pipeline.table.catalogId,
            columns: pipeline.table.columns,
            databaseName: product.databaseName,
            description: pipeline.table.description,
            inputFormat: pipeline.table.inputFormat,
            outputFormat: pipeline.table.outputFormat,
            parameters: pipeline.table.parameters,
            partitionKeys: pipeline.table.partitionKeys,
            s3Location: `s3://${bucketName}/${pipeline.destinationPrefix}`,
            serdeParameters: pipeline.table.serdeParameters,
            serializationLibrary: pipeline.table.serializationLibrary,
            tableName: pipeline.table.tableName,
        });
        table.node.addDependency(product.databaseName);
    }
    // this is a jumbled mess clean up once refecto
    createPipelineResources(stack, pipeline, dataProduct, bucketName) {
        switch (pipeline.type) {
            case enums_1.DataPipelineType.S3: {
                break;
            }
            case enums_1.DataPipelineType.STREAM: {
                this.addDataStream(stack, pipeline, bucketName);
                break;
            }
            case enums_1.DataPipelineType.JDBC: {
                this.createJDBCConnection(stack, pipeline);
                break;
            }
        }
        // rethink this whole section
        if (pipeline.job) {
            const jobScript = utils_1.packageAsset(stack, `${pipeline.name}Script`, pipeline.job.jobScript);
            pipeline.job.jobArgs['--TempDir'] = `s3://${this.logBucket.bucketName}/temp/`;
            pipeline.job.jobArgs['--spark-event-logs-path'] = `s3://${this.logBucket.bucketName}/logs/`;
            let s3Location = this.getDataSetBucketName(pipeline, pipeline.job.destinationLocation);
            if (pipeline.job.destinationLocation && s3Location) {
                pipeline.job.jobArgs['--DESTINATION_BUCKET'] = s3Location;
                const job = new glue_job_1.GlueJob(stack, `${pipeline.name}-etl-job`, {
                    deploymentBucket: jobScript.bucket,
                    jobScript: utils_1.toS3Path(jobScript),
                    name: pipeline.job.name,
                    workerType: pipeline.job.workerType,
                    description: pipeline.job.description,
                    glueVersion: pipeline.job.glueVersion,
                    jobArgs: pipeline.job.jobArgs,
                    maxCapacity: pipeline.job.maxCapacity,
                    maxConcurrentRuns: pipeline.job.maxConcurrentRuns,
                    maxRetries: pipeline.job.maxRetries,
                    numberOfWorkers: pipeline.job.numberOfWorkers,
                    roleName: pipeline.job.roleName,
                    timeout: pipeline.job.timeout,
                    jobType: pipeline.job.jobType,
                    readAccessBuckets: [
                        this.logBucket,
                    ],
                    writeAccessBuckets: [
                        this.logBucket,
                        aws_s3_1.Bucket.fromBucketName(stack, 'raw-bucket-role', s3Location),
                    ],
                });
                new glue_job_ops_1.GlueJobOps(stack, `${pipeline.name}-etl-job-ops`, {
                    job: job,
                });
                if (pipeline.streamProperties) {
                    this.dataStreams[pipeline.name].stream.grantRead(job.role);
                }
                new aws_lakeformation_1.CfnPermissions(stack, `${pipeline.name}-create-table-perm`, {
                    dataLakePrincipal: {
                        dataLakePrincipalIdentifier: job.role.roleArn,
                    },
                    resource: {
                        databaseResource: {
                            name: dataProduct.databaseName,
                        },
                    },
                    permissions: [
                        enums_1.Permissions.ALTER,
                        enums_1.Permissions.CREATE_TABLE,
                        enums_1.Permissions.DESCRIBE,
                    ],
                });
                if (pipeline.table) {
                    new aws_lakeformation_1.CfnPermissions(stack, `${pipeline.name}-access-table-perm`, {
                        dataLakePrincipal: {
                            dataLakePrincipalIdentifier: job.role.roleArn,
                        },
                        resource: {
                            tableResource: {
                                databaseName: dataProduct.databaseName,
                                name: pipeline.table.tableName,
                            },
                        },
                        permissions: [
                            enums_1.Permissions.SELECT,
                            enums_1.Permissions.DESCRIBE,
                        ],
                    });
                }
            }
        }
    }
    addDataStream(stack, pipeline, bucketName) {
        const schemaName = pipeline.name;
        const dataStreamStack = new aws_cdk_lib_1.NestedStack(stack, `${schemaName}-datastream-stack`);
        if (!pipeline.streamProperties) {
            throw Error("Cannot create a stream pipeline without 'streamProperties'");
        }
        this.dataStreams[pipeline.name] = new kinesis_stream_1.KinesisStream(dataStreamStack, 'DataStream', {
            shardCount: 1,
            streamName: pipeline.streamProperties.streamName,
        });
        const deliveryStream = new s3_delivery_stream_1.S3DeliveryStream(dataStreamStack, 'deliveryStream', {
            compression: s3_delivery_stream_1.CompressionType.UNCOMPRESSED,
            kinesisStream: this.dataStreams[pipeline.name].stream,
            s3Bucket: aws_s3_1.Bucket.fromBucketName(stack, 'get-bucket-for-kinesis', bucketName),
            s3Prefix: pipeline.destinationPrefix,
        });
        new kinesis_ops_1.KinesisOps(dataStreamStack, 'kinesis-ops', {
            stream: this.dataStreams[pipeline.name],
            deliveryStream: deliveryStream,
        });
        if (pipeline.streamProperties.lambdaDataGenerator) {
            const dataGeneratorFunction = new aws_lambda_1.Function(dataStreamStack, 'data-generator-function', {
                code: pipeline.streamProperties.lambdaDataGenerator.code,
                handler: pipeline.streamProperties.lambdaDataGenerator.handler,
                timeout: pipeline.streamProperties.lambdaDataGenerator.timeout,
                runtime: pipeline.streamProperties.lambdaDataGenerator.runtime,
                functionName: pipeline.streamProperties.lambdaDataGenerator.functionName,
                environment: {
                    KINESIS_STREAM: this.dataStreams[pipeline.name].stream.streamName,
                },
            });
            this.dataStreams[pipeline.name].stream.grantWrite(dataGeneratorFunction);
            const rule = new aws_events_1.Rule(stack, 'Rule', {
                schedule: pipeline.streamProperties.lambdaDataGenerator.schedule,
                ruleName: pipeline.streamProperties.lambdaDataGenerator.ruleName,
            });
            rule.addTarget(new aws_events_targets_1.LambdaFunction(dataGeneratorFunction));
        }
        return this.dataStreams[pipeline.name];
    }
    createJDBCConnection(stack, pipeline) {
        if (this.vpc && this.securityGroup) {
            new aws_glue_alpha_1.Connection(stack, `${pipeline.name}-glue-connection`, {
                type: aws_glue_alpha_1.ConnectionType.JDBC,
                connectionName: `${pipeline.name}-jdbc`,
                description: `JDBC connection for glue to use on pipeline ${pipeline.name}`,
                subnet: this.vpc.isolatedSubnets[0],
                securityGroups: [this.securityGroup],
                properties: {
                    JDBC_CONNECTION_URL: pipeline.jdbcProperties.jdbc,
                    USERNAME: pipeline.jdbcProperties.username,
                    PASSWORD: pipeline.jdbcProperties.password,
                },
            });
        }
        else {
            throw new Error('VPC required to create a JDBC pipeline.');
        }
    }
    createBuckets(stack, pipe, product) {
        /// This is confusing. Find a way to simplify
        const dataCatalogAccountId = product.dataCatalogAccountId ?
            product.dataCatalogAccountId : product.accountId;
        const crossAccount = product.dataCatalogAccountId ?
            product.dataCatalogAccountId != product.accountId ? true : false : false;
        pipe.tiers.forEach(r => {
            if (this.lakeKind() === enums_1.LakeKind.DATA_PRODUCT || this.lakeKind() === enums_1.LakeKind.DATA_PRODUCT_AND_CATALOG) {
                const bucketName = this.getDataSetBucketName(pipe, r);
                new data_lake_bucket_1.DataLakeBucket(stack, `s3-${r}-bucket-${pipe.name}`, {
                    bucketName: bucketName,
                    dataCatalogAccountId: dataCatalogAccountId,
                    logBucket: this.logBucket,
                    crossAccount: crossAccount,
                    s3Properties: product.s3BucketProps,
                }).bucket;
            }
            if (this.lakeKind() === enums_1.LakeKind.CENTRAL_CATALOG || this.lakeKind() === enums_1.LakeKind.DATA_PRODUCT_AND_CATALOG) {
                if (this.datalakeDbCreatorRoleArn == undefined)
                    throw new Error('Cannot have datalake without Data Lake DB Creator role defined.');
                const name = this.getDataSetBucketName(pipe, r).replace(/\W/g, '');
                const lfResource = this.registerDataLakeLocation(stack, this.datalakeDbCreatorRoleArn, name);
                this.locationRegistry.push(lfResource);
                if (this.datalakeAdminRoleArn) {
                    this.createDataLocationAccessPermission(stack, `${name}-admin`, this.datalakeAdminRoleArn, name, lfResource);
                }
            }
        });
    }
    registerDataLakeLocation(stack, datalakeDbCreatorRoleArn, bucketName) {
        const dlResource = new aws_lakeformation_1.CfnResource(stack, `lf-resource-${bucketName}`, {
            resourceArn: `arn:aws:s3:::${bucketName}`,
            useServiceLinkedRole: false,
            roleArn: datalakeDbCreatorRoleArn,
        });
        this.createDataLocationAccessPermission(stack, `${bucketName}-creator`, datalakeDbCreatorRoleArn, bucketName, dlResource);
        return dlResource;
    }
    createDataLocationAccessPermission(stack, name, roleArn, bucketName, resource) {
        const perm = new aws_lakeformation_1.CfnPermissions(stack, `datalake-creator-perm-${name}`, {
            dataLakePrincipal: {
                dataLakePrincipalIdentifier: roleArn,
            },
            resource: {
                dataLocationResource: {
                    s3Resource: `arn:aws:s3:::${bucketName}`,
                },
            },
            permissions: [
                enums_1.Permissions.DATA_LOCATION_ACCESS,
            ],
        });
        perm.node.addDependency(resource);
        return perm;
    }
}
exports.LakeImplStrategy = LakeImplStrategy;
class DataProductStrategy extends LakeImplStrategy {
    lakeKind() {
        return enums_1.LakeKind.DATA_PRODUCT;
    }
    addPipeline(stack, pipeline, dataProduct, bucketName) {
        this.createPipelineResources(stack, pipeline, dataProduct, bucketName);
    }
}
class CentralCatalogStrategy extends LakeImplStrategy {
    lakeKind() {
        return enums_1.LakeKind.CENTRAL_CATALOG;
    }
    addPipeline(stack, pipeline, dataProduct, bucketName) {
        if (pipeline.table) {
            this.createGlueTable(stack, pipeline, dataProduct, bucketName);
        }
    }
}
class ConsumerStrategy extends LakeImplStrategy {
    lakeKind() {
        return enums_1.LakeKind.CONSUMER;
    }
    // eslint-disable-next-line @typescript-eslint/ban-ts-comment
    // @ts-ignore
    addPipeline(stack, pipeline, dataProduct, bucketName) {
        return;
    }
}
class DataProductAndCatalogStrategy extends LakeImplStrategy {
    lakeKind() {
        return enums_1.LakeKind.DATA_PRODUCT_AND_CATALOG;
    }
    addPipeline(stack, pipeline, dataProduct, bucketName) {
        this.createPipelineResources(stack, pipeline, dataProduct, bucketName);
        if (pipeline.table) {
            this.createGlueTable(stack, pipeline, dataProduct, bucketName);
        }
    }
}
class LakeStrategyFactory {
    static getLakeStrategy(lakeKind) {
        return LakeStrategyFactory.strategies[lakeKind];
    }
}
exports.LakeStrategyFactory = LakeStrategyFactory;
LakeStrategyFactory.strategies = {
    [enums_1.LakeKind.DATA_PRODUCT]: new DataProductStrategy(),
    [enums_1.LakeKind.CENTRAL_CATALOG]: new CentralCatalogStrategy(),
    [enums_1.LakeKind.CONSUMER]: new ConsumerStrategy(),
    [enums_1.LakeKind.DATA_PRODUCT_AND_CATALOG]: new DataProductAndCatalogStrategy(),
};
//# sourceMappingURL=data:application/json;base64,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