import * as glue from '@aws-cdk/aws-glue-alpha';
import * as athena from 'aws-cdk-lib/aws-athena';
import * as ec2 from 'aws-cdk-lib/aws-ec2';
import * as iam from 'aws-cdk-lib/aws-iam';
import { IRole } from 'aws-cdk-lib/aws-iam';
import * as s3 from 'aws-cdk-lib/aws-s3';
import { Bucket } from 'aws-cdk-lib/aws-s3';
import { Construct } from 'constructs';
import { DataProduct } from './data-product';
import { DataTier, LakeKind, Stage } from './global/enums';
export interface CrossAccountProperties {
    readonly consumerAccountIds: string[];
    readonly dataCatalogOwnerAccountId: string;
}
export interface DataLakeProperties {
    /**
     * The name of the DataLake.
     *
     */
    readonly name: string;
    /**
     * The Stage the DataLake will be deployed.
     *
     */
    readonly stageName: Stage;
    /**
     * The List of DataProducts for this account
     *
     * @default - No data products
     */
    readonly dataProducts?: DataProduct[];
    /**
     * The Type of DataLake this instance is. This can be a DATA_PRODUCT only, CENTRAL_CATALOG, CONSUMER, or DATA_PRODUCT_AND_CATALOG type.
     */
    readonly lakeKind: LakeKind;
    /**
     * VPC for Glue jobs
     *
     * @default - No vpc
     * @description - The VPC that will be used if the Glue job needs access to resources within the account or internet access
     */
    readonly vpc?: ec2.Vpc;
    /**
     * List of Lake Formation TBAC policy tags.
     *
     * @default - No tags
     * @description - Define the tag taxonomy needed for the DataLake
     * @see https://docs.aws.amazon.com/lake-formation/latest/dg/TBAC-section.html
     */
    readonly policyTags?: {
        [name: string]: string;
    };
    /**
     * Cross account AWS account IDs
     *
     * @default - No cross account ids
     * @description - The cross account ids needed for setting up the Glue resource policy
     * @see https://aws.amazon.com/premiumsupport/knowledge-center/glue-data-catalog-cross-account-access/
     */
    readonly crossAccountAccess?: CrossAccountProperties;
    /**
     * Security group to attach to Glue jobs
     *
     * @default - No security group
     * @description - Security Group that will be used to allow port access in the VPC
     * @see https://docs.aws.amazon.com/glue/latest/dg/setup-vpc-for-glue-access.html
     */
    readonly glueSecurityGroup?: ec2.SecurityGroup;
    /**
     * Data Lake Admin role
     *
     * @default - Admin role created based on best practices
     * @description - IAM Role for DataLake admin access
     * @see https://docs.aws.amazon.com/lake-formation/latest/dg/permissions-reference.html
     */
    readonly datalakeAdminRole?: iam.Role;
    /**
     * Data Lake Database Creator role
     *
     * @default - Database creator role created based on best practices
     * @description - IAM Role for DataLake database creator access
     * @see https://docs.aws.amazon.com/lake-formation/latest/dg/permissions-reference.html
     */
    readonly datalakeCreatorRole?: iam.Role;
    readonly logBucketProps?: s3.BucketProps;
    /**
    * Create default Athena workgroup for querying data lake resources
    *
    * @default - false
    */
    readonly createAthenaWorkgroup?: Boolean;
}
export interface DataTierBucketProps {
    readonly lakeType: LakeKind;
    readonly pipelineName: string;
    readonly bucketName: string | undefined;
    readonly dataCatalogAccountId: string;
    readonly logBucket: Bucket;
    readonly crossAccount: boolean;
    readonly s3BucketProps: s3.BucketProps | undefined;
    readonly datalakeAdminRole: IRole;
    readonly datalakeDbCreatorRole: IRole;
    readonly tier: DataTier;
}
/**
 * A CDK construct to create a DataLake.
 */
export declare class DataLake extends Construct {
    readonly databases: {
        [name: string]: glue.Database;
    };
    readonly datalakeAdminRole: iam.IRole;
    readonly datalakeDbCreatorRole: iam.IRole;
    readonly logBucket: s3.Bucket;
    readonly stageName: Stage;
    readonly vpc?: ec2.Vpc;
    readonly athenaWorkgroup?: athena.CfnWorkGroup;
    readonly lakeKind: LakeKind;
    private readonly glueSecurityGroup?;
    private readonly crossAccountAccess?;
    private readonly logBucketProps;
    private readonly dataLakeStrategy;
    constructor(scope: Construct, id: string, props: DataLakeProperties);
    createDownloaderCustomResource(stageName: string): void;
    private createDatabase;
    private createPolicyTagsCustomResource;
    createCrossAccountGlueCatalogResourcePolicy(consumerAccountIds: string[], dataCatalogOwnerAccountId: string): void;
}
