"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DataLake = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const glue = require("@aws-cdk/aws-glue-alpha");
const aws_lambda_python_alpha_1 = require("@aws-cdk/aws-lambda-python-alpha");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const athena = require("aws-cdk-lib/aws-athena");
const ec2 = require("aws-cdk-lib/aws-ec2");
const iam = require("aws-cdk-lib/aws-iam");
const lf = require("aws-cdk-lib/aws-lakeformation");
const lambda = require("aws-cdk-lib/aws-lambda");
const logs = require("aws-cdk-lib/aws-logs");
const s3 = require("aws-cdk-lib/aws-s3");
const cr = require("aws-cdk-lib/custom-resources");
const constructs_1 = require("constructs");
const data_lake_strategy_1 = require("./data-lake-strategy");
const enums_1 = require("./global/enums");
const data_lake_admin_1 = require("./personas/data-lake-admin");
const data_lake_creator_1 = require("./personas/data-lake-creator");
const utils_1 = require("./utils");
/**
 * A CDK construct to create a DataLake.
 */
class DataLake extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.databases = {};
        this.stageName = props.stageName;
        this.crossAccountAccess = props.crossAccountAccess ? props.crossAccountAccess : undefined;
        this.vpc = props.vpc ? props.vpc : undefined;
        this.lakeKind = props.lakeKind;
        if (props.logBucketProps) {
            this.logBucketProps = props.logBucketProps;
        }
        else {
            this.logBucketProps = {
                lifecycleRules: [
                    {
                        expiration: aws_cdk_lib_1.Duration.days(30),
                    },
                ],
                removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
                autoDeleteObjects: true,
            };
        }
        if (this.vpc) {
            const securityGroupName = utils_1.buildUniqueName({
                name: 'glue',
                resourceUse: 'datalake',
                stage: this.stageName,
            }, 80);
            this.glueSecurityGroup = new ec2.SecurityGroup(this, 'glue-sg', {
                description: 'Glue self referential allow in out',
                vpc: this.vpc,
                securityGroupName: securityGroupName,
            });
            this.glueSecurityGroup.connections.allowFrom(this.glueSecurityGroup, ec2.Port.allTcp());
            this.glueSecurityGroup.connections.allowTo(this.glueSecurityGroup, ec2.Port.allTcp());
            new aws_cdk_lib_1.CfnOutput(this, 'GlueSecurityGroupName', { value: securityGroupName });
        }
        this.logBucket = new s3.Bucket(this, 'datalake-log-bucket', {
            bucketName: utils_1.buildS3BucketName({
                stage: props.stageName,
                resourceUse: 'log-bucket',
                name: props.name,
            }),
            ...this.logBucketProps,
        });
        new aws_cdk_lib_1.CfnOutput(this, 'DataLakeLogBucket', { value: this.logBucket.bucketName });
        if (props.datalakeAdminRole) {
            this.datalakeAdminRole = props.datalakeAdminRole;
        }
        else {
            this.datalakeAdminRole = new data_lake_admin_1.DataLakeAdministrator(this, `${props.name}-datalake-admin-role`, {
                name: utils_1.buildUniqueName({
                    name: props.name,
                    resourceUse: 'datalake-admin',
                    stage: this.stageName,
                }, 60),
            }).role;
        }
        if (props.datalakeCreatorRole) {
            this.datalakeDbCreatorRole = props.datalakeCreatorRole;
        }
        else {
            this.datalakeDbCreatorRole = new data_lake_creator_1.DataLakeCreator(this, `${props.name}-datalake-creator-role`, {
                name: utils_1.buildUniqueName({
                    name: props.name,
                    resourceUse: 'datalake-creator',
                    stage: this.stageName,
                }, 60),
            }).role;
        }
        const lfAdminRole = new lf.CfnDataLakeSettings(this, 'lf-datalake-role-admin-settings', {
            admins: [{
                    dataLakePrincipalIdentifier: this.datalakeAdminRole.roleArn,
                }],
        });
        lfAdminRole.node.addDependency(this.datalakeAdminRole);
        new aws_cdk_lib_1.CfnOutput(this, 'DataLakeAdminRole', { value: this.datalakeAdminRole.roleName });
        if (this.crossAccountAccess) {
            this.createCrossAccountGlueCatalogResourcePolicy(this.crossAccountAccess.consumerAccountIds, this.crossAccountAccess.dataCatalogOwnerAccountId);
        }
        if (props.createAthenaWorkgroup) {
            this.athenaWorkgroup = new athena.CfnWorkGroup(this, 'workgroup', {
                name: utils_1.buildUniqueName({
                    name: props.name,
                    resourceUse: 'workgroup',
                    stage: this.stageName,
                }, 60),
                description: 'Default Data Lake Workgroup',
                state: 'ENABLED',
                recursiveDeleteOption: true,
                workGroupConfiguration: {
                    enforceWorkGroupConfiguration: true,
                    resultConfiguration: {
                        outputLocation: `s3://${this.logBucket.bucketName}/results/`,
                    },
                    engineVersion: {
                        selectedEngineVersion: 'Athena engine version 2',
                        effectiveEngineVersion: 'Athena engine version 2',
                    },
                },
            });
            new aws_cdk_lib_1.CfnOutput(this, 'DataLakeAthenaWorkgroup', { value: this.athenaWorkgroup.name });
        }
        if (props.policyTags) {
            this.createPolicyTagsCustomResource(props.policyTags);
        }
        this.dataLakeStrategy = data_lake_strategy_1.LakeStrategyFactory.getLakeStrategy(props.lakeKind);
        if (props.dataProducts && props.dataProducts.length > 0) {
            props.dataProducts.forEach((product) => {
                if (this.databases[product.databaseName] == undefined) {
                    this.databases[product.databaseName] = this.createDatabase(product.databaseName);
                }
                product.pipelines.forEach((pipe) => {
                    this.dataLakeStrategy.createDataProduct({
                        stack: aws_cdk_lib_1.Stack.of(this),
                        pipe: pipe,
                        product: product,
                        database: this.databases[product.databaseName],
                        logBucket: this.logBucket,
                        stage: this.stageName,
                        datalakeAdminRoleArn: this.datalakeAdminRole.roleArn,
                        datalakeDbCreatorRoleArn: this.datalakeDbCreatorRole.roleArn,
                    });
                });
            });
        }
    }
    createDownloaderCustomResource(stageName) {
        // download the data sets with the custom resource after successfull creation of resource
        const onEvent = new aws_lambda_python_alpha_1.PythonFunction(this, 'DataloaderHandler', {
            runtime: lambda.Runtime.PYTHON_3_7,
            entry: path.join(__dirname, '../lambda/download-data'),
            timeout: aws_cdk_lib_1.Duration.minutes(15),
            functionName: utils_1.buildLambdaFunctionName({
                name: 'load-data',
                resourceUse: 'cr',
                stage: stageName,
            }),
        });
        // create readable and writable buckets for the datasets and set the appropriate S3 access
        onEvent.addToRolePolicy(new iam.PolicyStatement({
            actions: ['s3:*'],
            resources: ['*'],
        }));
        const dataLoadProvider = new cr.Provider(this, 'DataloaderProvider', {
            onEventHandler: onEvent,
            logRetention: logs.RetentionDays.ONE_DAY,
        });
        // CR to download the static datasets form the dataSets var passed in.
        new aws_cdk_lib_1.CustomResource(this, 'LoadDatalakeCustomResource', {
            serviceToken: dataLoadProvider.serviceToken,
            properties: {
                dataSets: this.dataLakeStrategy.downloadLocations,
                stackName: aws_cdk_lib_1.Stack.name,
                regionName: aws_cdk_lib_1.Aws.REGION,
            },
        });
    }
    createDatabase(databaseName) {
        const db = new glue.Database(this, `${databaseName}-database`, {
            databaseName: `${databaseName}`,
        });
        const dbPerm = new lf.CfnPermissions(this, `${databaseName}-lf-db-creator-permission`, {
            dataLakePrincipal: {
                dataLakePrincipalIdentifier: this.datalakeDbCreatorRole.roleArn,
            },
            resource: {
                databaseResource: {
                    name: databaseName,
                },
            },
            permissions: [
                enums_1.Permissions.ALTER,
                enums_1.Permissions.CREATE_TABLE,
                enums_1.Permissions.DROP,
            ],
        });
        dbPerm.node.addDependency(db);
        return db;
    }
    createPolicyTagsCustomResource(policyTags) {
        const onEvent = new aws_lambda_python_alpha_1.PythonFunction(this, 'create-policy-tags-handler', {
            runtime: lambda.Runtime.PYTHON_3_7,
            entry: path.join(__dirname, '../lambda/create-tags-handler'),
            role: this.datalakeAdminRole,
            functionName: utils_1.buildLambdaFunctionName({
                name: 'create-tags',
                resourceUse: 'cr',
                stage: this.stageName,
            }),
            timeout: aws_cdk_lib_1.Duration.minutes(15),
        });
        onEvent.node.addDependency(this.datalakeAdminRole);
        const myProvider = new cr.Provider(this, 'policy-tags-provider', {
            onEventHandler: onEvent,
            logRetention: logs.RetentionDays.ONE_DAY,
        });
        const outputs = new aws_cdk_lib_1.CustomResource(this, 'tag-creation-custom-resource', {
            serviceToken: myProvider.serviceToken,
            properties: {
                policyTags: policyTags,
                stackName: aws_cdk_lib_1.Stack.name,
                regionName: aws_cdk_lib_1.Aws.REGION,
                catalogId: aws_cdk_lib_1.Aws.ACCOUNT_ID,
            },
        });
        outputs.node.addDependency(this.datalakeAdminRole);
    }
    createCrossAccountGlueCatalogResourcePolicy(consumerAccountIds, dataCatalogOwnerAccountId) {
        const onCatalogEvent = new aws_lambda_python_alpha_1.PythonFunction(this, 'enable-hybrid-catalog-handler', {
            runtime: lambda.Runtime.PYTHON_3_7,
            entry: path.join(__dirname, '../lambda/enable-hybrid-catalog'),
            role: this.datalakeAdminRole,
            timeout: aws_cdk_lib_1.Duration.minutes(1),
            functionName: utils_1.buildLambdaFunctionName({
                name: 'create-catalog',
                resourceUse: 'cr',
                stage: this.stageName,
            }),
        });
        const catalogProvider = new cr.Provider(this, 'hybrid-catalog-provider', {
            onEventHandler: onCatalogEvent,
            logRetention: logs.RetentionDays.ONE_DAY,
        });
        new aws_cdk_lib_1.CustomResource(this, 'hybrid-catalog-custom-resource', {
            serviceToken: catalogProvider.serviceToken,
            properties: {
                stackName: aws_cdk_lib_1.Stack.name,
                regionName: aws_cdk_lib_1.Aws.REGION,
                consumerAccountIds: consumerAccountIds,
                producerAccountId: dataCatalogOwnerAccountId,
            },
        });
    }
}
exports.DataLake = DataLake;
_a = JSII_RTTI_SYMBOL_1;
DataLake[_a] = { fqn: "@randyridgley/cdk-datalake-constructs.DataLake", version: "0.0.52" };
//# sourceMappingURL=data:application/json;base64,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