"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.KinesisOps = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const cloudwatch = require("aws-cdk-lib/aws-cloudwatch");
const constructs_1 = require("constructs");
class KinesisOps extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.stream = props.stream;
        this.streamName = props.stream.stream.streamName;
        this.deliveryStream = props.deliveryStream;
        this.dashboard = new cloudwatch.Dashboard(this, 'dashboard', {
            dashboardName: `Kinesis_${this.streamName}`,
        });
        this.inputStreamIteratorAgeCriticalAlarm = new cloudwatch.Alarm(this, 'inputStream-iterator-age-critical-alarm', {
            alarmName: `${this.streamName} inputStream IteratorAge Long`,
            alarmDescription: 'Alarms if maximum iterator age of inputStream is more than 10 minute',
            metric: this.stream.metricGetRecordsIteratorAgeMilliseconds({
                period: aws_cdk_lib_1.Duration.minutes(5),
            }),
            threshold: 600000,
            comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            evaluationPeriods: 12,
            ...(props.inputStreamIteratorAgeCritical || {}),
        });
        this.inputStreamIteratorAgeWarningAlarm = new cloudwatch.Alarm(this, 'inputStream-iterator-age-warning-alarm', {
            alarmName: `${this.streamName} inputStream IteratorAge Long Warning`,
            alarmDescription: 'Alarms if maximum iterator age of inputStream is more than 5 minute',
            metric: this.stream.metricGetRecordsIteratorAgeMilliseconds({
                period: aws_cdk_lib_1.Duration.minutes(5),
            }),
            threshold: 30000,
            comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            evaluationPeriods: 12,
            ...(props.inputStreamIteratorAgeWarning || {}),
        });
        this.inputStreamReadThroughputWarningAlarm = new cloudwatch.Alarm(this, 'inputStream-read-throughput-warning-alarm', {
            alarmName: `${this.streamName} inputStream ReadThroughput Exceed Warning`,
            alarmDescription: 'Alarms if read provisioned throughput of inputStream is exceeded for least 2 hours',
            metric: this.stream.metricReadProvisionedThroughputExceeded({
                period: aws_cdk_lib_1.Duration.minutes(10),
            }),
            threshold: 0.15,
            comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_THRESHOLD,
            evaluationPeriods: 12,
            ...(props.inputStreamReadThroughputWarning || {}),
        });
        this.inputStreamWriteThroughputWarningAlarm = new cloudwatch.Alarm(this, 'inputStream-write-throughput-warning-alarm', {
            alarmName: `${this.streamName} inputStream WriteThroughput Exceed Warning`,
            alarmDescription: 'Alarms if write provisioned throughput of inputStream is exceeded for least 12 hours',
            metric: this.stream.metricWriteProvisionedThroughputExceeded({
                period: aws_cdk_lib_1.Duration.minutes(60),
            }),
            threshold: 0.15,
            comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_THRESHOLD,
            evaluationPeriods: 12,
            ...(props.inputStreamWriteThroughputWarning || {}),
        });
        this.inputStreamGetRecordsWarningAlarm = new cloudwatch.Alarm(this, 'inputStream-get-records-warning-alarm', {
            alarmName: `${this.streamName} inputStream GetRecords Success Low Warning`,
            alarmDescription: 'Alarms if GetRecords of inputStream not very successful for least 30 minutes',
            metric: this.stream.metricGetRecordsSuccess({
                period: aws_cdk_lib_1.Duration.minutes(5),
            }),
            threshold: 0.9,
            comparisonOperator: cloudwatch.ComparisonOperator.LESS_THAN_THRESHOLD,
            evaluationPeriods: 6,
            ...(props.inputStreamGetRecordsWarning || {}),
        });
        this.inputStreamPutRecordsWarningAlarm = new cloudwatch.Alarm(this, 'inputStream-put-records-warning-alarm', {
            alarmName: `${this.streamName} inputStream PutRecords Success Low Warning`,
            alarmDescription: 'Alarms if PutRecords of inputStream not very successful for least 12 hours',
            metric: this.stream.metricPutRecordsSuccess({
                period: aws_cdk_lib_1.Duration.minutes(60),
            }),
            threshold: 0.9,
            comparisonOperator: cloudwatch.ComparisonOperator.LESS_THAN_THRESHOLD,
            evaluationPeriods: 12,
            ...(props.inputStreamPutRecordsWarning || {}),
        });
        this.firehoseDeliveryToS3WarningAlarm = new cloudwatch.Alarm(this, 'deliveryStream-delivery-to-s3-warning-alarm', {
            alarmName: `${this.streamName} Firehose DeliveryToS3 Failure Warning`,
            alarmDescription: 'Alarms if firehose DeliveryToS3 failed for atleast 60 minutes',
            metric: this.deliveryStream.metricDeliveryToS3Success({
                statistic: cloudwatch.Statistic.AVERAGE,
                period: aws_cdk_lib_1.Duration.minutes(5),
            }),
            threshold: 1,
            comparisonOperator: cloudwatch.ComparisonOperator.LESS_THAN_THRESHOLD,
            evaluationPeriods: 12,
            treatMissingData: cloudwatch.TreatMissingData.NOT_BREACHING,
            ...(props.firehoseDeliveryToS3Warning || {}),
        });
        this.firehoseDeliveryToS3CriticalAlarm = new cloudwatch.Alarm(this, 'deliveryStream-delivery-to-s3-critical-alarm', {
            alarmName: `${this.streamName} Firehose DeliveryToS3 Failure Critical`,
            alarmDescription: 'Alarms if firehose DeliveryToS3 failed for atleast 24 hours',
            metric: this.deliveryStream.metricDeliveryToS3Success({
                statistic: cloudwatch.Statistic.AVERAGE,
                period: aws_cdk_lib_1.Duration.hours(1),
            }),
            threshold: 1,
            comparisonOperator: cloudwatch.ComparisonOperator.LESS_THAN_THRESHOLD,
            evaluationPeriods: 24,
            treatMissingData: cloudwatch.TreatMissingData.NOT_BREACHING,
            ...(props.firehoseDeliveryToS3Critical || {}),
        });
        this.alarmsSev2 = [
            this.inputStreamIteratorAgeCriticalAlarm,
            this.firehoseDeliveryToS3CriticalAlarm,
        ];
        this.alarmsSev3 = [
            this.inputStreamIteratorAgeWarningAlarm,
            this.inputStreamReadThroughputWarningAlarm,
            this.inputStreamWriteThroughputWarningAlarm,
            this.inputStreamGetRecordsWarningAlarm,
            this.inputStreamPutRecordsWarningAlarm,
            this.firehoseDeliveryToS3WarningAlarm,
        ];
        this.setupDashboard();
    }
    alarmWidgets(alarms, severity) {
        return alarms.map(alarm => new cloudwatch.AlarmWidget({
            title: `${alarm.alarmName} - Sev ${severity}`,
            alarm,
        }));
    }
    addWidgets(widgets) {
        for (let i = 0; i < widgets.length; i += 4) {
            this.dashboard.addWidgets(...widgets.slice(i, i + 4));
        }
    }
    setupDashboard() {
        const widgets = [
            new cloudwatch.GraphWidget({
                title: 'Kinesis Stream (Ingress)',
                left: [
                    this.stream.metricIncomingRecords({
                        label: 'Incoming Records',
                        statistic: 'sum',
                    }),
                ],
                right: [
                    this.stream.metricIncomingBytes({
                        label: 'Incoming Bytes',
                        statistic: 'sum',
                    }),
                ],
            }),
            new cloudwatch.GraphWidget({
                title: 'Kinesis Stream (Throttling)',
                left: [
                    this.stream.metricGetRecordsSuccess(),
                    this.stream.metricPutRecordsSuccess(),
                ],
                right: [
                    this.stream.metricReadProvisionedThroughputExceeded({
                        label: 'Throttled Reads',
                        statistic: 'sum',
                    }),
                    this.stream.metricWriteProvisionedThroughputExceeded({
                        label: 'Throttled Writes',
                        statistic: 'sum',
                    }),
                ],
            }),
            new cloudwatch.GraphWidget({
                title: 'Kinesis Stream (Delay)',
                left: [
                    this.stream.metricGetRecordsIteratorAgeMilliseconds({
                        label: 'Time-lag behind Kinesis Stream',
                        statistic: 'max',
                    }),
                ],
            }),
            new cloudwatch.GraphWidget({
                title: 'Firehose Delivery Stream (Ingress/Egress)',
                left: [
                    this.deliveryStream.metricIncomingRecords({
                        label: 'Incoming Records',
                        statistic: 'sum',
                    }),
                    this.deliveryStream.metricDeliveryToS3Records({
                        label: 'Outgoing Records',
                        statistic: 'sum',
                    }),
                ],
                right: [
                    this.deliveryStream.metricDeliveryToS3Success(),
                ],
            }),
            new cloudwatch.GraphWidget({
                title: 'Firehose Data Freshness',
                left: [
                    this.deliveryStream.metricDeliveryToS3DataFreshness({
                        label: 'Freshness',
                        statistic: 'max',
                        period: aws_cdk_lib_1.Duration.minutes(5),
                    }),
                ],
            }),
        ];
        widgets.push(...this.alarmWidgets(this.alarmsSev2, 2));
        widgets.push(...this.alarmWidgets(this.alarmsSev3, 3));
        this.addWidgets(widgets);
    }
}
exports.KinesisOps = KinesisOps;
_a = JSII_RTTI_SYMBOL_1;
KinesisOps[_a] = { fqn: "@randyridgley/cdk-datalake-constructs.KinesisOps", version: "0.0.52" };
//# sourceMappingURL=data:application/json;base64,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