"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.GlueCrawler = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cloudwatch = require("aws-cdk-lib/aws-cloudwatch");
const events = require("aws-cdk-lib/aws-events");
const glue = require("aws-cdk-lib/aws-glue");
const iam = require("aws-cdk-lib/aws-iam");
const lf = require("aws-cdk-lib/aws-lakeformation");
const constructs_1 = require("constructs");
const enums_1 = require("../global/enums");
class GlueCrawler extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.role = this.createGlueCrawlerRole(props);
        this.metricSuccessRule = this.crawlerRule('SuccessRule', props.name, 'Succeeded');
        this.metricFailureRule = this.crawlerRule('FailureRule', props.name, 'Failed');
        let s3TargetPaths = new Array();
        s3TargetPaths.push({
            path: `s3://${props.bucketName}/${props.bucketPrefix}`,
        });
        this.crawler = new glue.CfnCrawler(this, `data-lake-crawler-${props.name}-`, {
            name: props.name,
            role: this.role.roleArn,
            databaseName: props.databaseName,
            targets: {
                s3Targets: s3TargetPaths,
            },
        });
        new lf.CfnPermissions(this, 'glue-role-database-permission', {
            dataLakePrincipal: {
                dataLakePrincipalIdentifier: this.role.roleArn,
            },
            resource: {
                databaseResource: {
                    name: props.databaseName,
                },
            },
            permissions: [
                'CREATE_TABLE',
                'DESCRIBE',
            ],
        });
        new lf.CfnPermissions(this, 'datalake-creator-permission', {
            dataLakePrincipal: {
                dataLakePrincipalIdentifier: this.role.roleArn,
            },
            resource: {
                dataLocationResource: {
                    s3Resource: `arn:aws:s3:::${props.bucketName}`,
                },
            },
            permissions: [
                enums_1.Permissions.DATA_LOCATION_ACCESS,
            ],
        });
    }
    metricSuccess(props) {
        return this.ruleMetric(this.metricSuccessRule, props);
    }
    metricFailure(props) {
        return this.ruleMetric(this.metricFailureRule, props);
    }
    ruleMetric({ ruleName }, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/Events',
            metricName: 'TriggeredRules',
            dimensionsMap: { RuleName: ruleName },
            statistic: cloudwatch.Statistic.SUM,
            ...props,
        }).attachTo(this);
    }
    crawlerRule(id, crawlerName, ...states) {
        return new events.Rule(this, id, {
            ruleName: crawlerName + states.join(''),
            description: `Event triggered when Glue Crawler ${crawlerName} is in ${states.join(' or ')} state(s)`,
            eventPattern: {
                source: ['aws.glue'],
                detailType: ['Glue Crawler State Change'],
                detail: {
                    state: states,
                    crawlerName: [crawlerName],
                },
            },
        });
    }
    createGlueCrawlerRole(props) {
        const role = new iam.Role(this, 'Role', {
            roleName: props.roleName || props.name + 'Role',
            assumedBy: new iam.ServicePrincipal('glue.amazonaws.com'),
            managedPolicies: [
                iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSGlueServiceRole'),
                iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonS3FullAccess'),
            ],
        });
        role.addToPolicy(new iam.PolicyStatement({ actions: ['lakeformation:GetDataAccess'], resources: ['*'] }));
        return role;
    }
}
exports.GlueCrawler = GlueCrawler;
_a = JSII_RTTI_SYMBOL_1;
GlueCrawler[_a] = { fqn: "@randyridgley/cdk-datalake-constructs.GlueCrawler", version: "0.0.52" };
//# sourceMappingURL=data:application/json;base64,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