"use strict";
/**
 *  Copyright 2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.S3ToStepFunction = void 0;
const defaults = require("@aws-solutions-constructs/core");
const aws_events_rule_step_function_1 = require("@aws-solutions-constructs/aws-events-rule-step-function");
const core_1 = require("@aws-cdk/core");
const cloudtrail = require("@aws-cdk/aws-cloudtrail");
class S3ToStepFunction extends core_1.Construct {
    /**
     * @summary Constructs a new instance of the S3ToStepFunction class.
     * @param {cdk.App} scope - represents the scope for all the resources.
     * @param {string} id - this is a a scope-unique id.
     * @param {S3ToStepFunctionProps} props - user provided props for the construct
     * @since 0.9.0
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        let bucket;
        if (!props.existingBucketObj) {
            [this.s3Bucket, this.s3LoggingBucket] = defaults.buildS3Bucket(this, {
                bucketProps: props.bucketProps
            });
            bucket = this.s3Bucket;
        }
        else {
            bucket = props.existingBucketObj;
        }
        if (!props.hasOwnProperty('deployCloudTrail') || props.deployCloudTrail === true) {
            [this.cloudtrailBucket, this.cloudtrailLoggingBucket] = defaults.buildS3Bucket(this, {}, 'CloudTrail');
            this.cloudtrail = new cloudtrail.Trail(this, 'S3EventsTrail', {
                bucket: this.cloudtrailBucket
            });
            this.cloudtrail.addS3EventSelector([{
                    bucket
                }], {
                readWriteType: cloudtrail.ReadWriteType.ALL,
                includeManagementEvents: false
            });
        }
        let _eventRuleProps = {};
        if (props.eventRuleProps) {
            _eventRuleProps = props.eventRuleProps;
        }
        else {
            // By default the CW Events Rule will filter any 's3:PutObject' events for the S3 Bucket
            _eventRuleProps = {
                eventPattern: {
                    source: ['aws.s3'],
                    detailType: ['AWS API Call via CloudTrail'],
                    detail: {
                        eventSource: [
                            "s3.amazonaws.com"
                        ],
                        eventName: [
                            "PutObject"
                        ],
                        requestParameters: {
                            bucketName: [
                                bucket.bucketName
                            ]
                        }
                    }
                }
            };
        }
        const eventsRuleToStepFunction = new aws_events_rule_step_function_1.EventsRuleToStepFunction(this, 'test-events-rule-step-function-stack', {
            stateMachineProps: props.stateMachineProps,
            eventRuleProps: _eventRuleProps
        });
        this.stateMachine = eventsRuleToStepFunction.stateMachine;
        this.stateMachineLogGroup = eventsRuleToStepFunction.stateMachineLogGroup;
        this.cloudwatchAlarms = eventsRuleToStepFunction.cloudwatchAlarms;
    }
}
exports.S3ToStepFunction = S3ToStepFunction;
//# sourceMappingURL=data:application/json;base64,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