"""
make_fsg.py generate an FSG or a JSGF for a marked up text file.

This module takes a text file, marked up with units (e.g. w for word, m for
morpheme) and ids and converted to IPA, and outputs an FSG or a JSGF
file for processing by PocketSphinx, SoundSwallower or SoundSwallower.js
"""


import datetime
import os

import chevron
from slugify import slugify

from readalongs.log import LOGGER

FSG_TEMPLATE = """FSG_BEGIN {{name}}
NUM_STATES {{num_states}}
START_STATE 0
FINAL_STATE {{final_state}}

{{#states}}
TRANSITION {{current}} {{next}} 1.0 {{id}}
{{/states}}
FSG_END
"""


def get_ids(word_elements: list):
    """Extract the sequence of id's from word_elements with both an id and
    an arpabet pronounciation.

    Words with empty ARPABET are skipped because soundswallower and
    pocketsphinx will error out if we give it words with an empty pronunciation
    key. In general, what *would* it mean to align sounds to an empty sequence
    of phonemes, after all???

    Yields:
        text_ids
    """

    for e in word_elements:
        if "id" not in e.attrib:  # don't put in elements with no id
            continue
        if not e.attrib.get("ARPABET", "").strip():
            LOGGER.warning("Skipping node %s with no ARPABET", e.attrib["id"])
            continue
        yield e.attrib["id"]


def make_fsg(word_elements: list, filename: str = "'in-memory'") -> str:
    """Generate an FSG for the given words elements

    Returns: the text contents of the FSG file for processing by PocketSphinx
    """

    states = [
        {"id": text_id, "current": i, "next": i + 1}
        for i, text_id in enumerate(get_ids(word_elements))
    ]

    data = {
        # If name includes special characters, pocketsphinx throws a RuntimeError:
        # new_Decoder returned -1, so pass it through slugify() first
        "name": slugify(os.path.splitext(os.path.basename(filename))[0]),
        "states": states,
        "final_state": len(states),
        "num_states": len(states) + 1,
    }

    return chevron.render(FSG_TEMPLATE, data)


JSGF_TEMPLATE = """#JSGF 1.0 UTF-8;
grammar {{name}};

/**
    * Auto-generated JSGF grammar for the document {{name}}.
    *
    * @author Automatically generated by make_jsgf
    * @version 1.0
    * @since {{date}}
    */

public <s> = {{#words}} {{id}} {{/words}} ;
"""


def make_jsgf(word_elements: list, filename: str = "'in-memory'") -> str:
    """Generate a JSGF for the given words elements

    JSGF = Java Speech Grammar Format

    Returns:
        the text contents of the JSGF file for processing by SoundSwallower.js
    """
    data = {
        "name": os.path.splitext(os.path.basename(filename))[0],
        "date": datetime.datetime.today().strftime("%Y-%m-%d"),
        "words": [{"id": text_id} for text_id in get_ids(word_elements)],
    }

    return chevron.render(JSGF_TEMPLATE, data)
