"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.mapTagMapToCxschema = exports.validateNetworkProtocol = exports.ifUndefined = exports.determineProtocolAndPort = exports.defaultProtocolForPort = exports.defaultPortForProtocol = exports.renderAttributes = void 0;
const enums_1 = require("./enums");
/**
 * Render an attribute dict to a list of { key, value } pairs
 */
function renderAttributes(attributes) {
    const ret = [];
    for (const [key, value] of Object.entries(attributes)) {
        if (value !== undefined) {
            ret.push({ key, value });
        }
    }
    return ret;
}
exports.renderAttributes = renderAttributes;
/**
 * Return the appropriate default port for a given protocol
 */
function defaultPortForProtocol(proto) {
    switch (proto) {
        case enums_1.ApplicationProtocol.HTTP: return 80;
        case enums_1.ApplicationProtocol.HTTPS: return 443;
        default:
            throw new Error(`Unrecognized protocol: ${proto}`);
    }
}
exports.defaultPortForProtocol = defaultPortForProtocol;
/**
 * Return the appropriate default protocol for a given port
 */
function defaultProtocolForPort(port) {
    switch (port) {
        case 80:
        case 8000:
        case 8008:
        case 8080:
            return enums_1.ApplicationProtocol.HTTP;
        case 443:
        case 8443:
            return enums_1.ApplicationProtocol.HTTPS;
        default:
            throw new Error(`Don't know default protocol for port: ${port}; please supply a protocol`);
    }
}
exports.defaultProtocolForPort = defaultProtocolForPort;
/**
 * Given a protocol and a port, try to guess the other one if it's undefined
 */
// eslint-disable-next-line max-len
function determineProtocolAndPort(protocol, port) {
    if (protocol === undefined && port === undefined) {
        return [undefined, undefined];
    }
    if (protocol === undefined) {
        protocol = defaultProtocolForPort(port);
    }
    if (port === undefined) {
        port = defaultPortForProtocol(protocol);
    }
    return [protocol, port];
}
exports.determineProtocolAndPort = determineProtocolAndPort;
/**
 * Helper function to default undefined input props
 */
function ifUndefined(x, def) {
    return x !== null && x !== void 0 ? x : def;
}
exports.ifUndefined = ifUndefined;
/**
 * Helper function for ensuring network listeners and target groups only accept valid
 * protocols.
 */
function validateNetworkProtocol(protocol) {
    const NLB_PROTOCOLS = [enums_1.Protocol.TCP, enums_1.Protocol.TLS, enums_1.Protocol.UDP, enums_1.Protocol.TCP_UDP];
    if (NLB_PROTOCOLS.indexOf(protocol) === -1) {
        throw new Error(`The protocol must be one of ${NLB_PROTOCOLS.join(', ')}. Found ${protocol}`);
    }
}
exports.validateNetworkProtocol = validateNetworkProtocol;
/**
 * Helper to map a map of tags to cxschema tag format.
 * @internal
 */
function mapTagMapToCxschema(tagMap) {
    return Object.entries(tagMap)
        .map(([key, value]) => ({ key, value }));
}
exports.mapTagMapToCxschema = mapTagMapToCxschema;
//# sourceMappingURL=data:application/json;base64,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