"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const lib_1 = require("../lib");
const core_1 = require("@aws-cdk/core");
const kinesis = require("@aws-cdk/aws-kinesis");
const lambda = require("@aws-cdk/aws-lambda");
const iam = require("@aws-cdk/aws-iam");
const core_2 = require("@aws-solutions-constructs/core");
// Setup
const app = new core_1.App();
const stack = new core_1.Stack(app, core_2.generateIntegStackName(__filename));
stack.templateOptions.description = 'Integration Test for aws-kinesisstreams-lambda';
const lambdaRole = new iam.Role(stack, 'test-role', {
    assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
    inlinePolicies: {
        LambdaFunctionServiceRolePolicy: new iam.PolicyDocument({
            statements: [new iam.PolicyStatement({
                    actions: [
                        'logs:CreateLogGroup',
                        'logs:CreateLogStream',
                        'logs:PutLogEvents'
                    ],
                    resources: [`arn:${core_1.Aws.PARTITION}:logs:${core_1.Aws.REGION}:${core_1.Aws.ACCOUNT_ID}:log-group:/aws/lambda/*`]
                })]
        })
    }
});
const lambdaFn = new lambda.Function(stack, 'test-fn', {
    runtime: lambda.Runtime.NODEJS_14_X,
    handler: 'index.handler',
    code: lambda.Code.fromAsset(`${__dirname}/lambda`),
    role: lambdaRole,
});
const cfnLambdafunction = lambdaFn.node.findChild('Resource');
cfnLambdafunction.cfnOptions.metadata = {
    cfn_nag: {
        rules_to_suppress: [{
                id: 'W89',
                reason: `This is not a rule for the general case, just for specific use cases/industries`
            },
            {
                id: 'W92',
                reason: `Impossible for us to define the correct concurrency for clients`
            }]
    }
};
const stream = new kinesis.Stream(stack, 'test-stream', {
    shardCount: 2,
    encryption: kinesis.StreamEncryption.MANAGED
});
// Definitions
const props = {
    existingStreamObj: stream,
    existingLambdaObj: lambdaFn,
    kinesisEventSourceProps: {
        startingPosition: lambda.StartingPosition.LATEST,
        batchSize: 1
    },
};
new lib_1.KinesisStreamsToLambda(stack, 'test-ks-lambda', props);
// Synth
app.synth();
//# sourceMappingURL=data:application/json;base64,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