# Netskope SDK

Neskope SDK is Python library for dealing with API's to download the Netskope events. 

## Installation

Use the package manager [pip](https://pip.pypa.io/en/stable/) to install NetskopeSDK.

```bash
pip install netskopesdk
```

## Rest sdk Usage to pull the Alert & Events 

```python
from netskope_api.iterator.netskope_iterator import NetskopeIterator
from netskope_api.iterator.const import Const
from requests.exceptions import RequestException
import time

# Construct the params dict to pass the authentication details 
params = {
        Const.NSKP_TOKEN : "<REST-API-TOKEN>",
        Const.NSKP_TENANT_HOSTNAME : "<HOSTNAME>",
        # Optional param to pass the proxy hosts.
        Const.NSKP_PROXIES : {"<PROXY-HOSTS>"},
        Const.NSKP_EVENT_TYPE : "<EVENT-TYPE>",
        Const.NSKP_ITERATOR_NAME : "<ITERATOR-NAME>",
        Const.NSKP_USER_AGENT : "<SPLUNK-TENANT-HOSTNAME>"
    
        # To query specific alert pass the NSKP_EVENT_TYPE as "alert" and the alert type.
        # Const.NSKP_EVENT_TYPE : Const.EVENT_TYPE_ALERT,
        # Const.NSKP_ALERT_TYPE : Const.ALERT_TYPE_DLP
    }

DEFAULT_WAIT_TIME = 30
RESULT = "result"
WAIT_TIME = "wait_time"

# Create an Iterator
iterator = NetskopeIterator(params)

# Use the next() iterator to download the logs. 
# Consume the message indefinitely in a loop and ingest the data to SIEM
while True:
    response = (iterator.next())
    try:
        if response:
            data = response.json()
            if RESULT in data and len(data[RESULT]) != 0:
                # processData() 
                # sleep() the thread to avoid constant polling
                if WAIT_TIME in data:
                    time.sleep(data[WAIT_TIME])
                else:
                    time.sleep(DEFAULT_WAIT_TIME)
            else:
                print("No response received from the iterator")
                time.sleep(DEFAULT_WAIT_TIME)
    except Exception as e:
        time.sleep(DEFAULT_WAIT_TIME)
        raise RequestException(e)
```



## Rest sdk Usage to retrieve tokens used for subscribing to transaction events from PSL.

```python
from requests.exceptions import RequestException
from netskope_api.iterator.const import Const
from netskope_api.token_management.netskope_management import NetskopeTokenManagement

if __name__ == '__main__':
    params = {
        Const.NSKP_TOKEN: "",
        Const.NSKP_TENANT_HOSTNAME: "netskope.goskope.com"
        # Optional param to pass the proxy hosts.
        Const.NSKP_PROXIES : {"<PROXY-HOSTS>"}
    }

    sub_path_response = None
    sub_key_response = None
    try:
        # Create token_management client
        token_management = NetskopeTokenManagement(params)
        token_management_response = token_management.get()
        if token_management_response:
            if "subscription" in token_management_response:
                sub_path_response = token_management_response["subscription"]
            if "subscription-key" in token_management_response:
                sub_key_response = token_management_response["subscription-key"]
    except Exception as e:
        raise RequestException(e)

```
```
1. 401 response code means Customer is not authorized to create/get subscription key and path.
2. 403 response code means Subscription key and path are already present for the customer which is explained in the following section.
```
## Onboard existing customers using the API.
    regenerate_and_get() func can be used for onboarding existing customers who already downloaded the subscription key using tools → event streaming section.

    Regenerating subscription key will invalidate the existing key but subscription path will not be updated so that clients can continue consuming events where they left off while being onboarded using the API.

## Note: Regenerating subscription key will invalidate the existing key.

```python
from requests.exceptions import RequestException
from netskope_api.iterator.const import Const
from netskope_api.token_management.netskope_management import NetskopeTokenManagement

if __name__ == '__main__':
    params = {
        Const.NSKP_TOKEN: "",
        Const.NSKP_TENANT_HOSTNAME: "netskope.goskope.com"
        # Optional param to pass the proxy hosts.
        Const.NSKP_PROXIES : {"<PROXY-HOSTS>"}
    }

    sub_path_response = None
    sub_key_response = None
    try:
        # Create token_management client
        token_management = NetskopeTokenManagement(params)
        token_management_response = token_management.regenerate_and_get()
        if token_management_response:
            if "subscription" in token_management_response:
                sub_path_response = token_management_response["subscription"]
            if "subscription-key" in token_management_response:
                sub_key_response = token_management_response["subscription-key"]
    except Exception as e:
        raise RequestException(e)
```

## Example usage of web txn tokens using Pubsublite client library

Please install Pubsublite client library if not present.

```bash
pip install google-cloud-pubsublite
```


```python
import concurrent.futures
import logging
import os
from concurrent.futures._base import TimeoutError

from google.cloud.pubsublite.cloudpubsub import SubscriberClient
from google.cloud.pubsublite.types import FlowControlSettings, MessageMetadata
from google.pubsub_v1 import PubsubMessage
from requests.exceptions import RequestException

from netskope_api.iterator.const import Const
from netskope_api.token_management.netskope_management import NetskopeTokenManagement

_logger = logging.getLogger()


def callback(message: PubsubMessage):
    message_data = message.data.decode("utf-8")
    metadata = MessageMetadata.decode(message.message_id)
    _logger.info(
        f"Received {message_data} of ordering key {message.ordering_key} with id {metadata}."
    )
    message.ack()


class PSLSubscriberClient:
    def __init__(self):

        configs = {
            "messages_outstanding": 1000,
            "bytes_outstanding": 3.5 * 1024 * 1024,
            "timeout": 60,
            "thread_count": 1
        }
        self.configs = configs

    def make_default_thread_pool_executor(self):
        return concurrent.futures.ThreadPoolExecutor(self.configs.get("thread_count"))

    def stream(self, subscription_path):
        global streaming_pull_future

        per_partition_flow_control_settings = FlowControlSettings(
            # Must be >0.
            messages_outstanding=self.configs.get("messages_outstanding"),
            # Must be greater than the allowed size of the largest message.
            bytes_outstanding=self.configs.get("bytes_outstanding"),
        )

        executor = self.make_default_thread_pool_executor()
        with SubscriberClient(executor=executor) as subscriber_client:
            _logger.info(
                "Listening for messages on the pub sub lite subscription {}".format(subscription_path))
            streaming_pull_future = subscriber_client.subscribe(
                subscription_path,
                callback=callback,
                per_partition_flow_control_settings=per_partition_flow_control_settings,
            )
            try:
                timeout = self.configs.get("timeout")
                if timeout:
                    streaming_pull_future.result(timeout=self.configs.get("timeout"))
                else:
                    streaming_pull_future.result()
            except TimeoutError or KeyboardInterrupt:
                streaming_pull_future.cancel()
                assert streaming_pull_future.done()


if __name__ == '__main__':
    params = {
        Const.NSKP_TOKEN: "",
        Const.NSKP_TENANT_HOSTNAME: "netskope.goskope.com"
        # Optional param to pass the proxy hosts.
        Const.NSKP_PROXIES : {"<PROXY-HOSTS>"}
    }

    sub_path_response = None
    sub_key_response = None
    try:
        # Create token_management client
        token_management = NetskopeTokenManagement(params)
        token_management_response = token_management.get()
        if token_management_response:
            if "subscription" in token_management_response:
                sub_path_response = token_management_response["subscription"]
            if "subscription-key" in token_management_response:
                sub_key_response = token_management_response["subscription-key"]
    except Exception as e:
        raise RequestException(e)

    os.environ['GOOGLE_APPLICATION_CREDENTIALS'] = sub_key_response
    psl_client = PSLSubscriberClient()
    psl_client.stream(subscription_path=sub_path_response)
```
