"use strict";
const assert_1 = require("@aws-cdk/assert");
const iam = require("@aws-cdk/aws-iam");
const kms = require("@aws-cdk/aws-kms");
const core_1 = require("@aws-cdk/core");
const sqs = require("../lib");
function testGrant(action, ...expectedActions) {
    const stack = new core_1.Stack();
    const queue = new sqs.Queue(stack, 'MyQueue');
    const principal = new iam.User(stack, 'User');
    action(queue, principal);
    assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
        'PolicyDocument': {
            'Statement': [
                {
                    'Action': expectedActions,
                    'Effect': 'Allow',
                    'Resource': {
                        'Fn::GetAtt': [
                            'MyQueueE6CA6235',
                            'Arn'
                        ]
                    }
                }
            ],
            'Version': '2012-10-17'
        }
    }));
}
module.exports = {
    'default properties'(test) {
        const stack = new core_1.Stack();
        const q = new sqs.Queue(stack, 'Queue');
        test.deepEqual(q.fifo, false);
        assert_1.expect(stack).toMatch({
            'Resources': {
                'Queue4A7E3555': {
                    'Type': 'AWS::SQS::Queue'
                }
            }
        });
        test.done();
    },
    'with a dead letter queue'(test) {
        const stack = new core_1.Stack();
        const dlq = new sqs.Queue(stack, 'DLQ');
        new sqs.Queue(stack, 'Queue', { deadLetterQueue: { queue: dlq, maxReceiveCount: 3 } });
        assert_1.expect(stack).toMatch({
            'Resources': {
                'DLQ581697C4': {
                    'Type': 'AWS::SQS::Queue'
                },
                'Queue4A7E3555': {
                    'Type': 'AWS::SQS::Queue',
                    'Properties': {
                        'RedrivePolicy': {
                            'deadLetterTargetArn': {
                                'Fn::GetAtt': [
                                    'DLQ581697C4',
                                    'Arn'
                                ]
                            },
                            'maxReceiveCount': 3
                        }
                    }
                }
            }
        });
        test.done();
    },
    'addToPolicy will automatically create a policy for this queue'(test) {
        const stack = new core_1.Stack();
        const queue = new sqs.Queue(stack, 'MyQueue');
        queue.addToResourcePolicy(new iam.PolicyStatement({
            resources: ['*'],
            actions: ['sqs:*'],
            principals: [new iam.ArnPrincipal('arn')]
        }));
        assert_1.expect(stack).toMatch({
            'Resources': {
                'MyQueueE6CA6235': {
                    'Type': 'AWS::SQS::Queue'
                },
                'MyQueuePolicy6BBEDDAC': {
                    'Type': 'AWS::SQS::QueuePolicy',
                    'Properties': {
                        'PolicyDocument': {
                            'Statement': [
                                {
                                    'Action': 'sqs:*',
                                    'Effect': 'Allow',
                                    'Principal': {
                                        'AWS': 'arn'
                                    },
                                    'Resource': '*'
                                }
                            ],
                            'Version': '2012-10-17'
                        },
                        'Queues': [
                            {
                                'Ref': 'MyQueueE6CA6235'
                            }
                        ]
                    }
                }
            }
        });
        test.done();
    },
    'export and import': {
        'importing works correctly'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            // WHEN
            const imports = sqs.Queue.fromQueueArn(stack, 'Imported', 'arn:aws:sqs:us-east-1:123456789012:queue1');
            // THEN
            // "import" returns an IQueue bound to `Fn::ImportValue`s.
            test.deepEqual(stack.resolve(imports.queueArn), 'arn:aws:sqs:us-east-1:123456789012:queue1');
            test.deepEqual(stack.resolve(imports.queueUrl), { 'Fn::Join': ['', ['https://sqs.', { Ref: 'AWS::Region' }, '.', { Ref: 'AWS::URLSuffix' }, '/', { Ref: 'AWS::AccountId' }, '/queue1']] });
            test.deepEqual(stack.resolve(imports.queueName), 'queue1');
            test.done();
        },
        'importing fifo and standard queues are detected correctly'(test) {
            const stack = new core_1.Stack();
            const stdQueue = sqs.Queue.fromQueueArn(stack, 'StdQueue', 'arn:aws:sqs:us-east-1:123456789012:queue1');
            const fifoQueue = sqs.Queue.fromQueueArn(stack, 'FifoQueue', 'arn:aws:sqs:us-east-1:123456789012:queue2.fifo');
            test.deepEqual(stdQueue.fifo, false);
            test.deepEqual(fifoQueue.fifo, true);
            test.done();
        },
    },
    'grants': {
        'grantConsumeMessages'(test) {
            testGrant((q, p) => q.grantConsumeMessages(p), 'sqs:ReceiveMessage', 'sqs:ChangeMessageVisibility', 'sqs:GetQueueUrl', 'sqs:DeleteMessage', 'sqs:GetQueueAttributes');
            test.done();
        },
        'grantSendMessages'(test) {
            testGrant((q, p) => q.grantSendMessages(p), 'sqs:SendMessage', 'sqs:GetQueueAttributes', 'sqs:GetQueueUrl');
            test.done();
        },
        'grantPurge'(test) {
            testGrant((q, p) => q.grantPurge(p), 'sqs:PurgeQueue', 'sqs:GetQueueAttributes', 'sqs:GetQueueUrl');
            test.done();
        },
        'grant() is general purpose'(test) {
            testGrant((q, p) => q.grant(p, 'service:hello', 'service:world'), 'service:hello', 'service:world');
            test.done();
        },
        'grants also work on imported queues'(test) {
            const stack = new core_1.Stack();
            const queue = sqs.Queue.fromQueueAttributes(stack, 'Import', {
                queueArn: 'arn:aws:sqs:us-east-1:123456789012:queue1',
                queueUrl: 'https://queue-url'
            });
            const user = new iam.User(stack, 'User');
            queue.grantPurge(user);
            assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
                'PolicyDocument': {
                    'Statement': [
                        {
                            'Action': [
                                'sqs:PurgeQueue',
                                'sqs:GetQueueAttributes',
                                'sqs:GetQueueUrl'
                            ],
                            'Effect': 'Allow',
                            'Resource': 'arn:aws:sqs:us-east-1:123456789012:queue1'
                        }
                    ],
                    'Version': '2012-10-17'
                }
            }));
            test.done();
        }
    },
    'queue encryption': {
        'encryptionMasterKey can be set to a custom KMS key'(test) {
            const stack = new core_1.Stack();
            const key = new kms.Key(stack, 'CustomKey');
            const queue = new sqs.Queue(stack, 'Queue', { encryptionMasterKey: key });
            test.same(queue.encryptionMasterKey, key);
            assert_1.expect(stack).to(assert_1.haveResource('AWS::SQS::Queue', {
                'KmsMasterKeyId': { 'Fn::GetAtt': ['CustomKey1E6D0D07', 'Arn'] }
            }));
            test.done();
        },
        'a kms key will be allocated if encryption = kms but a master key is not specified'(test) {
            const stack = new core_1.Stack();
            new sqs.Queue(stack, 'Queue', { encryption: sqs.QueueEncryption.KMS });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::KMS::Key'));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::SQS::Queue', {
                'KmsMasterKeyId': {
                    'Fn::GetAtt': [
                        'QueueKey39FCBAE6',
                        'Arn'
                    ]
                }
            }));
            test.done();
        },
        'it is possible to use a managed kms key'(test) {
            const stack = new core_1.Stack();
            new sqs.Queue(stack, 'Queue', { encryption: sqs.QueueEncryption.KMS_MANAGED });
            assert_1.expect(stack).toMatch({
                'Resources': {
                    'Queue4A7E3555': {
                        'Type': 'AWS::SQS::Queue',
                        'Properties': {
                            'KmsMasterKeyId': 'alias/aws/sqs'
                        }
                    }
                }
            });
            test.done();
        },
        'grant also affects key on encrypted queue'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const queue = new sqs.Queue(stack, 'Queue', {
                encryption: sqs.QueueEncryption.KMS
            });
            const role = new iam.Role(stack, 'Role', {
                assumedBy: new iam.ServicePrincipal('someone')
            });
            // WHEN
            queue.grantSendMessages(role);
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
                'PolicyDocument': {
                    'Statement': [
                        {
                            'Action': [
                                'sqs:SendMessage',
                                'sqs:GetQueueAttributes',
                                'sqs:GetQueueUrl'
                            ],
                            'Effect': 'Allow',
                            'Resource': { 'Fn::GetAtt': ['Queue4A7E3555', 'Arn'] }
                        },
                        {
                            'Action': [
                                'kms:Encrypt',
                                'kms:ReEncrypt*',
                                'kms:GenerateDataKey*'
                            ],
                            'Effect': 'Allow',
                            'Resource': { 'Fn::GetAtt': ['QueueKey39FCBAE6', 'Arn'] }
                        }
                    ],
                    'Version': '2012-10-17'
                },
            }));
            test.done();
        },
    },
    'test ".fifo" suffixed queues register as fifo'(test) {
        const stack = new core_1.Stack();
        const queue = new sqs.Queue(stack, 'Queue', {
            queueName: 'MyQueue.fifo'
        });
        test.deepEqual(queue.fifo, true);
        assert_1.expect(stack).toMatch({
            'Resources': {
                'Queue4A7E3555': {
                    'Type': 'AWS::SQS::Queue',
                    'Properties': {
                        'QueueName': 'MyQueue.fifo',
                        'FifoQueue': true
                    }
                }
            }
        });
        test.done();
    },
    'test a fifo queue is observed when the "fifo" property is specified'(test) {
        const stack = new core_1.Stack();
        const queue = new sqs.Queue(stack, 'Queue', {
            fifo: true
        });
        test.deepEqual(queue.fifo, true);
        assert_1.expect(stack).toMatch({
            'Resources': {
                'Queue4A7E3555': {
                    'Type': 'AWS::SQS::Queue',
                    'Properties': {
                        'FifoQueue': true
                    }
                }
            }
        });
        test.done();
    },
    'test metrics'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const queue = new sqs.Queue(stack, 'Queue');
        // THEN
        test.deepEqual(stack.resolve(queue.metricNumberOfMessagesSent()), {
            dimensions: { QueueName: { 'Fn::GetAtt': ['Queue4A7E3555', 'QueueName'] } },
            namespace: 'AWS/SQS',
            metricName: 'NumberOfMessagesSent',
            period: core_1.Duration.minutes(5),
            statistic: 'Sum'
        });
        test.deepEqual(stack.resolve(queue.metricSentMessageSize()), {
            dimensions: { QueueName: { 'Fn::GetAtt': ['Queue4A7E3555', 'QueueName'] } },
            namespace: 'AWS/SQS',
            metricName: 'SentMessageSize',
            period: core_1.Duration.minutes(5),
            statistic: 'Average'
        });
        test.done();
    }
};
//# sourceMappingURL=data:application/json;base64,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