Neural Networks for NSE as low-dimensional LPV
---

# Active workflow

## Installation

```sh
# package for sparse cholesky factorizations 
# not needed but speed up with FEM norms and POD
apt install libsuitesparse-dev
pip install scikit-sparse==0.4.5

# fenics -- for the FEM part
apt install fenics  # see https://fenicsproject.org/download/

# install this module and helper modules
pip install .
```

## Generate the data

```sh
cd ../simulations-training-data
mkdir cached-data
mkdir train-data
source single-cylinder-tdp-sim.sh
# source double-cylinder-tdp-sim.sh

# python3 time_dep_nse_generic.py
cd -
```

## Check the NN

```sh
python3 data_fem_checks.py
python3 CNN_AE.py
```

## Data and Handling

Generally, the data is 

1. generated by the simulation
2. interpolated to *the pictures*
3. imported to PyTorch as tensors

The relevant formats and routines are as follows:

```python
import nse_nn_lpv.nse_data_helpers as ndh
```

1. `time_dep_nse_generic.py` computes the trajectories with snapshots `vvec` and interpolates them to the two pictures `vmatx` and `vmaty` at all time instances `ti`. The data is stored as a `json` file, say, `data.json` like
```json
{ti: {'vvec': vvec, 'vmatx': vmatx, 'vmaty': vmaty},
 'femdata': 'information of the simulation ...'}
```

2. `(datal, vvecl) = ndh.get_nse_img_data('data.json')` takes the data file `data.json` and returns the data as a list (of tuples) of `numpy` arrays: 
```python
datal = [(vmatx, vmaty, t_0), ..., (vmatx, vmaty, t_end)]
vvecl = [vvec_0, ..., ..., ..., ..., vvec_end]
```

3. `trn_nse_data = ndh.NSEDataset(nsedatal, vvecl)` takes the data lists and makes it available as a `pytorch` data set. In particular, the two *pictures* `vmatx`, `vmaty` are merged into a tensor.

4. `stst_dataloader = pytorch.DataLoader(trn_nse_data, batch_size=1, shuffle=True)` then defines a way to access the data. E.g., 
```python
(ttstset, tstvec) = next(iter(stst_dataloader))
```
returns a data point via
 * `ttstset`: the tensor of size `(batch_size, 2, width, height)`, where `(width, height)` are the dimensions of the *pictures*
 * `tstvec`: the corresponding `vvecs` (needed, e.g., for the loss function later)

## Python Machine-Learning Resources

 * an [overview](https://analyticsindiamag.com/top-7-python-neural-network-libraries-for-developers/)
   1. Tensorflow -- see below
   2. Pytorch -- see below
   3. NeuroLab
   4. ffnet
   5. Scikit-Neural Network
   6. Lasagne
   7. pyrenn

### Tensorflow

 * [website](https://www.tensorflow.org/)
 * [Article for understanding NN using TensorFlow](https://towardsdatascience.com/building-your-first-neural-network-in-tensorflow-2-tensorflow-for-hackers-part-i-e1e2f1dfe7a0)
 * Previous experience with building NN
* Visualization feature [TensorBoard](https://www.tensorflow.org/tensorboard)
* Based on keras 

### PyTorch

 * [website](https://pytorch.org/)
 * [Tutorial for simple NN](https://pytorch.org/tutorials/beginner/blitz/neural_networks_tutorial.html)
 * Recommended by colleagues (Lessig, Richter)

### Scikit-Learn

 * [website](https://scikit-learn.org/stable/index.html)
 * looks well maintained
 * many routines for data processing
 * a few on neural network 

## Install

```sh
# not needed anymore
# pip install -e .  # Python3 needed here! install the module (and one dependency)
```

`pip install -e` installs the module `nse_nn_lpv` to be used in the `tests/...`
but keeps track of all changes made in `nse_nn_lpv`.
