"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LambdaDestination = void 0;
const events = require("@aws-cdk/aws-events");
const targets = require("@aws-cdk/aws-events-targets");
const event_bridge_1 = require("./event-bridge");
/**
 * Use a Lambda function as a Lambda destination
 */
class LambdaDestination {
    constructor(fn, options = {}) {
        this.fn = fn;
        this.options = options;
    }
    /**
     * Returns a destination configuration
     */
    bind(scope, fn, options) {
        // Normal Lambda destination (full invocation record)
        if (!this.options.responseOnly) {
            // deduplicated automatically
            this.fn.grantInvoke(fn);
            return {
                destination: this.fn.functionArn,
            };
        }
        // Otherwise add rule to extract the response payload and use EventBridge
        // as destination
        if (!options) { // `options` added to bind() as optionnal to avoid breaking change
            throw new Error('Options must be defined when using `responseOnly`.');
        }
        // Match invocation result of the source function (`fn`) and use it
        // to trigger the destination function (`this.fn`).
        new events.Rule(scope, options.type, {
            eventPattern: {
                detailType: [`Lambda Function Invocation Result - ${options.type}`],
                resources: [`${fn.functionArn}:$LATEST`],
                source: ['lambda'],
            },
            targets: [
                new targets.LambdaFunction(this.fn, {
                    event: events.RuleTargetInput.fromEventPath('$.detail.responsePayload'),
                }),
            ],
        });
        const destination = new event_bridge_1.EventBridgeDestination(); // Use default event bus here
        return destination.bind(scope, fn);
    }
}
exports.LambdaDestination = LambdaDestination;
//# sourceMappingURL=data:application/json;base64,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