# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['gaphas']

package_data = \
{'': ['*']}

install_requires = \
['PyGObject>=3.20.0,<4.0.0',
 'importlib_metadata>=1.3.0,<2.0.0',
 'pycairo>=1.13.0,<2.0.0']

setup_kwargs = {
    'name': 'gaphas',
    'version': '2.1.1',
    'description': 'Gaphas is a GTK+ based diagramming widget',
    'long_description': '# Gaphas\n[![Build state](https://github.com/gaphor/gaphas/workflows/build/badge.svg)](https://github.com/gaphor/gaphas/actions)\n[![Maintainability](https://api.codeclimate.com/v1/badges/e9837cc647b72119fd11/maintainability)](https://codeclimate.com/github/gaphor/gaphas/maintainability)\n[![Test Coverage](https://api.codeclimate.com/v1/badges/e9837cc647b72119fd11/test_coverage)](https://codeclimate.com/github/gaphor/gaphas/test_coverage)\n![Docs build state](https://readthedocs.org/projects/gaphas/badge/?version=latest)\n[![Code style: black](https://img.shields.io/badge/code%20style-black-000000.svg)](https://github.com/ambv/black)\n[![standard-readme compliant](https://img.shields.io/badge/readme%20style-standard-brightgreen.svg?style=flat)](https://github.com/RichardLitt/standard-readme)\n[![Gitter](https://img.shields.io/gitter/room/nwjs/nw.js.svg)](https://gitter.im/Gaphor/Lobby?utm_source=badge&utm_medium=badge&utm_campaign=pr-badge&utm_content=badge)\n[![All Contributors](https://img.shields.io/badge/all_contributors-9-orange.svg?style=flat-square)](#contributors)\n\n> Gaphas is the diagramming widget library for Python.\n\n![Gaphas Demo](docs/images/gaphas-demo.gif)\n\nGaphas is a library that provides the user interface component (widget) for drawing diagrams. Diagrams can be drawn to screen and then easily exported to a variety of formats, including SVG and PDF. Want to build an app with chart-like diagrams? Then Gaphas is for you! Use this library to build a tree, network, flowchart, or other diagrams.\n\nThis library is currently being used by [Gaphor](https://github.com/gaphor/gaphor) for UML drawing,\n[RAFCON](https://github.com/DLR-RM/RAFCON) for state-machine based robot control, and [ASCEND](http://ascend4.org/) for solving mathematical models.\n\n## :bookmark_tabs: Table of Contents\n\n- [Background](#background)\n- [Install](#install)\n- [Usage](#usage)\n- [API](#api)\n- [Contributing](#contributing)\n- [License](#license)\n\n## :scroll: Background\n\nGaphas was built to provide the foundational diagramming portions of [Gaphor](https://github.com/gaphor/gaphor). Since Gaphor is built on GTK and Cairo, [PyGObject](https://pygobject.readthedocs.io/) provides access to the GUI toolkit and [PyCairo](https://pycairo.readthedocs.io/) to the 2D graphics library. However, there wasn\'t a project that abstracted these technologies to easily create a diagramming tool. Hence, Gaphas was created as a library to allow others to create a diagramming tool using GTK and Cairo.\n\nHere is how it works:\n\n- Items (Canvas items) can be added to a Canvas.\n- The Canvas maintains the tree structure (parent-child relationships between items).\n- A constraint solver is used to maintain item constraints and inter-item constraints.\n- The item (and user) should not be bothered with things like bounding-box calculations.\n- Very modular--e.g., handle support could be swapped in and swapped out.\n- Rendering using Cairo.\n\nThe main portions of the library include:\n\n- canvas - The main canvas class (container for Items).\n- items - Objects placed on a Canvas.\n- solver - A constraint solver to define the layout and connection of items.\n- view - Responsible for the calculation of bounding boxes which is stored in a quadtree data structure for fast access.\n- gtkview - A view to be used in GTK applications that interacts with users with tools.\n- painters - The workers used to paint items.\n- tools - Tools are used to handle user events (such as mouse movement and button presses).\n- aspects - Provides an intermediate step between tools and items.\n\n## :floppy_disk: Install\n\nTo install Gaphas, simply use pip:\n\n```bash\n$ pip install gaphas\n```\n\nUse of a\n[virtual environment](https://virtualenv.pypa.io/en/latest/) is highly recommended.\n\n### Development\n\nTo setup a development environment with Linux:\n\n```bash\n$ sudo apt-get install -y python3-dev python3-gi python3-gi-cairo\n    gir1.2-gtk-3.0 libgirepository1.0-dev libcairo2-dev\n$ pip install poetry\n$ poetry install\n```\n\n## :flashlight: Usage\n\n```python\nimport gi\ngi.require_version("Gtk", "3.0")\nfrom gi.repository import Gtk\n\nfrom gaphas import Canvas, GtkView\nfrom gaphas.examples import Box\nfrom gaphas.painter import DefaultPainter\nfrom gaphas.item import Line\nfrom gaphas.segment import Segment\n\n\ndef create_canvas(canvas, title):\n    # Setup drawing window\n    view = GtkView()\n    view.painter = DefaultPainter()\n    view.canvas = canvas\n    window = Gtk.Window()\n    window.set_title(title)\n    window.set_default_size(400, 400)\n    win_box = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL)\n    window.add(win_box)\n    win_box.pack_start(view, True, True, 0)\n\n    # Draw gaphas box\n    b2 = Box(60, 60)\n    b2.min_width = 40\n    b2.min_height = 50\n    b2.matrix.translate(170, 170)\n    canvas.add(b2)\n\n    # Draw gaphas line\n    line = Line()\n    line.matrix.translate(100, 60)\n    canvas.add(line)\n    line.handles()[1].pos = (30, 30)\n    segment = Segment(line, view=None)\n    segment.split_segment(0)\n\n    window.show_all()\n    window.connect("destroy", Gtk.main_quit)\n\n\nc = Canvas()\ncreate_canvas(c, "Simple Gaphas App")\nGtk.main()\n```\n\n### Overview\n\nThe `Canvas` class acts as a container for Item\'s . The item\'s parent/child relationships are maintained here, not in the `Item`.\n\nAn `Item` can have a set of handles which can be used to manipulate the item (although this is not necessary). Each item has its own coordinate system with x and y position, for example, a (0, 0) point `Item.matrix` is the transformation relative to the parent item of the `Item`, as defined in the `Canvas`.\n\nHandles can connect to [ports](https://gaphas.readthedocs.io/en/latest/ports.html). A `Port` is a location (line or point) where a handle is allowed to connect on another\nitem. The process of connecting depends on the case at hand, but most often involves the creation of some sort of constraint between the Handle and the item it is connecting to.\n\nThe Canvas also contains a [constraint solver](https://gaphas.readthedocs.io/en/latest/solver.html) (from solver.py) that can be used to solve mathematical dependencies\nbetween items (such as Handles that should be aligned). The constraint solver can also be used to keep constraints contained within the item satisfied, for example, to make sure a box maintains its rectangular shape.\n\nA View is used to visualize a Canvas. On a View, a Tool (from tool.py) can be applied, which will handle user input like button and key presses. Painters (from painter.py) are used to do the actual drawing. This module also makes it easy to draw to other media other than a screen, such as a printer or a PDF document.\n\n### Updating Item state\n\nIf items need updating, it sends out an update request on the Canvas (`Canvas.request_update()`). The canvas\nperforms an update by performing the following steps:\n\n1. Pre-update using `Item.pre_update(context)` for each item marked for update.\n2. Update the Canvas-to-Item matrices, for fast transformation of coordinates from the Canvas\' to the items\' coordinate      system.\n3. Solve the constraints.\n4. Normalize the items by setting the coordinates of the first handle to (0, 0).\n5. Update the Canvas-to-Item matrices for items that have been changed by normalization.\n6. Post-update using `Item.post_update(context)` for each item marked for update, including items that have been\n   marked during the constraint solving step.\n\nGaphas attempts to do as much updating as possible in the {pre|post}_update() methods, since they are called when the application is not handling user input.\n\nThe context contains a CairoContext. This can be used, for example, to calculate the dimensions of text. One thing to keep in mind is that updating is done from the Canvas. Items should not update sub-items. After the update steps are complete, the Item should be ready to be drawn.\n\n### Constraint solving\n\nConstraint solving is one of the big features of this library. The Solver is able to mathematically solve these constraint rules that are applied to an item or between items. Constraints are applied to items through variables owned by\nthe item. An example of applying a constraint to an item is that Element items use constraints to maintain their rectangular shape. An example of applying constraints between items is to apply a constraint between a line and a box in\norder to connect them.\n\nConstraints that apply to one item are pretty straight forward, as all variables live in the same coordinate system of the item. The variables, like the Handle\'s x and y coordinate can simply be put in a constraint.\n\nWhen two items are connected to each other and constraints are created, a problem shows up: variables live in separate coordinate systems. In order to overcome this problem, a Projection (from solver.py) has been defined. With a\nProjection instance, a variable can be "projected" on another coordinate system. In this case, the Canvas\' coordinate system is used when two items are connected to each other.\n\n### Drawing\n\nDrawing is done by the View. All items marked for redraw, the items that have been updated, will be drawn in the order in which they reside in the Canvas. The order starts with the first root item, then its children, then second root item, etc.\n\nThe view context passed to the Items draw() method has the following properties:\n\n- view - The view we\'re drawing to.\n- cairo - The CairoContext to draw to.\n- selected - True if the item is actually selected in the view.\n- focused - True if the item has the focus\n- hovered - True if the mouse pointer if over the item. Only the top-most item is marked as hovered.\n- dropzone - The item is marked as the drop zone. When this happens then an item is dragged over the item, and if it is     dropped, it will become a child of this item.\n\nThe View automatically calculates the bounding box for the item, based on the items drawn in the draw (context) function (this is only done when really necessary, e.g., after an update of the item). The bounding box is in viewport coordinates.\n\nThe actual drawing is done by Painters (painter.py). A series of Painters have been defined: one for handles, one for items, etc.\n\n### Tools\n\nBehaviour is added to the Canvas (view) by tools. Tools can be chained together in order to provide more complex behaviour.\n\nTo make it easy, a DefaultTool has been defined which is a ToolChain instance with the tools added as follows:\n\n- ToolChain - Delegates to a set of individual tools and keeps track of which tool has grabbed the focus. This normally     happens when the user presses a mouse button. Once this happens, the tool requests a "grab" and all events, like motion   or button release, are sent directly to the focused tool.\n\n- HoverTool - Makes the item under the mouse button the "hovered item." When such an item is drawn, its context.hovered_item flag will be set to True.\n\n- HandleTool - Allows for handles to be dragged around and focuses the item when its handle is clicked on.\n\n- ItemTool - Selects items and enables dragging items around.\n\n- TextEditTool - A demo tool that features a text edit pop-up.\n\n- RubberbandTool - Invoked when the mouse button is pressed on a section of the view where no items or handles are present. It allows the user to select items using a "rubber band" selection box.\n\n### Interaction\n\nTools handle Interaction with the Canvas view (visual component). Although the default tools do a fair amount of work, in most cases you\'ll desire to create some custom connection behaviour. In order to implement these, HandleTool provides hooks including connect, disconnect, and glue.\n\nOne of the challenges you\'ll likely face is what to do when an item is removed from the Canvas and there are other items (lines) connected to it. Gaphas provides a solution to this by providing a disconnect handler to the handle instance once it is connected. A function can be assigned to this disconnect handler, which is then called when the item that is connected to is removed from the Canvas.\n\n### Undo\n\nGaphas has a simple built-in system for registering changes in its classes and notifying the application. This code resides in state.py.\n\nThere is also a "reverter" framework in place. This system is notified when objects change their state, and the framework will figure out the reverse operation that has to be applied in order to undo the operation.\n\n## :mag: API\n\nThe API can be separated into a [Model-View-Controller](https://en.wikipedia.org/wiki/Model%E2%80%93view%E2%80%93controller) with these three parts:\n\n1. The Model, including the canvas and items\n2. The View, called view\n3. The Controller, called tools\n\n### Canvas and Items\n\n#### Class: `gaphas.canvas.Canvas`\n\nThe `Canvas` is a container for items.\n\n```python\ncanvas = Canvas()\n```\n\n#### Class: `gaphas.item.Item`\n\nBase class (or interface) for items on a `Canvas`.\n\n```python\nitem = Item()\n```\n\n##### Properties:\n\n- `matrix`: The item\'s transformation matrix\n- `canvas`: The canvas, which owns an item\n- `constraints`: list of item constraints, automatically registered when the item is added to a canvas; may be extended in subclasses\n\n#### Class: `gaphas.connector.Handle`\n\nHandles are used to support modifications of Items.\n\nIf the handle is connected to an item, the `connected_to` property should refer to the item. A `disconnect` handler should\nbe provided that handles the required disconnect behaviour, for example, cleaning up the constraints and `connected_to`.\n\n- pos (`gaphas.connector.Position`): The position of the item, default value is (0, 0).\n- strength (int): The strength of the handle to use in the constraint solver; default value is NORMAL, which is 20.\n- connectable (bool): Makes the handle connectable to other items; default value is False.\n- movable (bool): Makes the handle moveable; default value is True.\n\n  ```python\n  handle = Handle((10, 10), connectable=True)\n  ```\n\n#### Class: `gaphas.connector.LinePort`\n\nThe Line Port is part of an item that provides a line between two handles.\n\n- start (`gaphas.connector.Position`): The start position of the line.\n- end (`gaphas.connector.Position`): The end position of the line.\n\n```python\np1, p2 = (0.0, 0.0), (100.0, 100.0)\nport = LinePort(p1, p2)\n```\n\n#### Class: `gaphas.connector.PointPort`\n\nThe Point Port connects the handle to an item using a port at the location of the handle.\n\n```python\nh = Handle((10, 10))\nport = PointPort(h.pos)\n```\n\n#### Class: `gaphas.solver.Solver`\n\nA Solver solves constraints.\n\n```python\na, b, c = Variable(1.0), Variable(2.0), Variable(3.0)\nsolver = Solver()\nc_eq = EquationConstraint(lambda a,b: a+b, a=a, b=b)\nsolver.add_constraint(c_eq)\n```\n\n#### Class: `gaphas.constraint.EqualsConstraint`\n\nMake \'a\' and \'b\' equal.\n\n```python\na, b = Variable(1.0), Variable(2.0)\neq = EqualsConstraint(a, b)\neq.solve_for(a)\n```\n\n#### Class: `gaphas.constraint.LessThanConstraint`\n\nEnsure one variable stays smaller than another.\n\n```python\na, b = Variable(3.0), Variable(2.0)\nlt = LessThanConstraint(smaller=a, bigger=b)\nlt.solve_for(a)\n```\n\n#### Class: `gaphas.constraint.CenterConstraint`\n\nEnsures a Variable is kept between two other variables.\n\n```python\na, b, center = Variable(1.0), Variable(3.0), Variable()\neq = CenterConstraint(a, b, center)\neq.solve_for(a)\n```\n\n#### Class: `gaphas.constraint.EquationConstraint`\n\nSolve a linear equation.\n\n```python\na, b, c = Variable(), Variable(4), Variable(5)\ncons = EquationConstraint(lambda a, b, c: a + b - c, a=a, b=b, c=c)\ncons.solve_for(a)\n```\n\n#### Class: `gaphas.constraint.BalanceConstraint`\n\nKeeps three variables in line, maintaining a specific ratio.\n\n```python\na, b, c = Variable(2.0), Variable(3.0), Variable(2.3, WEAK)\nbc = BalanceConstraint(band=(a,b), v=c)\nc.value = 2.4\n```\n\n#### Class: `gaphas.constraint.LineConstraint`\n\nSolves the equation where a line is connected to a line or side at a specific point.\n\n```python\nline = (Variable(0), Variable(0)), (Variable(30), Variable(20))\npoint = (Variable(15), Variable(4))\nlc = LineConstraint(line=line, point=point)\n```\n\n### View\n\n#### Class: `gaphas.view.View`\n\nView class for `gaphas.canvas.Canvas` objects.\n\n```python\ncanvas = Canvas()\nview = View(canvas=canvas)\n```\n\n#### Class: `gaphas.view.GtkView`\n\nGTK+ widget for rendering a `gaphas.canvas.Canvas` to a screen.\n\n```python\ncanvas = Canvas()\nwin = Gtk.Window()\nview = GtkView(canvas=canvas)\nwin.add(view)\n```\n\n#### Class: `gaphas.painter.PainterChain`\n\nChain up a set of painters.\n\n```python\nview.painter = (\n    PainterChain()\n    .append(FreeHandPainter(ItemPainter()))\n    .append(HandlePainter())\n    .append(FocusedItemPainter())\n    .append(ToolPainter())\n)\n```\n\n#### Class: `gaphas.painter.DrawContext`\n\nSpecial context for drawing the item. It contains a cairo context and properties like selected and focused.\n\n- **kwargs: Optional cairo properties for a context.\n\n```python\nDrawContext(\n    painter=self,\n    cairo=cairo,\n    selected=(item in view.selected_items),\n    focused=(item is view.focused_item),\n    hovered=(item is view.hovered_item),\n    dropzone=(item is view.dropzone_item),\n)\n```\n\n#### Class: `gaphas.painter.ItemPainter`\n\nPainter to draw an item.\n\n```python\nsvgview = View(view.canvas)\nsvgview.painter = ItemPainter()\n```\n\n#### Class: `gaphas.painter.CairoBoundingBoxContext`\n\nIt is used to intercept `stroke()`, `fill()`, and other context operations so that the bounding box of the item involved can be calculated.\n\n- cairo (cairo.Context): The cairo context to intercept.\n\n```python\ncairo = CairoBoundingBoxContext(cairo)\n```\n\n#### Class: `gaphas.painter.BoundingBoxPainter`\n\nA type of ItemPainter which is used to calculate the bounding boxes (in canvas coordinates) for the items.\n\n```python\nview.bounding_box_painter = BoundingBoxPainter(ItemPainter())\n```\n\n#### Class: `gaphas.painter.HandlePainter`\n\nDraw handles of items that are marked as selected in the view.\n\n#### Class: `gaphas.painter.ToolPainter`\n\nAllows the Tool defined on a view to conduct drawing.\n\n#### Class: `gaphas.painter.FocusedItemPainter`\n\nUsed to draw on top of all the other layers for the focused item.\n\n### Tools\n\nInteracting with the Canvas is done through tools. Tools tell _what_ has to be done (like moving). To make an element move aspects are defined. Aspects tell how the behaviour has to be performed.\n\n#### Class: `gaphas.tools.ToolChain`\n\nUsed to chain tools together. For example, chain a HoverTool, HandleTool, and SelectionTool in order to combine their functionality in to a new tool.\n\n- view (`gaphas.view.View`): The view to use for the tool chain.\n\n```python\n(ToolChain(view)\n.append(HoverTool())\n.append(ConnectHandleTool())\n.append(PanTool())\n.append(ZoomTool())\n.append(ItemTool())\n.append(TextEditTool())\n.append(RubberbandTool())\n)\n```\n\n#### Class: `gaphas.tools.HoverTool`\n\nMakes the item under the mouse cursor the hovered item.\n\n- view (`gaphas.view.View`): The view to use for the tool; default is None.\n\n#### Class: `gaphas.tools.ItemTool`\n\nDoes selection and dragging of items.\n\n- view (`gaphas.view.View`): The view to use for the tool; default is None.\n\n#### Class: `gaphas.tools.HandleTool`\n\nTool to move handles around.\n\n- view (`gaphas.view.View`): The view to use for the tool; default is None.\n\n#### Class: `gaphas.tools.RubberbandTool`\n\nAllows the user to drag a "rubber band" for selecting items in an area.\n\n- view (`gaphas.view.View`): The view to use for the tool; default is None.\n\n#### Class: `gaphas.tools.PanTool`\n\nCaptures drag events with the middle mouse button and uses them to translate the Canvas within the view.\n\n- view (`gaphas.view.View`): The view to use for the tool; default is None.\n\n#### Class: `gaphas.tools.ZoomTool`\n\nTool for zooming using two different user inputs:\n\n1. Ctrl + middle-mouse dragging in the up and down direction\n2. Ctrl + mouse-wheel\n\n- view (`gaphas.view.View`): The view to use for the tool; default is None.\n\n#### Class: `gaphas.tools.PlacementTool`\n\nTool for placing items on the Canvas.\n\n- view (`gaphas.view.View`): The view to use for the tool.\n- factory (factory object): A Canvas item factory for creating new items.\n- handle_tool (`gaphas.tools.HandleTool`): The handle tool to use.\n- handle_index (int): The index of the handle to be used by the handle tool.\n\n```python\ndef on_clicked(button):\n    view.tool.grab(PlacementTool(view, factory(view, MyLine), HandleTool(), 1))\n```\n\n#### Class: `gaphas.aspects.ItemFinder`\n\nFind an item on the Canvas.\n\n- view (`gaphas.view.View`): The view to use in order to search for an item.\n\n#### Class: `gaphas.aspects.ItemSelection`\n\nManages selection and unselection of items.\n\n- item (`gaphas.item.Item`): The item to set as focused or unfocused.\n- view (`gaphas.view.View`): The view to focus or unfocus on.\n\n#### Class: `gaphas.aspects.ItemInMotion`\n\nManages motion of an item.\n\n- item (`gaphas.item.Item`): The item to move.\n- view (`gaphas.view.View`): The view to to use for move coordinates.\n\n#### Class: `gaphas.aspects.ItemHandleFinder`\n\nFinds handles.\n\n- item (`gaphas.item.Item`): The item.\n- view (`gaphas.view.View`): The view to get the handle at the position from.\n\n```python\ncanvas = Canvas()\nline = Line()\ncanvas.add(line)\nview = View(canvas)\nfinder = HandleFinder(line, view)\n```\n\n#### Class: `gaphas.aspects.ElementHandleSelection`\n\nSelects the handle of a `gaphas.item.Element`.\n\n- item (`gaphas.item.Item`): The Element item that the handle belongs to.\n- handle (`gaphas.connector.Handle`): The handle to select or unselect.\n- view (`gaphas.view.View`): The view that can be used to apply the cursor to.\n\n#### Class: `gaphas.aspects.ItemHandleInMotion`\n\nMove a handle.\n\n- item (`gaphas.item.Item`): The item that the handle belongs to.\n- handle (`gaphas.connector.Handle`): The handle to move.\n- view (`gaphas.view.View`): The view to use for the coordinate system.\n\n#### Class: `gaphas.aspects.ItemConnector`\n\nConnect or disconnect an item\'s handle to another item or port.\n\n- item (`gaphas.item.Item`): The item that the handle belongs to.\n- handle (`gaphas.connector.Handle`): The handle to connect.\n\n#### Class: `gaphas.aspects.ItemConnectionSink`\n\nMakes an item a sink, which is another item that an item\'s handle is connected to like a connected item or port.\n\n- item (`gaphas.item.Item`): The item to look for ports on.\n- port (`gaphas.connector.Port`): The port to use as the sink.\n\n#### Class: `gaphas.aspects.ItemPaintFocused`\n\nPaints on top of all items, just for the focused item and only when it\'s hovered (see\n `gaphas.painter.FocusedItemPainter`).\n\n- item (`gaphas.item.Item`): The focused item.\n- view (`gaphas.view.View`): The view to paint with.\n\n### Extended Behaviour\n\nBy importing the following modules, extra behaviour is added to the default view behaviour.\n\n#### Class: `gaphas.segment.LineSegment`\n\nSplit and merge line segments.\n\n- item (`gaphas.item.Item`): The item of the segment.\n- view (`gaphas.view.View`): The view to use for the split coordinate system.\n\n#### Class: `gaphas.segment.SegmentHandleFinder`\n\nExtends the `gaphas.aspects.ItemHandleFinder` to find a handle on a line, and create a new handle if the mouse is located between two handles. The position aligns with the points drawn by the SegmentPainter.\n\n- item (`gaphas.item.Item`): The item.\n- view (`gaphas.view.View`): The view to get the handle at the position from.\n\n#### Class: `gaphas.segment.SegmentHandleSelection`\n\nExtends the `gaphas.aspects.ItemHandleSelection` to merge segments if the handle is released.\n\n- item (`gaphas.item.Item`): The item that the handle belongs to.\n- handle (`gaphas.connector.Handle`): The handle to select or unselect.\n- view (`gaphas.view.View`): The view to use to apply the cursor to.\n\n#### Class: `gaphas.segment.LineSegmentPainter`\n\nThis painter draws pseudo-handles on a `gaphas.item.Line` by extending `gaphas.aspects.ItemPaintFocused`. Each line can be split by dragging those points, which will result in a new handle.\n\nConnectHandleTool takes care of performing the user interaction required for this feature.\n\n- item (`gaphas.item.Item`): The focused item.\n- view (`gaphas.view.View`): The view to paint with.\n\n#### Class: `gaphas.guide.ElementGuide`\n\nProvides a guide to align items for `gaphas.item.Element`.\n\n- item (`gaphas.item.Item`): The item to provide guides for.\n\n#### Class: `gaphas.guide.LineGuide`\n\nProvides a guide to align items for `gaphas.item.Line`.\n\n- item (`gaphas.item.Item`): The item to provide guides for.\n\n#### Class: `gaphas.guide.GuidedItemInMotion`\n\nMove the item and lock the position of any element that\'s located at the same position.\n\n- item (`gaphas.item.Item`): The item to move.\n- view (`gaphas.view.View`): The view with guides to use for move coordinates.\n\n```python\ncanvas = Canvas()\nview = GtkView(canvas)\nelement = Element()\nguider = GuidedItemInMotion(element, view)\nguider.start_move((0, 0))\n```\n\n#### Class: `gaphas.guide.GuidedItemHandleInMotion`\n\nMove a handle and lock the position of any element that\'s located at the same position.\n\n- item (`gaphas.item.Item`): The item that the handle belongs to.\n- handle (`gaphas.connector.Handle`): The handle to move.\n- view (`gaphas.view.View`): The view with guides to use for the coordinate system.\n\n#### Class: `gaphas.guide.GuidePainter`\n\nPaints on top of all items with guides, just for the focused item and only when it\'s hovered.\n\n- item (`gaphas.item.Item`): The focused item.\n- view (`gaphas.view.View`): The view with guides to paint with.\n\n### Miscellaneous\n\n#### Class: `gaphas.tree.Tree`\n\nA Tree structure with the nodes stored in a depth-first order.\n\n```python\ntree = Tree()\ntree.add("node1")\ntree.add("node2", parent="node1")\n```\n\n#### Class: `gaphas.matrix.Matrix`\n\nAdds @observed messages on state changes to the cairo.Matrix.\n\n- xx (float): xx component of the affine transformation\n- yx (float): yx component of the affine transformation\n- xy (float): xy component of the affine transformation\n- yy (float): yy component of the affine transformation\n- x0 (float): X translation component of the affine transformation\n- y0 (float): Y translation component of the affine transformation\n\n```python\nmatrix = Matrix(1, 0, 0, 1, 0, 0)\n```\n\n#### Class: `gaphas.table.Table`\n\nTable is a storage class that can be used to store information, like one would in a database table, with indexes on the desired "columns." It includes indexing and is optimized for lookups.\n\n- columns (tuple): The columns of the table.\n- index (tuple):\n\n```python\nfrom collections import namedtuple\nC = namedtuple(\'C\', "foo bar baz")\ns = Table(C, (2,))\n```\n\n#### Class: `gaphas.quadtree.Quadtree`\n\nA quadtree is a tree data structure in which each internal node has up to four children. Quadtrees are most often used to partition a two\n\n- bounds (tuple): The boundaries of the quadtree (x, y, width, height).\n- capacity (int); The number of elements in one tree bucket; default is 10.\n\n```python\nqtree = Quadtree((0, 0, 100, 100))\n```\n\n#### Class: `gaphas.geometry.Rectangle`\n\nRectangle object which can be added (union), substituted (intersection), and points and rectangles can be tested to be in the rectangle.\n\n- x (int): X position of the rectangle.\n- y (int): Y position of the rectangle.\n- width (int): Width of the rectangle.\n- height (int): Hiehgt of the rectangle.\n\n```python\nrect = Rectangle(1, 1, 5, 5)\n```\n\n#### Decorator: @AsyncIO\n\nSchedule an idle handler at a given priority.\n\n- single (bool): Schedules the decorated function to be called only a single time.\n- timeout (int): The time between calls of the decorated function.\n- priority (int): The GLib.PRIORITY constant to set the event priority.\n\n```python\n\n@AsyncIO(single=True, timeout=60)\ndef c2(self):\n    print(\'idle-c2\', GLib.main_depth())\n```\n\n#### Decorator: @nonrecursive\n\nEnforces a function or method to not be executed recursively.\n   api/decorators\n\n```python\n\nclass A(object):\n    @nonrecursive\n    def a(self, x=1):\n        self.a(x+1)\n```\n\n#### Decorator: @recursive\n\nLimits the recursion for a specific function.\n\n- limit (int): The limit for the number of recursive loops a function can be called; default is 10000.\n\n```python\n@recursive(10)\ndef a(self, x=0):\n    self.a()\n```\n\n## :heart: Contributing\n\nThanks goes to these wonderful people ([emoji key](https://github.com/kentcdodds/all-contributors#emoji-key)):\n\n<!-- ALL-CONTRIBUTORS-LIST:START - Do not remove or modify this section -->\n<!-- prettier-ignore -->\n<table><tr><td align="center"><a href="https://github.com/amolenaar"><img src="https://avatars0.githubusercontent.com/u/96249?v=4" width="100px;" alt="Arjan Molenaar"/><br /><sub><b>Arjan Molenaar</b></sub></a><br /><a href="https://github.com/gaphor/gaphas/commits?author=amolenaar" title="Code">💻</a> <a href="https://github.com/gaphor/gaphas/issues?q=author%3Aamolenaar" title="Bug reports">🐛</a> <a href="https://github.com/gaphor/gaphas/commits?author=amolenaar" title="Documentation">📖</a> <a href="#review-amolenaar" title="Reviewed Pull Requests">👀</a> <a href="#question-amolenaar" title="Answering Questions">💬</a> <a href="#plugin-amolenaar" title="Plugin/utility libraries">🔌</a></td><td align="center"><a href="https://ghuser.io/danyeaw"><img src="https://avatars1.githubusercontent.com/u/10014976?v=4" width="100px;" alt="Dan Yeaw"/><br /><sub><b>Dan Yeaw</b></sub></a><br /><a href="https://github.com/gaphor/gaphas/commits?author=danyeaw" title="Code">💻</a> <a href="https://github.com/gaphor/gaphas/commits?author=danyeaw" title="Tests">⚠️</a> <a href="#review-danyeaw" title="Reviewed Pull Requests">👀</a> <a href="https://github.com/gaphor/gaphas/issues?q=author%3Adanyeaw" title="Bug reports">🐛</a> <a href="#question-danyeaw" title="Answering Questions">💬</a> <a href="#infra-danyeaw" title="Infrastructure (Hosting, Build-Tools, etc)">🚇</a> <a href="https://github.com/gaphor/gaphas/commits?author=danyeaw" title="Documentation">📖</a></td><td align="center"><a href="https://github.com/wrobell"><img src="https://avatars2.githubusercontent.com/u/105664?v=4" width="100px;" alt="wrobell"/><br /><sub><b>wrobell</b></sub></a><br /><a href="https://github.com/gaphor/gaphas/commits?author=wrobell" title="Code">💻</a> <a href="https://github.com/gaphor/gaphas/commits?author=wrobell" title="Tests">⚠️</a> <a href="#review-wrobell" title="Reviewed Pull Requests">👀</a></td><td align="center"><a href="https://github.com/jlstevens"><img src="https://avatars3.githubusercontent.com/u/890576?v=4" width="100px;" alt="Jean-Luc Stevens"/><br /><sub><b>Jean-Luc Stevens</b></sub></a><br /><a href="https://github.com/gaphor/gaphas/commits?author=jlstevens" title="Code">💻</a> <a href="https://github.com/gaphor/gaphas/issues?q=author%3Ajlstevens" title="Bug reports">🐛</a> <a href="https://github.com/gaphor/gaphas/commits?author=jlstevens" title="Documentation">📖</a></td><td align="center"><a href="http://www.franework.de"><img src="https://avatars1.githubusercontent.com/u/1144966?v=4" width="100px;" alt="Franz Steinmetz"/><br /><sub><b>Franz Steinmetz</b></sub></a><br /><a href="https://github.com/gaphor/gaphas/commits?author=franzlst" title="Code">💻</a> <a href="https://github.com/gaphor/gaphas/issues?q=author%3Afranzlst" title="Bug reports">🐛</a></td><td align="center"><a href="https://github.com/adrianboguszewski"><img src="https://avatars3.githubusercontent.com/u/4547501?v=4" width="100px;" alt="Adrian Boguszewski"/><br /><sub><b>Adrian Boguszewski</b></sub></a><br /><a href="https://github.com/gaphor/gaphas/commits?author=adrianboguszewski" title="Code">💻</a></td><td align="center"><a href="https://github.com/Rbelder"><img src="https://avatars3.githubusercontent.com/u/15119522?v=4" width="100px;" alt="Rico Belder"/><br /><sub><b>Rico Belder</b></sub></a><br /><a href="https://github.com/gaphor/gaphas/issues?q=author%3ARbelder" title="Bug reports">🐛</a> <a href="#review-Rbelder" title="Reviewed Pull Requests">👀</a></td></tr><tr><td align="center"><a href="http://www.boduch.ca"><img src="https://avatars2.githubusercontent.com/u/114619?v=4" width="100px;" alt="Adam Boduch"/><br /><sub><b>Adam Boduch</b></sub></a><br /><a href="https://github.com/gaphor/gaphas/issues?q=author%3Aadamboduch" title="Bug reports">🐛</a></td><td align="center"><a href="https://github.com/janettech"><img src="https://avatars3.githubusercontent.com/u/13398384?v=4" width="100px;" alt="Janet Jose"/><br /><sub><b>Janet Jose</b></sub></a><br /><a href="https://github.com/gaphor/gaphas/commits?author=janettech" title="Documentation">📖</a></td></tr></table>\n\n<!-- ALL-CONTRIBUTORS-LIST:END -->\n\nThis project follows the [all-contributors](https://github.com/kentcdodds/all-contributors) specification. Contributions of any kind are welcome!\n\n1. Check for open issues or open a fresh issue to start a discussion around a feature idea or a bug.\n    There is a [first-timers-only](https://github.com/gaphor/gaphas/issues?utf8=%E2%9C%93&q=is%3Aissue+is%3Aopen+label%3Afirst-timers-only) tag for issues that should be ideal for people who are not very familiar with the codebase yet.\n2. Fork [the repository](https://github.com/gaphor/gaphas) on GitHub to    start making your changes to the **master**       branch (or branch off of it).\n3. Write a test which shows that the bug was fixed or that the feature\n   works as expected.\n4. Send a pull request and bug the maintainers until it gets merged and\n   published. :smile:\n\nSee [the contributing file](CONTRIBUTING.md)!\n\n## :copyright: License\n\nCopyright (C) Arjan Molenaar and Dan Yeaw\n\nLicensed under the [Apache License 2.0](LICENSE).\n\nSummary: You can do what you like with Gaphas, as long as you include the required notices. This permissive license contains a patent license from the contributors of the code.\n',
    'author': 'Arjan J. Molenaar',
    'author_email': 'gaphor@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://gaphas.readthedocs.io/',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.6,<4.0',
}


setup(**setup_kwargs)
