"""
Lemon.Server: server
By Sasen Perera 2022
"""
import inspect
import webbrowser

from parse import parse
from webob import Request, Response
from waitress import serve
from whitenoise import WhiteNoise
from .middleware import Middleware
from requests import Session as RequestsSession
from wsgiadapter import WSGIAdapter as RequestsWSGIAdapter

class Server:
    """Server Methods"""
    def __init__(self, static_dir="public"):
        self.routes = {}
        self.exception_handler = None
        self.whitenoise = WhiteNoise(self.wsgi_app, root=static_dir)
        self.middleware = Middleware(self)
        self.exception_handler = None

    def wsgi_app(self, environ, start_response):
        """WSGI App"""
        request = Request(environ)

        response = self.handle_request(request)

        return response(environ, start_response)

    def __call__(self, environ, start_response):
        """The code to run when the class is called as a function"""
        path_info = environ["PATH_INFO"]

        if path_info.startswith("/public"):
            environ["PATH_INFO"] = path_info[len("/public"):]
            return self.whitenoise(environ, start_response)

        return self.middleware(environ, start_response)

    def add_route(self, path, handler):
        "Django style Route Adding."
        assert path not in self.routes, "Such route already exists."

        self.routes[path] = handler

    def add_exception_handler(self, exception_handler):
        "Add Exception Handler"
        self.exception_handler = exception_handler
    
    def route(self, path):
        "route decorator: @server.route('/path')"
        assert path not in self.routes, "Such route already exists."
        def wrapper(handler):
            self.routes[path] = handler
            return handler

        return wrapper

    def default_response(self, response):
        "Default Response"
        response.status_code = 404
        response.text = "<!DOCTYPE html><html><body style='text-align:center;margin:10px;'><h1>404 Not Found</h1><br><p>generated by Lemon</p></body></html>"

    def find_handler(self, request_path):
        "Find Handler"
        for path, handler in self.routes.items():
            parse_result = parse(path, request_path)
            if parse_result is not None:
                return handler, parse_result.named

        return None, None

    def handle_request(self, request):
        """Handle Requests"""
        response = Response()

        handler, kwargs = self.find_handler(request_path=request.path)

        try:
            if handler is not None:
                if inspect.isclass(handler):
                    handler = getattr(handler(), request.method.lower(), None)
                    if handler is None:
                        raise AttributeError("Method now allowed", request.method)

                handler(request, response, **kwargs)
            else:
                self.default_response(response)
        except Exception as e:
            if self.exception_handler is None:
                raise e
            else:
                self.exception_handler(request, response, e)

        return response

    def add_cookie(self, response, key, value, max_age=None, expires=None, path="/", domain=None, secure=False, httponly=False, samesite=None):
        "Add Cookie"
        response.set_cookie(key, value, max_age=max_age, expires=expires, path=path, domain=domain, secure=secure, httponly=httponly, samesite=samesite)

    def delete_cookie(self, response, key, path="/", domain=None):
        "Delete Cookie"
        response.delete_cookie(key, path=path, domain=domain)
    
    def get_cookie(self, request, key):
        "Get Cookie"
        return request.cookies.get(key)

    def add_middleware(self, middleware_cls):
        """Add Middleware"""
        self.middleware.add(middleware_cls)

    def test_session(self, base_url="http://testserver"):
        "Testing Session"
        session = RequestsSession()
        session.mount(prefix=base_url, adapter=RequestsWSGIAdapter(self))
        return session

    def run(self, host="127.0.0.1", port=8000):
        "Runs app with waitress"
        host = host.lower()
        print(f"Running on http://localhost:{port} | http://127.0.0.1:{port}" if host == "127.0.0.1" or host == "localhost" else f"Running on http://{host}:{port}")
        print("To stop server press Ctrl+C")
        try:
            webbrowser.open(f"http://localhost:{port}" if host == "localhost" or "127.0.0.1" else f"http://{host}:{port}")
            try:
                serve(self, host=host, port=port)
            except Exception as e:
                print(f"lemon.server: {e}")
        except Exception as e:
            print(f"Lemon.Server: {e}")